/*
 *
 *
 */
package cn.gongler.util.db;

import cn.gongler.util.GonglerUtil;
import cn.gongler.util.function.ExceptionFunction;

import java.sql.*;
import java.util.Arrays;
import java.util.Objects;
import java.util.Spliterators;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import static cn.gongler.util.GonglerUtil.IterableAdapter;
import static cn.gongler.util.GonglerUtil.toRuntimeException;

/**
 * 2013.01.18
 *
 * @author gongler
 */
public class DbUtil {

    private static final long serialVersionUID = 8761451535995732352L;//DbUtil @since 2016-08-19

    private DbUtil() {
    }

    public static void Rollback(Connection conn) {
        if (conn != null) {
            try {
                conn.rollback();
            } catch (SQLException ex) {
                ex.printStackTrace();
            }
        }
    }

    public static int ExecuteUpdate(Connection conn, String sql, Object... args) {
        try (PreparedStatement smt = conn.prepareStatement(sql)) {
            return PreparedStatementWithArgs(smt, args).executeUpdate();
        } catch (Exception e) {
            Rollback(conn);
            throw GonglerUtil.toRuntimeException(e);
        }
    }

    public static int ExecuteUpdate(Connection conn, PreparedStatement smt, Object... args) {//20160818add
        try {
            return PreparedStatementWithArgs(smt, args).executeUpdate();
        } catch (Exception e) {
            Rollback(conn);
            throw GonglerUtil.toRuntimeException(e);
        }
    }

    public static boolean Execute(Connection conn, String sql, Object... args) {
        try (PreparedStatement smt = conn.prepareStatement(sql)) {
            return PreparedStatementWithArgs(smt, args).execute();
        } catch (Exception e) {
            Rollback(conn);
            throw GonglerUtil.toRuntimeException(e);
        }
    }

    /**
     * 调用无OUT参数、无返回值的存储过程
     *
     * @param conn 连接
     * @param sql  查询语句
     * @param args 参数
     * @return 成功与否
     */
    public static boolean ExecuteSimpleCall(Connection conn, String sql, Object... args) {
        int whyCnt = 0;
        for (int i = 0; i < sql.length(); i++) {
            if (sql.charAt(i) == '?') {
                whyCnt++;
            }
        }
        int argCnt = args.length;
        if (whyCnt != argCnt) {
            System.out.println("\n!!!!!!!?Cnt=" + whyCnt + "<>argCnt" + argCnt + ", " + sql + Arrays.toString(args) + "\n\n");
        }
        try (CallableStatement smt = conn.prepareCall(sql)) {
            return PreparedStatementWithArgs(smt, args).execute();
        } catch (Exception e) {
            Rollback(conn);
            throw GonglerUtil.toRuntimeException(e);
        }
    }

    public interface RowHandler {//20160914add

        default void begin() {
        }//2022年2月23日

        void rowHandle(ResultSet rs, int index) throws Exception;

        default void finished(int count) {
        }//2022年2月23日
    }

    public static void ExecuteQuery(Connection conn, String query, final RowHandler hander) {
        try (Statement stmt = conn.createStatement();
             ResultSet rs = stmt.executeQuery(query)) {
            ExecuteQuery(conn, rs, hander);
        } catch (SQLException e) {
            throw GonglerUtil.toRuntimeException(new SQLException(e.toString() + " when " + query, e));
        }
    }

    private static void ExecuteQuery(Connection conn, final ResultSet rs, final RowHandler hander) {
        if (conn == null || rs == null) {
            return;
        }
        try {
            int index = 0;
            while (rs.next()) {
                hander.rowHandle(rs, index++);
            }
        } catch (Exception e) {
            throw GonglerUtil.toRuntimeException(e);
        }
    }

    public static void ExecuteQuery(Connection conn, String query, final RowHandler hander, Object... args) {
        try (PreparedStatement smt = conn.prepareStatement(query)) {
            PreparedStatementWithArgs(smt, args);
            ResultSet rs = smt.executeQuery();

            int index = 0;
            while (rs.next()) {
                hander.rowHandle(rs, index++);
            }
        } catch (Exception e) {
            throw GonglerUtil.toRuntimeException(new SQLException(e.toString() + " when " + query, e));
        }
    }

    public static <S extends PreparedStatement> S PreparedStatementWithArgs(S smt, Object... args) throws SQLException {
        int argIndex = 1;
        for (Object arg : args) {
            if (arg instanceof Integer) {
                smt.setInt(argIndex, (Integer) arg);
            } else if (arg instanceof Long) {
                smt.setLong(argIndex, (Long) arg);
            } else if (arg instanceof String) {
                smt.setString(argIndex, (String) arg);
            } else if (arg instanceof Timestamp) {
                smt.setTimestamp(argIndex, (Timestamp) arg);
            } else if (arg instanceof Double) {
                smt.setDouble(argIndex, (Double) arg);
            } else if (arg instanceof Float) {
                smt.setFloat(argIndex, (Float) arg);
            } else if (arg instanceof RowId) {
                smt.setRowId(argIndex, (RowId) arg);
            } else if (arg instanceof Short) {
                smt.setShort(argIndex, (Short) arg);
            } else if (arg instanceof Blob) {
                smt.setBlob(argIndex, (Blob) arg);
            } else if (arg == null) {//gongler20181114debug 添加空值分支
                smt.setObject(argIndex, null);
            } else {
                throw new IllegalArgumentException("index=" + argIndex + ", arg=" + arg);
            }
            argIndex++;
        }
        return smt;
    }

    ////////////////////////////////////////////////////////////////////////////
    public static <T> Stream<T> Stream(ResultSet rs, ExceptionFunction<ResultSet, T> t) {//20160818add
        return StreamSupport.stream(Spliterators.spliteratorUnknownSize(IterableAdapter(rs).iterator(), 0), false).map(t::applyWithThrowAny).filter(Objects::nonNull);
    }

    public static <T, R> R Stream(Connection conn, ExceptionFunction<ResultSet, T> t, ExceptionFunction<Stream<T>, R> mapper, String query, Object... args) {//20160818add
        try (PreparedStatement smt = conn.prepareStatement(query)) {
            return mapper.applyWithThrowAny(Stream(PreparedStatementWithArgs(smt, args).executeQuery(), t));
        } catch (Exception e) {
            throw toRuntimeException(e);
        }
    }

    /**
     * <pre>
     * try (Connection conn = DriverManager.getConnection("jdbc:oracle:thin:@222.223.9.162:21521:tsgjgps1", "ts", "tsgprs1")) {
     * System.out.println("" + conn);
     * long count = Stream(conn, r -&gt; r.getString("pack_note"), "select * from ext_pack_send_queue where rownum《?", 10).limit(100).peek(System.out::println).sorted().count();
     * System.out.println("count=" + count);
     * }
     * </pre>
     *
     * @param conn  连接
     * @param t     处理
     * @param query 查询语句
     * @param args  参数
     * @param <T>   T
     * @return 返回结果
     */
    public static <T> Stream<T> Stream(Connection conn, ExceptionFunction<ResultSet, T> t, String query, Object... args) {//20160818add
        try (PreparedStatement smt = conn.prepareStatement(query)) {
            return Stream(PreparedStatementWithArgs(smt, args).executeQuery(), t).collect(Collectors.toList()).stream();//由于在返回前ResultSet已经被自动关闭，所以必须先转移到List然后在生成Stream
        } catch (SQLException e) {
            throw toRuntimeException(e);
        }
    }

//    public static void main(String[] args) throws SQLException {
//        try (Connection conn = DriverManager.getConnection("jdbc:oracle:thin:@222.223.9.162:21521:tsgjgps1", "ts", "tsgprs1")) {
//            System.out.println("" + conn);
//            long count = Stream(conn, r -> r.getString("pack_note"), "select * from ext_pack_send_queue where rownum<?", 10).limit(100)
//                .peek(System.out::println).sorted()
//                .count();
//            System.out.println("count=" + count);
//        }
//    }
}
