package cn.gongler.util.concurrent;

/*
 * created by gongler at 2016.7.25.
 *
 *
 */

import cn.gongler.util.ITask;
import cn.gongler.util.Recently;
import cn.gongler.util.concurrent.LinkedTable.LinkedNode;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.function.BiConsumer;

import static cn.gongler.util.GonglerUtil.StartDaemonThread;

/**
 * 多线程同步处理接收的数据，且能保证同一车辆数据的单线程顺序执行。
 *
 * @author gongler
 * @since 2016.08.01
 */
public class ConcurrentBusExecutor implements BiConsumer<Long, ITask> {

    private static final long serialVersionUID = -6245783377880067812L;//ConcurrentBusExecutor @since 2016-09-12

    protected final Thread[] threads;
    private final Map<Long, BusContext> busMap = new ConcurrentHashMap<>();
    private final LinkedTable<BusContext> busLinkedTable = new LinkedTable<>(BusContext.class);
    private final Recently<TaskWrapper> taskHis = new Recently<>(30);//20160909add
    private final Recently<TaskWrapper> tooLongTimeTaskHis = new Recently<>(15);//20160912add

    public ConcurrentBusExecutor(int nThread) {
        threads = new Thread[nThread];
        for (int i = 0; i < nThread; i++) {
            threads[i] = StartDaemonThread("ConcurrentBusExecutor.worker" + i, () -> {
                LinkedNode previousNode = busLinkedTable.headNode();
                int frame = 0;
                while (true) {
                    frame++;
                    //SystemOutPrintln("frame" + (frame++) + ", " + Thread.currentThread());
                    LinkedTable<BusContext>.LinkedNode nextWaitingNode = nextWaitingBus(previousNode);
                    if (nextWaitingNode == null) {//如果没有任何任务需要处理，则休眠后再检查
                        Thread.sleep(100L);////Thread.sleep(1000L);//
                        continue;
                    } else {
                        previousNode = nextWaitingNode;
                    }
                    BusContext bus = nextWaitingNode.get();
                    try (CloseRegister end = CloseRegister.of(bus::leaveQueue)) {//try {//java bug:
                        TaskWrapper task;
                        while ((task = bus.pollTask()) != null) {
                            try {
                                task.execute();
                            } catch (Throwable ex) {//catch (SQLException | 
                                ex.printStackTrace();
                            }
                        }
                    }
                }
            });
        }
        log("\n init finished.");
    }

//    public int threadCount() {
//        return threads.length;
//    }

    /**
     * @param busKey 保证相同busKey任何一个时刻都只有1个线程在处理，以便规避数据库死锁。
     * @param task   任务
     */
    @Override
    public void accept(Long busKey, ITask task) {//public void accept(Long busKey, Runnable task) {
        TaskWrapper taskWrap = new TaskWrapper(busKey, task);
        log("push()" + busKey + ", " + task);
        taskHis.push(taskWrap);
        boolean ret = Bus(busKey).push(taskWrap);//return preQueue.offer(Tuple2.of(Bus(busKey), task));
        //log("after push():" + busLinkedTable.toDetailString());
    }

    /**
     * @param previousNode
     * @return 返回下一个等待处理的节点，可以是自己（自己放在最后一个），如果没找到，则返回null
     * @throws InterruptedException
     */
    private LinkedTable<BusContext>.LinkedNode nextWaitingBus(LinkedTable<BusContext>.LinkedNode previousNode) throws InterruptedException {
        return previousNode.loopFindNextNormalNode(node -> node.get().isReadReadly());
    }

    public long size() {
        long count = 0;
        for (BusContext bus : busLinkedTable) {
            count += bus.queue.size();
        }
        return count;
    }

    public long total() {
        long count = 0;
        for (BusContext bus : busLinkedTable) {
            count += bus.pushCount;
        }
        return count;
    }

    public Map<Long, Integer> toMap() {
        Map<Long, Integer> map = new TreeMap<>();
        for (BusContext bus : busLinkedTable) {
            if (!bus.queue.isEmpty()) {
                map.put(bus.busId(), bus.queue.size());
            }
        }
        return map;
    }

    public Map<Long, Collection<TaskWrapper>> toTaskMap() {
        Map<Long, Collection<TaskWrapper>> map = new TreeMap<>();
        for (BusContext bus : busLinkedTable) {
            Collection tasks = bus.tasks();
            if (!bus.queue.isEmpty()) {//if (!tasks.isEmpty()) {
                map.put(bus.busId(), Collections.unmodifiableCollection(tasks));
            }
        }
        return map;
    }

    public Collection<TaskWrapper> getTasks(Long busId) {
        LinkedNode node = busLinkedTable.findNode(bus -> bus.busId().equals(busId));
        BusContext bus = (BusContext) node.get();
        return Collections.unmodifiableCollection(bus.queue);
    }

    private BusContext Bus(Long busKey) {
        BusContext bus = busMap.computeIfAbsent(busKey, id -> {
            BusContext b = new BusContext(id);
            busLinkedTable.add(b);
            return b;
        });
        return bus;
    }

    public class TaskWrapper {

        private final Long busId;
        private final ITask task;
        private final LocalDateTime createdTime;
        private LocalDateTime beginToHandleTime = null, finishedTime = null;
        private long millis = -1L;

        TaskWrapper(Long busId, ITask task) {
            this.busId = busId;
            this.task = task;
            createdTime = LocalDateTime.now();
        }

        void execute() throws Exception {
            beginToHandle();
            try {
                task.run();
            } finally {
                finished();
            }
        }

        void beginToHandle() {
            beginToHandleTime = LocalDateTime.now();
        }

        void finished() {
            finishedTime = LocalDateTime.now();
            millis = Duration.between(beginToHandleTime, finishedTime).toMillis();//20160912add
            if (millis > 2000L) {
                tooLongTimeTaskHis.push(this);
            }
        }

        public Long busId() {//gongleradd_20160912_1316
            return busId;
        }

        public ITask task() {//gongleradd_20160912_1316
            return this.task;
        }

        public long millns() {
            return millis;
        }

        public LocalDateTime createdTime() {
            return this.createdTime;
        }

        public LocalDateTime beginToHandleTime() {
            return this.beginToHandleTime;
        }

        public LocalDateTime finishedTime() {
            return this.finishedTime;
        }

        public TaskState taskState() {
            final LocalDateTime beginToHandleTime = this.beginToHandleTime, finishedTime = this.finishedTime;
            TaskState stat = beginToHandleTime == null ? TaskState.TASK_WAITING : finishedTime == null ? TaskState.TASK_RUNNING : TaskState.TASK_FINISHED;
            return stat;
        }

        @Override
        public String toString() {
            final LocalDateTime beginToHandleTime = this.beginToHandleTime, finishedTime = this.finishedTime;
            TaskState stat = taskState();//beginToHandleTime == null ? TaskState.TASK_WAITING : finishedTime == null ? TaskState.TASK_RUNNING : TaskState.TASK_FINISHED;
            StringBuilder buf = new StringBuilder(128).append("task_").append(createdTime.toLocalTime()).append("_").append(stat).append("(");
            switch (stat) {
                case TASK_WAITING:
                    buf.append("waiting:").append(Duration.between(createdTime, LocalDateTime.now()));
                    break;
                case TASK_RUNNING:
                    buf.append("waited:").append(Duration.between(createdTime, beginToHandleTime)).append(", executing:").append(Duration.between(beginToHandleTime, LocalDateTime.now()));
                    break;
                case TASK_FINISHED:
                    buf.append("waited:").append(Duration.between(createdTime, beginToHandleTime)).append(", executed:").append(Duration.between(beginToHandleTime, finishedTime));
                    break;
            }
            return buf.append(")_").append(task()).toString();
        }

    }

    public enum TaskState {
        TASK_WAITING,
        TASK_RUNNING,
        TASK_FINISHED
    }

    private static class BusContext {

        private final Long busId;
        private final BlockingQueue<TaskWrapper> queue = new LinkedBlockingQueue<>();
        TaskWrapper previousPollTask = null;
        long pushCount = 0;
        long offerCount = 0;

        public BusContext(Long busKey) {
            this.busId = busKey;
        }

        public Long busId() {
            return busId;
        }

        private boolean push(TaskWrapper task) {
            pushCount++;
            return queue.offer(task);
        }

        TaskWrapper pollTask() {
            TaskWrapper ret = queue.poll();
            if (ret != null) {
                offerCount++;
                this.previousPollTask = ret;
            }
            return ret;
        }

        public Collection<TaskWrapper> tasks() {
            List<TaskWrapper> ret = new ArrayList<>();
            TaskWrapper prev = this.previousPollTask;
            if (prev != null) {//if (prev != null && prev.taskState() != TaskState.TASK_FINISHED) {
                ret.add(prev);
            }
            ret.addAll(queue);
            return ret;
        }

        private boolean using = false;
        private Thread usingThread = null;

        synchronized boolean isReadReadly() {//有待处理数据，且无人占用。
            if (!queue.isEmpty() && using == false) {
                using = true;
                usingThread = Thread.currentThread();
                return true;
            } else {
                return false;
            }
        }

        synchronized void leaveQueue() {
            using = false;
            usingThread = null;
            log("releaveQueue()//" + this);
        }

        @Override
        public String toString() {
            return busId() + "_" + using + "_" + usingThread + "_queue:" + queue.size();

        }

    }

    private static boolean debug = false;

    private static void log(String msg) {
        if (debug) {
            System.out.println(msg);
        }
    }

    public Recently<TaskWrapper> tooLongTimeTaskHis() {//20160914add
        return tooLongTimeTaskHis;
    }

    public Recently<TaskWrapper> taskHis() {//20160914add
        return taskHis;
    }

    @Override
    public String toString() {
        return this.getClass().getSimpleName() + ":" + size() + "/" + total() + ",threads:" + this.threads.length;
    }

}
