/*
 */
package cn.gongler.util.collection;

import java.time.Duration;
import java.util.AbstractQueue;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.TimeUnit;

/**
 * 系统提供的DelayQueue并不方便，特实现便利延迟队列
 *
 * @author gongler
 * @since 2019.05.22
 */
public class EasyDelayQueue<E> extends AbstractQueue<E> implements BlockingQueue<E> {

    private static final long serialVersionUID = 1L;

    private final long defaultDelayMillis;
    final DelayQueue<Entry<E>> queue;

    public EasyDelayQueue(Duration defaultDelay) {
        this.defaultDelayMillis = defaultDelay.toMillis();
        this.queue = new DelayQueue<>();
    }

    @Override
    public Iterator<E> iterator() {
        Iterator<Entry<E>> itr = queue.iterator();
        return new Iterator<E>() {
            @Override
            public boolean hasNext() {
                return itr.hasNext();
            }

            @Override
            public E next() {
                Entry<E> entry = itr.next();
                return entry == null ? null : entry.getValue();
            }
        };
    }

    @Override
    public int size() {
        return queue.size();
    }

    @Override
    public boolean offer(E e) {
        return queue.offer(new Entry(e, defaultDelayMillis));
    }

    public boolean offer(E e, Duration delay) {
        long millis = delay.toMillis();
        return queue.offer(new Entry(e, millis));
    }

    @Override
    public E poll() {
        Entry<E> entry = queue.poll();
        return entry == null ? null : entry.getValue();
    }

    @Override
    public E peek() {
        Entry<E> entry = queue.peek();
        return entry == null ? null : entry.getValue();
    }

    @Override
    public void put(E e) throws InterruptedException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public boolean offer(E e, long timeout, TimeUnit unit) throws InterruptedException {
        return queue.offer(new Entry(e, this.defaultDelayMillis), timeout, unit);
    }

    @Override
    public E take() throws InterruptedException {
        return queue.take().getValue();
    }

    @Override
    public E poll(long timeout, TimeUnit unit) throws InterruptedException {
        Entry<E> entry = queue.poll(timeout, unit);
        return entry == null ? null : entry.getValue();
    }

    @Override
    public int remainingCapacity() {
        return queue.remainingCapacity();
    }

    @Override
    public int drainTo(Collection<? super E> c) {
        return drainTo(c, Integer.MAX_VALUE);
    }

    @Override
    public int drainTo(Collection<? super E> c, int maxElements) {
        Collection<Entry<E>> list = new ArrayList<>(maxElements);
        int count = queue.drainTo(list, maxElements);
        list.forEach(e -> c.add(e.getValue()));
        return count;
    }

    static class Entry<T> implements Delayed {

        final long destTime;
        final T e;

        Entry(T e, long millis) {
            this.destTime = System.currentTimeMillis() + millis;
            this.e = e;
        }

        @Override
        public long getDelay(TimeUnit unit) {
            return unit.convert((destTime - System.currentTimeMillis()), TimeUnit.MILLISECONDS);
        }

        @Override
        public int compareTo(Delayed o) {
            return Long.compareUnsigned(this.getDelay(TimeUnit.MILLISECONDS), o.getDelay(TimeUnit.MILLISECONDS));
        }

        private T getValue() {
            return e;
        }
    }

}
