package cn.gongler.util.bytes;

import cn.gongler.util.GonglerUtil;
import cn.gongler.util.bytes.function.IntByteConsumer;

import java.util.Arrays;
import java.util.function.IntConsumer;
import java.util.function.Supplier;
import java.util.stream.IntStream;

import static cn.gongler.util.bytes.HexUtil.*;

/**
 * 字节数据片段接口。超出片段外抛出异常；增加负值index；支持流；支持数组扩容
 */
public interface IBytesRange {
    byte[] buf();

    int from();

    int size();

    /**
     * 正值从头开始计算偏移;负值从尾开始计算偏移
     *
     * @param index 索引
     * @return 索引位置的字节值
     */
    default int getByte(int index) {
        GonglerUtil.require((index >= 0 && index < size()) || (index < 0 && index >= -size()));
        return buf()[from() + (index >= 0 ? index : size() + index)];
    }

    default IBytesRange getSubRange(int pos, int size) {
        int absoluteIndex = from() + pos >= 0 ? pos : size() + pos;
        GonglerUtil.require((absoluteIndex >= 0 && absoluteIndex < from() + size()) && pos + size <= size());
        return IBytesRange.of(this::buf, from() + pos, size);
    }

    default String toHex() {
        return BytesToHex(buf(), from(), size());
    }

    default String toHexMessage() {
        return "(size:" + size() + ")" + BytesToHex(buf(), from(), size(), " ");
    }

//    default String toString(){return toHex();}


    default byte[] toBytes() {
        return Arrays.copyOfRange(buf(), from(), from() + size());//存在隐患: return from() == 0 && from() + size() == buf().length ? buf() : Arrays.copyOfRange(buf(), from(), from() + size());
    }

    default byte[] toBytes(int pos, int size) {
        int absoluteIndex = from() + (pos >= 0 ? pos : size() + pos);
        GonglerUtil.require((absoluteIndex >= 0 && absoluteIndex < from() + size()) && pos + size <= size());
        return Arrays.copyOfRange(buf(), absoluteIndex, absoluteIndex + size);
    }

    default IBytesRange println() {
        System.out.println(BytesToMessage(toBytes()));
        return this;
    }


    default void forEach(IntByteConsumer consumer) {
        for (int i = 0; i < size(); i++) {
            consumer.accept(i, getByte(i));
        }
    }

    default void forEach(IntConsumer consumer) {
        for (int i = 0; i < size(); i++) {
            consumer.accept(getByte(i));
        }
    }

    default IntStream toIntStream() {//2022年10月16日
        return IntStream.range(0, size()).map(this::getByte);
    }

    static IBytesRange of(Supplier<byte[]> bufGetter, int from, int size) {
        return SimpleByteArrayRange.of(bufGetter, from, size);
    }

    static IBytesRange of(Supplier<byte[]> bufGetter) {
        return SimpleByteArrayRange.of(bufGetter, 0, bufGetter.get().length);
    }

    static IBytesRange of(byte[] buf, int from, int size) {
        return SimpleByteArrayRange.of(() -> buf, from, size);
    }

    static IBytesRange of(byte[] buf) {
        return SimpleByteArrayRange.of(() -> buf, 0, buf.length);
    }

    static IBytesRange of(Byte[] buf, int from, int size) {
        byte[] ret = new byte[size];
        for (int i = 0; i < size; i++) {
            ret[i] = buf[from + i];
        }
        return of(ret);
    }

    static IBytesRange of(Byte[] buf) {
        return of(buf, 0, buf.length);
    }

    static IBytesRange of(CharSequence hex) {
        return of(HexToBytes(hex));
    }

}
