package cn.gongler.util.bytes;

import cn.gongler.util.GonglerUtil;
import cn.gongler.util.ITask;
import cn.gongler.util.annotation.Unstable;
import cn.gongler.util.io.DelegateOutputStream;

import java.io.*;
import java.lang.Byte;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.nio.ByteOrder;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.*;
import java.util.function.*;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import static cn.gongler.util.GonglerUtil.*;
import static cn.gongler.util.bytes.Bytes.EMPTY_BYTES;
import static cn.gongler.util.bytes.HexUtil.*;

/**
 * 字节数组构造器
 *
 * @author gongler
 * @since 3.1
 */
public class BytesBuilder implements ToBytes {

    private static final long serialVersionUID = 8467353696428321139L;//BytesBuilder @since 2016-08-19

    /**
     * 推荐的通用实例。适用于各国。
     * addNum()大端字节顺序；addString()中默认字符集UTF_8，定长字节不足时默认右侧填充'\0'
     * @return 新建实例。
     */
    public static BytesBuilder of() {
        return BigEndian();
    }

    /**
     * 这里方法将创建支持中国最新字符集标准。目前是GB18030，该标准兼容GB2312、GBK。
     * @return 新建实例。
     */
    public static BytesBuilder GBK() {
        return of().gbk();
    }

    /**
     * 提供支持append(hex)方法的视图支持。简化从StringBuilder/StringBuffer移植
     *
     * @return HexBuilder
     */
    protected static HexBuilder supportAppendHex() {//2022年10月24日不再对外提供
        return of().asHexBuilder();
    }

    public static BytesBuilder LittleEndian() {
        return new BytesBuilder().byteOrder(ByteOrder.LITTLE_ENDIAN);
    }

    private static BytesBuilder BigEndian() {
        return new BytesBuilder().byteOrder(ByteOrder.BIG_ENDIAN);
    }

    public static BytesBuilder StringFilledSpace() {
        return new BytesBuilder().stringFilledSpace();
    }

    private ByteOrder byteOrder = ByteOrder.BIG_ENDIAN;

    /**
     * 初始容量，不够时会自动扩展
     */
    protected final int initCapacity;


    /**
     *
     */
    protected byte[] buf;//自动扩展
    /**
     *
     */
    protected int pos = 0;

    private final BitsBuilder bitBuilder = new BitsBuilder(this::addByte);

    protected int mark = 0;

    private Map<Integer, StringJoiner> traceMap = new LinkedHashMap<>();

    private BytesBuilder _note(Object obj) {
        StringJoiner join = traceMap.computeIfAbsent(pos, a -> new StringJoiner(","));
        join.add(Objects.toString(obj));
        return this;
    }

    /**
     *
     */
    protected BytesBuilder() {
        this(512);
    }

    /**
     * @param initCapacity capacity
     */
    protected BytesBuilder(int initCapacity) {
        this.initCapacity = initCapacity;
        this.buf = new byte[initCapacity];
    }

//    public BytesBuilder initCapacity(int initCapacity) {
//        if (initCapacity > buf().length) {
//            prepareNextWriteBytes(initCapacity - pos);
//        }
//        return this;
//    }


    //////////////////////////////////////////////////////////////////
    //构建和配置

    private BytesBuilder byteOrder(boolean isBigEndian) {
        this.byteOrder = isBigEndian ? ByteOrder.BIG_ENDIAN : ByteOrder.LITTLE_ENDIAN;
        return this;
    }

    public BytesBuilder byteOrder(ByteOrder byteOrder) {
        this.byteOrder = byteOrder;
        return this;
    }

    public ByteOrder byteOrder() {
        return byteOrder;
    }

    public boolean isBigEndian() {
        return ByteOrder.BIG_ENDIAN.equals(this.byteOrder);
    }

    private Charset charset = StandardCharsets.UTF_8;//2022年10月11日

    public BytesBuilder charset(Charset charset) {
        this.charset = charset;
        return this;
    }

    public BytesBuilder utf8() {
        this.charset = StandardCharsets.UTF_8;
        return this;
    }

    public BytesBuilder gbk() {
        this.charset = Charset.forName("GB18030");
        return this;
    }

    public Charset charset() {
        return charset;
    }

    public int nullValType = 1;//有时未赋值，只是需要占位。因此默认改为允许默认null，测试时可以用更严格模式。//0空值抛出异常（严格模式）；1空值占位（容错模式，适宜调试场景）；2空值不写（无意义。不推荐。推荐采用字节数值控制）；

    public BytesBuilder exceptionIfNullVal() {
        this.nullValType = 0;
        return this;
    }

    public BytesBuilder placeHolderIfNullVal() {
        this.nullValType = 1;
        return this;
    }

    //    public BytesBuilder ignoreIfNullVal() {
//        this.nullValType = 2;
//        return this;
//    }
    public BytesBuilder nullValType(int nullValType) {
        this.nullValType = nullValType;
        return this;
    }

    /**
     * 是否字符串尾部需要添加'\0'(C风格字符串)
     *
     * @param enable
     * @return this
     */
    private BytesBuilder stringTailNul(boolean enable) {
        this.stringTailNul = enable;
        return this;
    }

    /**
     * 字符串尾部自动添加'\0'(C风格字符串)
     *
     * @return this
     */
    public BytesBuilder stringTailNul() {
        this.stringTailNul = true;
        return this;
    }


    //////////////////////////////////////////////////////////////////
    //内部方法等

    protected byte[] buf() {//由于会扩容，因此需要获取到最新buf的途径
        return buf;
    }

    /**
     * @param newBytes 字节数
     */
    protected void prepareNextWriteBytes(int newBytes) {
        GonglerUtil.require(pos + newBytes <= limited);
        if (pos + newBytes > buf.length) {
            buf = Arrays.copyOf(buf, 64 + Math.max(buf.length * 2, this.pos + newBytes));//20160216debug 原数组大小翻倍，并不能保证肯定容下新增数据。
        }
    }


    /**
     * @param byteCnt 字节数
     * @return this
     */
    public BytesBuilder skip(int byteCnt) {
        requireBitsFinished();
        prepareNextWriteBytes(byteCnt);
        movePos(byteCnt);//pos += bytes;
        return this;
    }


    protected void movePos(int byteCnt) {
        this.pos += byteCnt;
    }

    public BytesBuilder mark() {
        this.mark = this.pos;
        return this;
    }

    public BytesBuilder moveToMark() {
        this.pos = this.mark;
        return this;
    }

    protected int pos() {
        return pos;
    }

    protected BytesBuilder pos(int newPos) {
        final int bufSize = buf().length;
        if (newPos > bufSize) {
            prepareNextWriteBytes(newPos - bufSize);
        }
        pos = newPos;
        return this;
    }

    public int size() {//对使用者pos之后是预留缓存是不存在的。只有pos前字节数据可见可知
        return pos;
    }

    public BytesBuilder toSize(int newSize) {//2022年10月13日
        return pos(newSize);
    }


    /**
     * 只能修改游标之前的字节，用于逆序输出字节值。可以配合skip(bytes)方法使用。
     *
     * @param backIndex 回退字节数
     * @param byteVal   值
     * @return this
     */
    public BytesBuilder setBack(int backIndex, int byteVal) {
        if (backIndex <= 0) {
            throw new IllegalArgumentException("为了防止忘记移动游标，强制只能修改之前数据");
        }
        buf()[pos - backIndex] = (byte) byteVal;
        return this;
    }

    private int limited = Integer.MAX_VALUE;

    public BytesBuilder limited(int limited) {
        this.limited = limited;
        return this;
    }

    public int limited() {
        return limited;
    }

    private <T> T handleNullVal(T val, T defaultVal) {
        if (val == null) {
            if (nullValType == 0) {
                throw new NullPointerException();
            } else if (nullValType == 1) {
                return defaultVal;
            }
        }
        return val;
    }

    /**
     * 有时会遇到有的字节需要后面字节生成后才能确定要写什么值。例：字节数、校验等。
     *
     * @return 副本
     */
    @Unstable
    public BytesBuilder duplicate() {//2022年10月6日
        Supplier<byte[]> bufSupplier = this::buf;
        BytesBuilder view = new BytesBuilder() {
            public byte[] buf() {
                return buf = bufSupplier.get();
            }
        };
        view.buf = this.buf();
        view.pos = view.mark = this.pos;
        view.byteOrder = this.byteOrder;
        view.limited = this.limited;
        return view;
    }

    @Unstable
    public BytesBuilder duplicate(int newPos) {//2022年10月6日
        BytesBuilder ret = duplicate();
        return duplicate().pos(newPos);
    }

    /**
     * 有时会遇到有的字节需要后面字节生成后才能确定要写什么值。例：字节数、校验等。
     *
     * @param byteCnt 跳过的字节数。也是副本最多可写的字节数
     * @return 副本
     */
    @Unstable
    public BytesBuilder slice(int byteCnt) {//2022年10月6日
        BytesBuilder view = duplicate();
        view.limited = byteCnt;
        this.skip(byteCnt);
        return view;
    }

//    @Unstable
//    public Runnable consumer(int byteCnt, Supplier<Bytesable> item) {
//        Supplier<byte[]> getBuf = ()->buf();
//        BytesBuilder view = new BytesBuilder(){
//            public byte[] buf(){
//                return buf = getBuf.get();
//            }
//        };
//        view.buf = this.buf();
//        view.pos = this.pos;
//        view.defaultIsBigEndian = this.defaultIsBigEndian;
//        view.limited = byteCnt;
//
//        this.skip(byteCnt);
//        return ()->item.get().toBytes(view);
//    }

    protected BytesBuilder fillValueAtPos(int byteValue) {//buf[pos]=
        buf()[pos] = (byte) byteValue;
        return this;
    }


    //////////////////////////////////////////////////////////////////
    //规划的扩展方法

    /**
     * 允许用绝对位置修改值
     *
     * @param globalIndex 全局索引
     * @param byteVal     字节取值
     * @return this
     */
    public BytesBuilder setByte(int globalIndex, int byteVal) {//2022年10月5日
        buf()[globalIndex] = (byte) byteVal;
        return this;
    }

    public BytesBuilder setBytes(int globalIndex, byte[] block) {
        for (int i = 0; i < block.length; i++) {
            buf()[globalIndex + i] = block[i];
        }
        return this;
    }

    public BytesBuilder setBytes(int globalIndex, ToBytes item) {
        byte[] block = item.toBytes();
        for (int i = 0; i < block.length; i++) {
            buf()[globalIndex + i] = block[i];
        }
        return this;
    }

    public BytesBuilder setNum(int globalIndex, int byteCnt, long val) {
        duplicate().pos(globalIndex).addNum(byteCnt, val);
        return this;
    }

    @Unstable
    public BytesBuilder setString(int globalIndex, int byteCnt, CharSequence msg, Charset charset) {
        duplicate().pos(globalIndex).addString(byteCnt, msg, charset);
        return this;
    }

    //////////////////////////////////////////////////////////////////
    //add 内部方法


    protected BytesBuilder _addByte(int byteVal) {
//        requireBitsFinished();
//        prepareNextWriteBytes(1);
        fillValueAtPos(byteVal);//buf[pos] = (byte) byteVal;
        movePos(1);//
        return this;
    }

    /**
     * @param buf  数组
     * @param from 起始位置
     * @param size 允许大于buf.length
     * @return this
     */
    protected BytesBuilder _addBytes(byte[] buf, int from, int size) {
        requireBitsFinished();
        if (size > 0) {//0时省略写的动作
            prepareNextWriteBytes(size);
            System.arraycopy(buf, from, buf(), pos, Math.min(buf.length - from, size));
            movePos(size);//pos+=length;
        }
        return this;
    }

    //////////////////////////////////////////////////////////////////
    //add byte

    /**
     * 添加1个字节
     *
     * @param byteVal byte value
     * @return this
     */
    public BytesBuilder addByte(int byteVal) {
        requireBitsFinished();
        prepareNextWriteBytes(1);
        return _addByte(byteVal);
    }

    public BytesBuilder addByte(Byte byteVal) {
        byteVal = handleNullVal(byteVal, (byte) 0);
        return addByte(byteVal.intValue());
    }


    public BytesBuilder addByteRepeat(int byteVal, int repeatTimes) {
        for (int i = 0; i < repeatTimes; i++) {
            addByte(byteVal);
        }
        return this;
    }

    //////////////////////////////////////////////////////////////////
    //BCD

    public BytesBuilder addBcd(int byteCnt, Long val) {
        val = handleNullVal(val, 0L);
        return addBcd(byteCnt, val.longValue());
    }

    public BytesBuilder addBcd(int byteCnt, Integer val) {
        val = handleNullVal(val, 0);
        return addBcd(byteCnt, val.longValue());
    }

    public BytesBuilder addBcd(int byteCnt, long val) {
        this.skip(byteCnt);
        for (int i = 0; i < byteCnt; i++) {
            int byteVal = (int) (val % 100);
            int high = byteVal / 10;
            int low = byteVal % 10;
            this.setBack(1 + i, (high << 4) | low);
            val /= 100;
        }
        return this;
    }

    private final static BigInteger INT100 = new BigInteger("100");
    private final static BigInteger INT_FF = new BigInteger("255");

    public BytesBuilder addBcd(int byteCnt, BigInteger value) {
        this.skip(byteCnt);
        value = handleNullVal(value, BIG_INTEGER_0);
        for (int i = 0; i < byteCnt; i++) {
            int byteVal = value.mod(INT100).intValue();
            int high = byteVal / 10;
            int low = byteVal % 10;
            this.setBack(1 + i, (high << 4) | low);
            value = value.divide(INT100);
        }
        return this;
    }

    /**
     * 处理BCD超过8字节的情况
     *
     * @param byteCnt 字节数
     * @param bcd     BCD值
     * @return this
     */
    public BytesBuilder addBcd(int byteCnt, String bcd) {
        bcd = handleNullVal(bcd, "0");
        return addBcd(byteCnt, new BigInteger(bcd));
    }

    //////////////////////////////////////////////////////////////////
    //add Java Number class

    /**
     * @param value value
     * @return this
     */
    public BytesBuilder addShort(Short value) {
        value = handleNullVal(value, (short) 0);
        return addNum(2, value);
    }

    /**
     * @param value value
     * @return this
     */
    public BytesBuilder addInt(Integer value) {
        value = handleNullVal(value, 0);
        return addNum(4, value);
    }

    /**
     * @param value value
     * @return this
     */
    public BytesBuilder addLong(Long value) {//20160315gongler DEBUG public BytesBuilder addLong(int b) {
        value = handleNullVal(value, 0L);
        return addNum(8, value);
    }


    /**
     * 如果用12.134 按0.001为单位保存。则scale是3
     *
     * @param byteCnt 字节数
     * @param value   含小数的数
     * @param scale   正数表示单位为10^scale。可以参考Oracle中的NUMBER（precision, scale）
     * @return this
     */
    public BytesBuilder addDouble(int byteCnt, Double value, int scale) {//2022年10月19日
        value = handleNullVal(value, 0D);
        //long longVal = (long) (value.doubleValue() * Math.pow(10, scale));
        long longVal = new BigDecimal(value).movePointRight(scale).setScale(0, RoundingMode.HALF_UP).longValue();
        return addNum(byteCnt, longVal);
    }
    //////////////////////////////////////////////////////////////////
    //addNum

    /**
     * 默认大端字节顺序
     *
     * @param byteCnt 字节数
     * @param value   值
     * @return builder
     */
    public BytesBuilder addNum(int byteCnt, long value) {
        return isBigEndian() ? addNumBigEndian(byteCnt, value) : addNumLittleEndian(byteCnt, value);
    }

    public BytesBuilder addNum(int byteCnt, Long value) {
        value = handleNullVal(value, 0L);
        return addNum(byteCnt, value.longValue());
    }

    public BytesBuilder addNum(int byteCnt, Integer value) {
        value = handleNullVal(value, 0);
        return addNum(byteCnt, value.longValue());
    }

    public BytesBuilder addNum(int byteCnt, Integer value, Integer defaultVal) {
        return addNum(byteCnt, WithDefault(value, defaultVal).longValue());
    }

    public BytesBuilder addNum(int byteCnt, Short value) {
        value = handleNullVal(value, (short) 0);
        return addNum(byteCnt, value.longValue());
    }

    public BytesBuilder addNum(int byteCnt, Number value) {//2022年10月13日
        value = handleNullVal(value, 0);
        return addNum(byteCnt, value.longValue());
    }

    /**
     * @param value 空值会抛异常
     * @return this
     */
    public BytesBuilder addNum1(long value) {//快捷方式不允许空值参数
        return addNum(1, value);
    }

    public BytesBuilder addNum2(long value) {
        return addNum(2, value);
    }

    public BytesBuilder addNum3(long value) {
        return addNum(3, value);
    }

    public BytesBuilder addNum4(long value) {
        return addNum(4, value);
    }

    public BytesBuilder addNum8(long value) {
        return addNum(8, value);
    }

    /**
     * 小端字节顺序
     *
     * @param byteCnt 字节数
     * @param value   值
     * @return this
     */
    public BytesBuilder addNumLittleEndian(int byteCnt, long value) {
        _note("LE" + byteCnt + ":" + value);
        for (int i = 0; i < byteCnt; i++) {
            addByte(((int) (value >> (8 * i))) & 0xFF);
        }
        return this;
    }

    public BytesBuilder addNumBigEndian(int byteCnt, long value) {
        _note("BE" + byteCnt + ":" + value);
        for (int i = byteCnt - 1; i >= 0; i--) {
            addByte(((int) (value >> (8 * i))) & 0xFF);
        }
        return this;
    }

    public BytesBuilder addNum(int byteCnt, long value, ByteOrder byteOrder) {
        if (ByteOrder.BIG_ENDIAN.equals(byteOrder)) {
            addNumBigEndian(byteCnt, value);
        } else {
            addNumLittleEndian(byteCnt, value);
        }
        return this;
    }

    /**
     * 自动解码整数字符串。可以参看Long::decode说明
     *
     * @param byteCnt 字节数
     * @param decode  "255", "0xFF", "#FF", "0b111111", "-12"
     * @return this
     */
    public BytesBuilder addNum(int byteCnt, String decode) {
        long val = DecodeLong(decode);
        return addNum(byteCnt, val);
    }

    private static final BigInteger BIG_INTEGER_0 = new BigInteger("0");

    //////////////////////////////////////////////////////////////////
    //BigInteger
    public BytesBuilder addNum(int byteCnt, BigInteger value) {
        value = handleNullVal(value, BIG_INTEGER_0);
        return isBigEndian() ? addNumBigEndian(byteCnt, value) : addNumLittleEndian(byteCnt, value);
    }

    public BytesBuilder addNumLittleEndian(int byteCnt, BigInteger value) {
        _note("LE" + byteCnt + ":" + value);
        value = handleNullVal(value, BIG_INTEGER_0);
        for (int i = 0; i < byteCnt; i++) {
            addByte(value.shiftRight(8 * i).mod(INT_FF).intValue());
        }
        return this;
    }

    public BytesBuilder addNumBigEndian(int byteCnt, BigInteger value) {
        _note("BE" + byteCnt + ":" + value);
        value = handleNullVal(value, BIG_INTEGER_0);
        for (int i = byteCnt - 1; i >= 0; i--) {
            addByte(value.shiftRight(8 * i).mod(INT_FF).intValue());
        }
        return this;
    }

    public BytesBuilder addNum(int byteCnt, BigInteger value, ByteOrder byteOrder) {
        if (ByteOrder.BIG_ENDIAN.equals(byteOrder)) {
            addNumBigEndian(byteCnt, value);
        } else {
            addNumLittleEndian(byteCnt, value);
        }
        return this;
    }

    //////////////////////////////////////////////////////////////////
    //addNumByHex 十六进制表示的数值类型

    /**
     * addNumByHex
     *
     * @param byteCnt 字节数
     * @param hexNum  HEX字符数可以不是偶数,允许过长，过长左截断
     * @return this
     */
    public BytesBuilder addHex(int byteCnt, CharSequence hexNum) {
        hexNum = handleNullVal(hexNum, "0");
        //requireLessThenOrEqual(hexNum.length(), byteCnt * 2);
        if (hexNum.length() > byteCnt * 2) {//超长则截左
            hexNum = hexNum.subSequence(hexNum.length() - byteCnt * 2, hexNum.length());
        } else if (hexNum.length() < byteCnt * 2) {//不够长则左补'0'
            hexNum = Repeat(byteCnt * 2 - hexNum.length(), "0") + hexNum;
        }

        return addBytes(hexNum);
    }

    /**
     * addBytes(CharSequence hex)的别名。推荐使用addBytes(CharSequence hex)
     * @param hex hex，必须成对
     * @return this
     */
    public BytesBuilder addHex(CharSequence hex) {//2022年10月10日，与addBytes(hex)等价
        return addBytes(hex);
    }

    /**
     * addHex(hex)的别名。这是为方便从StringBuilder/StringBuffer.append(hex)移植，增加的过渡性别名方法，增加迁移的平顺性。
     *
     * @param hex hex字符串
     * @return this
     */
    public BytesBuilder append(CharSequence hex) {//2022年10月23日，addHex(hex)等价
        return addBytes(hex);
    }

    public BytesBuilder addHex(Iterable<String> hexList) {//不要改成Iterable<CharSequence>，否则将无法接收List<String>//2022年10月10日 允许不定长
        requireBitsFinished();
        hexList.forEach(this::addBytes);
        return this;
    }

    /**
     * @param byteCntPerItem 单条统一字节数
     * @param hexList        主要是String的List
     * @return this
     */
    public BytesBuilder addHex(int byteCntPerItem, Iterable<String> hexList) {//2022年10月10日 <String>更合理，且不推荐抽象泛型 Iterable<CharSequence>
        requireBitsFinished();
        hexList.forEach(a -> addBytes(byteCntPerItem, a));
        return this;
    }

    public BytesBuilder addHex(Stream<String> stream) {//2022年10月10日 mapper后不会存在Stream<CharSequence> ,只会是Stream<String>
        requireBitsFinished();
        stream.forEach(this::addBytes);
        return this;
    }

    public BytesBuilder addHex(String... hexArray) {//2022年10月10日 String..更合理 addHex(CharSequence... hexArray)
        requireBitsFinished();
        Arrays.stream(hexArray).forEach(this::addBytes);
        return this;
    }

    public <T> BytesBuilder addHex(Iterable<T> list, Function<T, String> mapper) {//<T, String>更合理：Function<T, CharSequence> mapper)
        list.forEach(a -> addBytes(mapper.apply(a)));
        return this;
    }

    //////////////////////////////////////////////////////////////////
    //addString
    private char stringFilledChar = '\0';

    /**
     * 允许定义定长字符串不足时右侧填充的空白字符。默认是'\0'。
     *
     * @param ch 填充字符。默认是'\0'
     * @return this
     */
    protected BytesBuilder stringFilledChar(char ch) {//疑似过度设计，暂不对外开放。
        stringFilledChar = ch;
        return this;
    }

    public BytesBuilder stringFilledSpace() {
        return stringFilledChar(' ');
    }

    public BytesBuilder stringFilledNul() {
        return stringFilledChar('\0');
    }

    protected BytesBuilder execute(ITask task) {//2022年10月25日
        task.executeWithThrowAny();
        return this;
    }

    private boolean stringTailNul = false;

    /**
     * 默认不追加'\0'。如果需要，请先调用stringTailNul()
     *
     * @param msg msg
     * @return this
     */
    public BytesBuilder addString(CharSequence msg) {
        return addString(0, msg);
    }

    public BytesBuilder addString(CharSequence msg, Charset charset) {
        return addString(0, msg, charset);
    }

    public BytesBuilder addUtf8(CharSequence msg) {
        addString(msg, StandardCharsets.UTF_8);
        return this;
    }

    public BytesBuilder addGbk(CharSequence msg) {
        addString(msg, Charset.forName("GB18030"));
        return this;
    }


    /**
     * byteCnt:正值。定长字符串；负值：含长度头的字符串。
     *
     * @param byteCnt 正值：定长字符串的字节数，不足字节结尾用多个'\0'补齐，过长右截断;负值：前置放一个长度头。长度头字节数由n绝对值定义。
     * @param msg     字符串
     * @return this
     */
    public BytesBuilder addString(int byteCnt, CharSequence msg) {
        return addString(byteCnt, msg, charset());
    }

    public BytesBuilder addString(int byteCnt, CharSequence msg, Charset charset) {
        _note(charset.toString() + "_L" + byteCnt + ":" + msg);
        String msg2 = handleNullVal(msg, "").toString();
        if (stringTailNul) msg2 = msg2 + '\0';
        return addBytes(Str.of(byteCnt, msg2, charset).filledChar(stringFilledChar).toBytes());
    }

    public BytesBuilder addUtf8(int byteCnt, CharSequence msg) {
        addString(byteCnt, msg, StandardCharsets.UTF_8);
        return this;
    }

    public BytesBuilder addGbk(int byteCnt, CharSequence msg) {
        addString(byteCnt, msg, Charset.forName("GB18030"));
        return this;
    }


    /**
     * 长度头+消息。
     * 1个字节长度头存放消息字节数
     *
     * @param msg msg
     * @return this
     */
    public BytesBuilder addL1String(CharSequence msg) {
        return addString(-1, msg, charset());
    }

    public BytesBuilder addL1String(CharSequence msg, Charset charset) {
        return addString(-1, msg, charset);
    }

    /**
     * 长度头+消息。
     * 2个字节头存放消息字节数
     *
     * @param msg msg
     * @return this
     */
    public BytesBuilder addL2String(CharSequence msg) {
        return addString(-2, msg, charset());
    }

    public BytesBuilder addL2String(CharSequence msg, Charset charset) {
        return addString(-2, msg, charset);
    }

    //////////////////////////////////////////////////////////////////
    //数值字符串
    /**
     * 有时字符串有前补齐'0'的需求, 通常取值范围局限在[-_.0-9a-zA-Z]用来表示id的情况。而不会出现多字节字符。所以不需要字符集参数
     * 过长时，左截断。
     *
     * @param byteCnt 字节数。
     * @param prefix  前置符号，通常是'0'或' '
     * @param msg:    ASCII code。过长时，左截断。
     * @return this
     */
    public BytesBuilder addString(int byteCnt, char prefix, CharSequence msg) {
        require(byteCnt > 0, "byteCnt必须大于0:" + byteCnt);
        _note(charset.toString() + "_L" + byteCnt + ",prefix:" + prefix + "," + msg);
        String msg2 = handleNullVal(msg, "").toString();
        byte[] body = msg2.getBytes();
        byte[] prefixArray = new byte[byteCnt];
        for (int i = prefixArray.length - 1, j = body.length - 1; i >= 0; i--, j--) {
            prefixArray[i] = j >= 0 ? body[j] : (byte) prefix;
        }
        return addBytes(prefixArray);
    }

    /**
     * 把正整数作为ASCII码输出，字节不足左补'0'。过长左截断。
     * 是addString(int byteCnt, char prefix, CharSequence msg)的便利方法。
     *
     * @param byteCnt     字节数
     * @param unsignedNum 数值
     * @return this
     */
    public BytesBuilder addString(int byteCnt, Integer unsignedNum) {
        return addString(byteCnt, '0', Integer.toUnsignedString(WithDefault(unsignedNum, 0)));
    }

    public BytesBuilder addString(int byteCnt, Long unsignedNum) {
        return addString(byteCnt, '0', Long.toUnsignedString(WithDefault(unsignedNum, 0L)));
    }

    public BytesBuilder addString(int byteCnt, BigInteger unsignedNum) {
        return addString(byteCnt, '0', WithDefault(unsignedNum, BIG_INTEGER_0).toString());
    }


//    ////////////////////////////////////////////////////////////////////////////
//    private static Map<Class, Function<Object, byte[]>> map = new HashMap<>();
//
//    /**
//     * <pre>
//     * BytesBuilder.register(String.class, String::getBytes);
//     * BytesBuilder build = BytesBuilder.BigEndian();
//     * build.add("abc");
//     * System.out.println("" + build.toString());
//     * </pre>
//     *
//     * @param <T>     T
//     * @param aClass  类
//     * @param factory 工厂
//     */
//    public static <T> void register(Class<T> aClass, Function<T, byte[]> factory) {
//        map.put(aClass, (Function<Object, byte[]>) factory);
//    }
//
//    public BytesBuilder addBytes(Object obj) {
//        requireBitsFinished();
//        Objects.requireNonNull(obj);
//        Class aClass = obj.getClass();
//        do {
//            Function<Object, byte[]> mapper = map.get(aClass);
//            if (mapper != null) {
//                byte[] bytes = mapper.apply(obj);
//                addBytes(bytes);
//                return this;
//            }
//            aClass = aClass.getSuperclass();
//        } while (aClass != null);
//        throw new UnsupportedOperationException(obj.getClass().getName());
//    }

    //////////////////////////////////////////////////////////////////
    //addByte

    public BytesBuilder addByteStream(InputStream in) {//2022年10月6日
        try (InputStream bin = new BufferedInputStream(in)) {
            byte[] inBuf = new byte[1024];
            int cnt;
            while ((cnt = bin.read(inBuf)) >= 0) {
                addBytes(inBuf, 0, cnt);
            }
        } catch (Exception e) {
            throw toRuntimeException(e);
        }
        return this;
    }

    /**
     * IntStream.range(1, 10).forEach(System.out::println);
     * IntStream.generate(Random.next()).forEach(System.out::println);
     *
     * @param stream IntStream
     * @return this
     */
    public BytesBuilder addByteStream(IntStream stream) {//2022年10月6日
        stream.forEach(this::addByte);
        return this;
    }

    public BytesBuilder addByteStream(Stream<Byte> stream) {//2022年10月6日
        stream.mapToInt(Byte::intValue).forEach(this::addByte);
        return this;
    }

    /**
     * IntStream.range(1, 10).forEach(System.out::println);
     * IntStream.generate(Random.next()).forEach(System.out::println);
     *
     * @param stream HexStream 每行一条记录
     * @return this
     */
    public BytesBuilder addHexStream(Stream<CharSequence> stream) {//2022年10月15日
        stream.forEach(this::addBytes);
        return this;
    }

    //////////////////////////////////////////////////////////////////
    //addHexFile

    protected BytesBuilder addHex(Reader hexReader) {//2022年10月6日, 必要性不大，暂不开放
        try (BufferedReader bin = new BufferedReader(hexReader)) {
            String line;
            while ((line = bin.readLine()) != null) {
                String hex = line.split("#")[0].replaceAll("\\s", "");//去掉行注释，并抹掉空白字符
                if (HexUtil.isStrictHex(hex)) {
                    addBytes(hex);
                } else {
                    throw new IllegalArgumentException(hex);
                }
            }
        } catch (Exception e) {
            throw toRuntimeException(e);
        }
        return this;
    }

    public BytesBuilder addHexFile(Path hexFile) {//2022年10月6日
        return addHexFile(hexFile, StandardCharsets.UTF_8);
    }

    private BytesBuilder addHexFile(Path hexFile, Charset charset) {//2022年10月6日 暂不都对外开放
        try {
            return addHex(new InputStreamReader(new FileInputStream(hexFile.toFile()), charset));
        } catch (FileNotFoundException e) {
            throw toRuntimeException(e);
        }
    }

    public BytesBuilder addFile(Path binFile) {//2022年10月6日
        try {
            return addByteStream(Files.newInputStream(binFile));
        } catch (IOException e) {
            throw toRuntimeException(e);
        }
    }

    //////////////////////////////////////////////////////////////////
    //addBytes

    /**
     * @param buf  数组
     * @param from 数组中开始位置
     * @param size 字节数.不能超过buf-from的长度
     * @return this
     */
    public BytesBuilder addBytes(byte[] buf, int from, int size) {
        requireLessThenOrEqual(from + size, buf.length);
        return _addBytes(buf, from, size);
    }


    /**
     * @param bytes 字节数组
     * @return this
     */
    public BytesBuilder addBytes(byte[] bytes) {
        bytes = handleNullVal(bytes, EMPTY_BYTES);
        return _addBytes(bytes, 0, bytes.length);
    }

    public BytesBuilder addBytes(CharSequence hex) {
        hex = handleNullVal(hex, "");
        return addBytes(HexToBytes(hex));
    }

    /**
     * 扩展接口：便于写各种头。例如BCD头
     * @param before 前置逻辑，可用于添加数据长度头
     * @param bytes 字节块
     * @return this
     */
    public BytesBuilder addBytes(BiConsumer<BytesBuilder, Integer> before, byte[] bytes) {
        bytes = handleNullVal(bytes, EMPTY_BYTES);
        before.accept(this, bytes.length);
        return _addBytes(bytes, 0, bytes.length);
    }

    public BytesBuilder addBytes(BiConsumer<BytesBuilder, Integer> before, CharSequence hex) {
        return addBytes(before, HexToBytes(hex));
    }

//    public BytesBuilder addBytes(int byteCntPerItem, List<byte[]> list) {//2022年10月10日
//        list.forEach(a -> addBytes(byteCntPerItem, a));
//        return this;
//    }


    /**
     * 长时截断右侧；短时右补齐
     *
     * @param byteCnt 字节数。
     * @param bytes   输出字节块， 允许null
     * @return this
     */
    public BytesBuilder addBytes(int byteCnt, byte[] bytes) {
        bytes = handleNullVal(bytes, emptyByteArray());
        if (byteCnt < 0) return addBytes((b, l) -> b.addNum(-byteCnt, l), bytes);//彩蛋：负值可用来表示长度头
        return _addBytes(bytes, 0, byteCnt);
    }

    public BytesBuilder addBytes(int byteCnt, CharSequence hex) {
        return addBytes(byteCnt, HexToBytes(hex));
    }

    public BytesBuilder addL1Bytes(byte[] bytes) {//2022年10月29日
        return addBytes(-1, bytes);
    }

    public BytesBuilder addL2Bytes(byte[] bytes) {
        return addBytes(-2, bytes);
    }

    public BytesBuilder addL1Bytes(CharSequence hex) {
        return addL1Bytes(HexToBytes(hex));
    }

    public BytesBuilder addL2Bytes(CharSequence hex) {
        return addL2Bytes(HexToBytes(hex));
    }


    public BytesBuilder addBytes(BytesBuilder another) {//2022年10月6日
        another.requireBitsFinished();
        return addBytes(another.buf(), 0, another.pos);
    }

    public BytesBuilder addBytes(IBytesRange bytesRange) {//2022年10月6日
        return addBytes(bytesRange.buf(), bytesRange.from(), bytesRange.size());
    }


    public BytesBuilder addBytes(Byte[] bytes) {
        bytes = handleNullVal(bytes, emptyArray());
        return addBytes(bytes, 0, bytes.length);
    }

    public BytesBuilder addBytes(Byte[] bytes, int from, int size) {
        for (int i = 0; i < size; i++) {
            addByte(bytes[from + i]);
        }
        return this;
    }

    /**
     * @param item 提供者
     * @return this
     */
    public BytesBuilder addBytes(Bytesable item) {
        requireBitsFinished();
        _note(String.valueOf(item));
        item.toBytes(this);
        return this;
    }

    public BytesBuilder addBytes(Iterable<Bytesable> items) {
        requireBitsFinished();
        items.forEach(a -> a.toBytes(this));
        return this;
    }

    /**
     * 也相当于Supplier
     *
     * @param item 对象
     * @return this
     */
    public BytesBuilder addBytes(ToBytes item) {
        requireBitsFinished();
        _note(String.valueOf(item));
        return addBytes(item.toBytes());
    }

    public BytesBuilder addBytes(List<ToBytes> items) {
        items.forEach(a -> addBytes(a.toBytes()));
        return this;
    }

    //过于繁复，使用不便改为bytes[]    public <T> BytesBuilder addBytes(T obj, Function<T, Bytesable> mapper) {
//        requireBitsFinished();
//        mapper.apply(obj).toBytes(this);
//        return this;
//    }
    public <T> BytesBuilder addBytes(T obj, Function<T, byte[]> mapper) {
        addBytes(mapper.apply(obj));
        return this;
    }

    public <T> BytesBuilder addBytes(Iterable<T> list, Function<T, byte[]> mapper) {
        list.forEach(a -> addBytes(mapper.apply(a)));
        return this;
    }

    public BytesBuilder addBytes(Stream<byte[]> stream) {//2022年10月6日
        stream.forEach(this::addBytes);
        return this;
    }

    ////////////////////////////////////////////////////////////////////////////
    //BIT

    public BytesBuilder addBit(Boolean bitVal) {
        bitBuilder.addBit(WithDefault(bitVal, false));
        return this;
    }

    public BytesBuilder addBit(Boolean bitVal, Boolean defaultVal) {
        bitBuilder.addBit(WithDefault(bitVal, defaultVal));
        return this;
    }

    public BytesBuilder addBitRepeat(Boolean bitVal, int repeatTimes) {
        bitBuilder.addBitRepeat(WithDefault(bitVal, false), repeatTimes);
        return this;
    }

    /**
     * 字节从高位到低位的顺序添加
     *
     * @param bitVal 1 or 0
     * @return this
     */

    public BytesBuilder addBit(int bitVal) {
        bitBuilder.addBit(bitVal);
        return this;
    }

    /**
     * @param bitVal null按0处理
     * @return this
     */
    public BytesBuilder addBit(Integer bitVal) {
        bitBuilder.addBit(WithDefault(bitVal, 0).intValue());
        return this;
    }

    public BytesBuilder addBit(Integer bitVal, Integer defaultVal) {
        bitBuilder.addBit(WithDefault(bitVal, defaultVal).intValue());
        return this;
    }

    public BytesBuilder addBitRepeat(Integer bitVal, int repeatTimes) {
        bitBuilder.addBitRepeat(WithDefault(bitVal, 0), repeatTimes);
        return this;
    }

    public BytesBuilder addBit(int times, IntFunction<Boolean> bitValMapper) {//2022年10月30日
        Repeat(times, no -> addBit(bitValMapper.apply(no)));
        return this;
    }

//    public BytesBuilder addBits(int bitCnt, long val) {
//        bitBuilder.addBits(bitCnt, val);
//        return this;
//    }

    /**
     * @param bitCnt 字节数
     * @param val    null时按0处理
     * @return this
     */
    public BytesBuilder addBits(int bitCnt, Integer val) {
        return addBits(bitCnt, val, 0);
    }

    public BytesBuilder addBits(int bitCnt, Integer val, Integer defaultVal) {
        bitBuilder.addBits(bitCnt, WithDefault(val, defaultVal).longValue());
        return this;
    }

    public BytesBuilder addBits(int bitCnt, Long val) {
        return addBits(bitCnt, val, 0L);
    }

    public BytesBuilder addBits(int bitCnt, Long val, Long defaultVal) {
        bitBuilder.addBits(bitCnt, WithDefault(val, defaultVal).longValue());
        return this;
    }

    public BytesBuilder addBits(List<Boolean> bits) {
        bitBuilder.addBits(bits);
        return this;
    }

    /**
     *
     * @param bitsStr example: "01001001"
     * @return this
     */
    public BytesBuilder addBits(CharSequence bitsStr) {
        bitBuilder.addBits(bitsStr);
        return this;
    }

//过度设计
// /**
//     * @param byteCnt   less than or equals 8
//     * @param trueBitIndexs bitIndex List
//     * @return this
//     */
//    public BytesBuilder addBits(int byteCnt, Iterable<Integer> trueBitIndexs) {
//        Bits bits = Bits.of().byteSize(byteCnt);
//        for (Integer index : trueBitIndexs) {
//            requireLessThen(index, byteCnt * 8);
//            bits.bit(index, true);
//        }
//        addNum(byteCnt, bits);
//        return this;
//    }

    /**
     * @param byteCnt   less than or equals 8
     * @param trueBitIndexs bitIndex List
     * @return this
     */
    public BytesBuilder addBits(int byteCnt, IntStream trueBitIndexs) {
        Bits bits = Bits.of().byteSize(byteCnt);
        trueBitIndexs.forEach(a->{
            requireLessThen(a, byteCnt * 8);
            bits.bit(a, true);
        });
        return addNum(byteCnt, bits);
    }

    /**
     *
     * @param byteCnt 字节数
     * @param indexs 如果未查到则视为false
     * @return this
     */
    public BytesBuilder addBits(int byteCnt, Map<Integer, Boolean> indexs) {//2022年10月30日
        Bits bits = Bits.of().byteSize(byteCnt);
        indexs.forEach((k, v)->{
            requireLessThen(k, byteCnt * 8);
            bits.bit(k, v);
        });
        return addNum(byteCnt, bits);
    }

    protected BytesBuilder requireBitsFinished() {
        bitBuilder.requireFinished();
        return this;
    }

    //////////////////////////////////////////////////////////////////
    //杂项

    /**
     * 添加n个字节，字节值与序号相关。方便生成不同的批量测试数据
     *
     * @param times     字节数
     * @param byteMaker 根据位置生成测试字节
     * @return this
     */
    @Unstable
    public BytesBuilder addTest(int times, IntUnaryOperator byteMaker) {
        for (int i = 0; i < times; i++) {
            addByte(byteMaker.applyAsInt(i));
        }
        return this;
    }

    @Unstable
    public BytesBuilder addStatistics(int byteCnt, ToIntFunction<IntStream> statistics) {
        return addNum(byteCnt, statistics.applyAsInt(IntStream.range(0, pos).map(a -> buf()[a] & 0xFF)));
    }

    /**
     * 根据已经输出的字节块计算后面的输出。例如后置的字节数、校验等
     *
     * @param mapper 映射
     * @return this
     */
    public BytesBuilder addBytesRefPrevious(Function<IBytesRange, byte[]> mapper) {
        requireBitsFinished();
        byte[] bytes = mapper.apply(IBytesRange.of(this::buf, 0, pos));
        addBytes(bytes);
        return this;
    }

//过度设计
//    /**
//     * 写入已经创建的字节数
//     *
//     * @param byteCnt 字节数
//     * @return this
//     */
//    @Unstable
//    public BytesBuilder addNumPreviousCount(int byteCnt) {
//        requireBitsFinished();
//        addNum(byteCnt, pos);
//        return this;
//    }

    public OutputStream getOutputStream() {
        return DelegateOutputStream.consumer(this::addByte);
    }

    /**
     * 可以用来查看写入了几个字节
     *
     * @return 从Mark到当前位置的字节数
     */
    public int distanceFromMark() {
        return pos - mark;
    }

    //////////////////////////////////////////////////////////////////
    //trace

    /**
     * 显示最后4字节的HEX。可用于调试时看看最终写入的是什么字节
     *
     * @param consumer hex消费器
     * @return this
     */
    public BytesBuilder peek(Consumer<CharSequence> consumer) {
        return peek(8, consumer);
    }

    public BytesBuilder peek(int byteCnt, Consumer<CharSequence> consumer) {
        if (byteCnt == 0) byteCnt = size();
        String msg = BytesToMessage(buf, Math.max(0, pos - byteCnt), pos);
        consumer.accept(msg);
        return this;
    }

    public BytesBuilder println() {
        System.out.println(toHexMessage());
        return this;
    }

    public BytesBuilder note(CharSequence note) {
        _note(note);
        return this;
    }


    //////////////////////////////////////////////////////////////////
    //仿StringBuilder视图(实验)

    private HexBuilder hexBuilderView = new HexBuilder();

    public HexBuilder asHexBuilder() {
        return hexBuilderView;
    }

    /**
     * 提供仿StringBuilder.append(hex)风格的方法，方便从StringBuilder移植。并提供常用的BytesBuilder方法，更多反方请用asBytesBuilder()恢复所有方法
     */
    @Unstable
    public class HexBuilder implements CharSequence {

        /**
         * addHex(CharSequence hex) 的别名
         *
         * @param hex 不能为空。必须是双个数的‘0’~‘F’字符。
         * @return this
         */
        public HexBuilder append(CharSequence hex) {
            return addHex(hex);
        }

        public HexBuilder addHex(CharSequence hex) {
            BytesBuilder.this.addHex(hex);
            return this;
        }

        public HexBuilder addHex(int byteCnt, CharSequence val) {
            BytesBuilder.this.addHex(byteCnt, val);
            return this;
        }

        public HexBuilder addBcd(int byteCnt, Long val) {
            BytesBuilder.this.addBcd(byteCnt, val);
            return this;
        }

        public HexBuilder addBcd(int byteCnt, Integer val) {
            BytesBuilder.this.addBcd(byteCnt, val);
            return this;
        }

        public HexBuilder addNum(int byteCnt, Long val) {
            BytesBuilder.this.addNum(byteCnt, val);
            return this;
        }

        public HexBuilder addNum(int byteCnt, Integer val) {
            BytesBuilder.this.addNum(byteCnt, val);
            return this;
        }

        public HexBuilder skip(int byteCnt) {
            BytesBuilder.this.skip(byteCnt);
            return this;
        }

        public HexBuilder addString(int byteCnt, String val) {
            BytesBuilder.this.addString(byteCnt, val);
            return this;
        }

        public HexBuilder addString(String val) {
            BytesBuilder.this.addString(val);
            return this;
        }

        public HexBuilder addBits(CharSequence binStr) {
            BytesBuilder.this.addBits(binStr);
            return this;
        }

        public HexBuilder addBytes(byte[] val) {
            BytesBuilder.this.addBytes(val);
            return this;
        }

        public HexBuilder addBytes(ToBytes val) {
            BytesBuilder.this.addBytes(val);
            return this;
        }

        public HexBuilder addBytes(int byteCnt, byte[] val) {
            BytesBuilder.this.addBytes(byteCnt, val);
            return this;
        }

        public HexBuilder addBytes(int byteCnt, CharSequence hex) {
            BytesBuilder.this.addBytes(byteCnt, hex);
            return this;
        }

        public BytesBuilder asBytesBuilder() {
            return BytesBuilder.this;
        }

        public byte[] toBytes() {
            return BytesBuilder.this.toBytes();
        }

        public String toHex() {
            return BytesBuilder.this.toHex();
        }

        @Override
        public int length() {
            return size() * 2;
        }

        @Override
        public char charAt(int index) {
            int ch = buf()[index / 2];
            return HexUtil.hexCharInByte(ch, index % 2);
        }

        @Override
        public CharSequence subSequence(int start, int end) {
            StringBuilder buf = new StringBuilder(end - start);
            for (int i = start; i < end; i++) {
                buf.append(charAt(i));
            }
            return buf.toString();
        }

        public String toString() {
            return BytesBuilder.this.toHex();
        }

    }

    //////////////////////////////////////////////////////////////////
    //result

    /**
     * @return bytes
     */
    @Override
    public byte[] toBytes() {
        requireBitsFinished();
        return Arrays.copyOf(buf(), pos);
    }

    public String toHex() {
        requireBitsFinished();
        return BytesToHex(buf(), 0, pos);
    }

    public IBytesRange toBytesRange() {
        requireBitsFinished();
        return SimpleByteArrayRange.of(this::buf, 0, pos);
    }

    public BytesLoader toBytesLoader() {
        return BytesLoader.of(buf(), 0, pos);
    }

    public int reduce(int identity, IntBinaryOperator op) {
        requireBitsFinished();
        int result = identity;
        for (int i = 0; i < pos; i++) {
            result = op.applyAsInt(result, buf()[i] & 0xFF);
        }
        return result;
    }

    public IntStream toIntStream() {
        return IntStream.range(0, pos).map(a -> buf()[a] & 0xFF);
    }

    public BytesBuilder saveToFile(Path file) {
        requireBitsFinished();
        try {
            Files.write(file, toBytes());
        } catch (IOException e) {
            throw toRuntimeException(e);
        }
        return this;
    }

    public BytesBuilder saveToFile(String filename) {
        return saveToFile(Paths.get(filename));
    }

    public BytesBuilder saveToHexFile(Path file) {
        requireBitsFinished();
        try {
            Files.write(file, Collections.singletonList(toHex()), StandardCharsets.UTF_8);
        } catch (IOException e) {
            throw toRuntimeException(e);
        }
        return this;
    }

    public BytesBuilder saveToHexFile(String filename) {
        return saveToHexFile(Paths.get(filename));
    }

    /**
     * 提供更易于人眼阅读的HEX消息
     *
     * @return this
     */
    public String toHexMessage() {
        int size = size();
        StringBuilder build = new StringBuilder().append("(bytes:").append(size).append(")");
        //build.append(HexUtil.BytesToMessage(toBytes()));
        for (int i = 0; i < size; i++) {
            Object msg = traceMap.get(i);
            if (msg != null) build.append("(").append(msg).append(")");
            build.append(HexUtil.ByteToHex(buf()[i])).append(" ");
        }
        if (bitBuilder.remainBitCnt() > 0) {
            build.append("+bits").append(bitBuilder);
        }
        return build.toString();
    }

    /**
     * 如果业务输出推荐用toHex()。以便在未写完bits及时报出异常。本方法这时会以"+111.."的形式追加。适合作为紧凑HEX日志输出
     *
     * @return 紧凑HEX。如果bit有未写完的字节会用"+111.."形式追加到尾部
     */
    @Override
    public String toString() {
        StringBuilder build = new StringBuilder();
        build.append(HexUtil.BytesToHex(toBytes()));//20221023 build.append(HexUtil.BytesToMessage(toBytes()));
        if (bitBuilder.remainBitCnt() > 0) {
            build.append("+bits").append(bitBuilder);
        }
        return build.toString();
    }

    /**
     * 孵化中
     */
    @Unstable
    public static class Builder {//2022年10月11日
        private static final long serialVersionUID = 1L;
        ByteOrder byteOrder = ByteOrder.BIG_ENDIAN;
        private Charset charset = StandardCharsets.UTF_8;
        public int nullValType = 0;//0空值抛出异常（严格模式，默认）；1空值占位（容错模式，适宜调试场景）；2空值不写（无意义。不推荐。推荐采用字节数值控制）；

        public static Builder of() {
            return new Builder();
        }

        private Builder() {
        }

        public Builder littleEndian() {
            byteOrder = ByteOrder.LITTLE_ENDIAN;
            return this;
        }

        public Builder bigEndian() {
            byteOrder = ByteOrder.BIG_ENDIAN;
            return this;
        }

        public Builder byteOrder(ByteOrder defaultByteOrder) {
            this.byteOrder = defaultByteOrder;
            return this;
        }

        public ByteOrder byteOrder() {
            return byteOrder;
        }

        public Builder charset(Charset charset) {
            this.charset = charset;
            return this;
        }

        public Builder utf8() {
            this.charset = StandardCharsets.UTF_8;
            return this;
        }

        public Builder gbk() {
            this.charset = Charset.forName("GB18030");
            return this;
        }

        public Charset charset() {
            return charset;
        }

        public Builder exceptionIfNullVal() {
            this.nullValType = 0;
            return this;
        }

        public Builder placeHolderIfNullVal() {
            this.nullValType = 1;
            return this;
        }

        public Builder nullValType(int nullValType) {
            this.nullValType = nullValType;
            return this;
        }

        public int nullValType() {
            return nullValType;
        }

        private char stringFilledChar = '\0';

        /**
         * 允许定义定长字符串不足时右侧填充的空白字符。默认是'\0'。
         *
         * @param ch 填充字符。默认是'\0'
         * @return this
         */
        public Builder stringFilledChar(char ch) {
            stringFilledChar = ch;
            return this;
        }

        private boolean stringTailNul = false;

        public Builder stringTailNul() {
            stringTailNul = true;
            return this;
        }

        public BytesBuilder build() {
            return BytesBuilder.of()
                    .byteOrder(byteOrder)
                    .charset(charset)
                    .nullValType(nullValType)
                    .stringFilledChar(stringFilledChar)
                    .stringTailNul(stringTailNul);
        }
    }

}
