package cn.gongler.util.bytes;

import cn.gongler.util.GonglerUtil;

import java.util.List;
import java.util.function.IntConsumer;

import static cn.gongler.util.GonglerUtil.WithDefault;

/**
 * @author honger
 * @date 2022/10/4
 */

class BitsBuilder {
    private static final long serialVersionUID = 1L;

    private final IntConsumer consumer;
    private byte bitBuf = 0;
    private int bitIndex = 7;

    public static BitsBuilder of(IntConsumer consumer) {
        return new BitsBuilder(consumer);
    }


    protected BitsBuilder(IntConsumer consumer) {
        this.consumer = consumer;
    }

    private byte bitBufIsFull() {
        byte ret = bitBuf;
        bitBuf = 0;
        bitIndex = 7;
        return ret;
    }

    public BitsBuilder addBit(Boolean bitVal) {
        bitBuf = BitUtil.setBit(bitBuf, bitIndex--, WithDefault(bitVal, false));
        if (bitIndex < 0) {
            consumer.accept(bitBufIsFull());
        }
        return this;
    }

    public BitsBuilder addBitRepeat(Boolean bitVal, int repeatTimes) {
        for (int i = 0; i < repeatTimes; i++) {
            addBit(bitVal);
        }
        return this;
    }

    /**
     * 字节从高位到低位的顺序添加
     *
     * @param bitVal 1 or 0
     * @return this
     */
    public BitsBuilder addBit(int bitVal) {
        GonglerUtil.require(bitVal == 0 || bitVal == 1, "非法值" + bitVal);
        addBit(bitVal != 0);
        return this;
    }

    public BitsBuilder addBitRepeat(int bitVal, int repeatTimes) {
        for (int i = 0; i < repeatTimes; i++) {
            addBit(bitVal);
        }
        return this;
    }

    public BitsBuilder addBits(int bitCnt, long bitsVal) {
        for (int i = 0; i < bitCnt; i++) {
            addBit(((bitsVal >>> (bitCnt - 1 - i) & 1) == 1));
        }
        return this;
    }

    public BitsBuilder addBits(CharSequence bitsStr) {
        GonglerUtil.require(bitsStr.toString().matches("[01]+"), "非法bit值" + bitsStr);
        for (int i = 0; i < bitsStr.length(); i++) {
            addBit(bitsStr.charAt(i) == '1');
        }
        return this;
    }

    /**
     * bit.get(index) == null equals bit.get(index) == false
     * @param bits
     * @return this
     */
    public BitsBuilder addBits(List<Boolean> bits) {
        bits.forEach(this::addBit);
        return this;
    }

    public int remainBitCnt() {
        return 7 - bitIndex;
    }

    public BitsBuilder requireFinished() {
        GonglerUtil.require(remainBitCnt() == 0, "remain bits not enough to a byte:" + remainBitCnt());
        return this;
    }


    public String toString() {
        StringBuilder buf = new StringBuilder();
        for (int i = 7; i >= bitIndex; i--) {
            buf.append(BitUtil.getBit(bitBuf, i) ? '1' : '0');
        }
        return buf.toString();
    }

}
