package cn.gongler.util.bytes;

import java.lang.Byte;
import java.util.function.IntFunction;
import java.util.stream.IntStream;

import static cn.gongler.util.GonglerUtil.*;

/**
 * 设置或获得一个数的1个二进制或多个二进制位的数值
 * <blockquote><pre>
 *     int val = 123;
 *     val = Bits.of(val).bit(0, 1).bits(1, 3, 5).toInt();
 * </pre></blockquote>
 * <p>
 * 局限：不能超过8个字节
 *
 * @author honger
 * date 2022/10/12
 */

public class Bits extends Number implements Comparable<Bits> {
    private static final long serialVersionUID = 1L;

    public static Bits of() {
        return of(0L);
    }

    public static Bits of(long val) {
        return new Bits(val, 64);
    }

    //过度设计
// public static Bits ByteSize(int byteCnt) {
//        return BitSize(byteCnt * 8);
//    }
//
//    public static Bits BitSize(int bitCnt) {
//        return new Bits(0L, bitCnt);
//    }
//
    public static Bits of(Long bits) {
        return new Bits(WithDefault(bits, 0L), 8);
    }

    public static Bits of(Integer bits) {
        return new Bits(0xFFFFFFFFL & WithDefault(bits, 0), 4);
    }

    public static Bits of(Short bits) {
        return new Bits(0xFFFFL & WithDefault(bits, (short) 0), 2);
    }

    public static Bits of(Byte bits) {
        return new Bits(0xFFL & WithDefault(bits, (byte) 0), 1);
    }

    ////////////////////////////////////////////////////////////////////////////////////////
    // prepare

    protected long bits;
    protected int bitCnt;

    protected Bits(long bits, int bitCnt) {
        requireLessThenOrEqual(bitCnt, 8 * 8);
        this.bits = bits;
        this.bitCnt = bitCnt;
    }


    public Bits byteSize(int byteCnt) {
        requireLessThenOrEqual(byteCnt, 8);
        this.bitCnt = byteCnt * 8;
        return this;
    }

    public Bits bitSize(int bitCnt) {
        requireLessThenOrEqual(bitCnt, 8 * 8);
        this.bitCnt = bitCnt;
        return this;
    }
    ////////////////////////////////////////////////////////////////////////////////////////
    // bit getter and setter

    /**
     * 获取指定的bit中的值
     *
     * @param bitIndex 索引
     * @return bit值
     */
    public int bit(int bitIndex) {
        requireLessThenOrEqual(bitIndex, bitCnt);
        return BitUtil.getBit(bits, bitIndex) ? 1 : 0;
    }

    public Bits bit(int bitIndex, int bitValue) {
        requireLessThenOrEqual(bitIndex, bitCnt);
        bits = BitUtil.setBit(bits, bitIndex, bitValue != 0);
        return this;
    }

    public Bits bit(IntStream trueBitIndex) {//2022年10月30日
        trueBitIndex.forEach(a -> bit(a, true));
        return this;
    }

    public Bits bitLeft1(int bitIndex, boolean bitVal) {//2022年10月30日
        return bit(bitCnt - bitIndex, bitVal);
    }

    public Bits bitLeft1(IntStream trueBitIndexs) {//2022年10月30日
        trueBitIndexs.forEach(a -> bitLeft1(a, true));
        return this;
    }

    /**
     * @param bitIndex 索引
     * @param bitValue null按0计算
     * @return this
     */
    public Bits bit(int bitIndex, Integer bitValue) {
        requireLessThenOrEqual(bitIndex, bitCnt);
        bit(bitIndex, WithDefault(bitValue, 0).intValue());
        return this;
    }

    public boolean bitBool(int bitIndex) {
        return bit(bitIndex) != 0;
    }

    public Bits bit(int bitIndex, Boolean bitValue) {
        return bit(bitIndex, WithDefault(bitValue, false) ? 1 : 0);
    }

    public Bits bit(int bitIndex, CharSequence binString) {
        binString = WithDefault(binString, "0");
        require(binString.toString().matches("[01]?"));
        return bit(bitIndex, "0".equals(binString.toString()) ? 0 : 1);
    }

    public String bitString(int bitIndex) {
        return bit(bitIndex, String::valueOf);
    }


    public <T> T bit(int bitIndex, IntFunction<T> mapper) {
        return mapper.apply(bit(bitIndex));
    }

    ////////////////////////////////////////////////////////////////////////////////////////
    // bits getter and setter


    public int bits(int bitIndex, int bitCnt) {
        requireLessThenOrEqual(bitIndex + bitCnt, this.bitCnt);
        return BitUtil.getBits(bits, bitIndex, bitCnt);
    }

    /**
     * 设置多个bit位存放的数值。例如：把bit2~bit5,存放10.
     *
     * @param bitIndex 多个bit索引中最小的那个。例子中的bit2
     * @param bitCnt   例子中的4。(bit2+bit3+bit4+bit5)
     * @param bitValue 设置的值。本例：10
     * @return this
     */
    public Bits bits(int bitIndex, int bitCnt, int bitValue) {
        requireLessThenOrEqual(bitIndex + bitCnt, this.bitCnt);
        bits = BitUtil.setBits(bits, bitIndex, bitCnt, bitValue);
        return this;
    }

    /**
     * @param bitIndex 位索引
     * @param bitCnt   位数
     * @param bitValue null 按0处理
     * @return this
     */
    public Bits bits(int bitIndex, int bitCnt, Integer bitValue) {
        requireLessThenOrEqual(bitIndex + bitCnt, this.bitCnt);
        bits(bitIndex, bitCnt, WithDefault(bitValue, 0).intValue());
        return this;
    }


    public String bitsString(int bitIndex, int bitCnt) {
        return toString(bitIndex, bitIndex + bitCnt);
    }

//过度设计
//    /**
//     * @param bitIndex  最低位的index
//     * @param binString 二进制的字符串。例如"101011", 不能为null
//     * @return this
//     */
//    public Bits bits(int bitIndex, CharSequence binString) {
//        int bitCnt = binString.length();
//        int bitValue = Integer.valueOf(binString.toString(), 2);
//        return bits(bitIndex, bitCnt, bitValue);
//    }

    public <T> T bits(int bitIndex, int bitCnt, IntFunction<T> mapper) {
        return mapper.apply(bits(bitIndex, bitCnt));
    }

    ////////////////////////////////////////////////////////////////////////////////////////
    // from-to(Exclusive)

    public int fromTo(int lowBitIndex, int highBitIndexExclusive) {
        return bits(lowBitIndex, highBitIndexExclusive - lowBitIndex);
    }

    public Bits fromTo(int lowBitIndex, int highBitIndexExclusive, int value) {
        return bits(lowBitIndex, highBitIndexExclusive - lowBitIndex, value);
    }

    public String fromToString(int lowBitIndex, int highBitIndexExclusive) {
        return toString(lowBitIndex, highBitIndexExclusive);
    }

    public <T> T fromTo(int lowBitIndex, int highBitIndexExclusive, IntFunction<T> mapper) {
        return bits(lowBitIndex, highBitIndexExclusive - lowBitIndex, mapper);
    }

    ////////////////////////////////////////////////////////////////////////////////////////
    // toResult()

    public long toLong() {
        return bits;
    }

    public int toInt() {
        return (int) bits;
    }

    public short toShort() {
        return (short) bits;
    }

    public byte toByte() {
        return (byte) bits;
    }


    ////////////////////////////////////////////////////////////////////////////////////////
    // toString()

    public String toString(int lowBitIndex, int highBitIndexExclusive) {
        requireLessThenOrEqual(highBitIndexExclusive, this.bitCnt);
        StringBuilder buf = new StringBuilder(bitCnt);
        for (int i = highBitIndexExclusive - 1; i >= lowBitIndex; i--) {
            buf.append(bit(i));
        }
        return buf.toString();
    }

    public String toString() {
        return toString(0, bitCnt);
    }

    ////////////////////////////////////////////////////////////////////////////////////////
    // Number interface

    @Override
    public int compareTo(Bits o) {
        return Long.compare(bits, o.bits);
    }

    @Override
    public int intValue() {
        return toInt();//(int)bits;
    }

    @Override
    public long longValue() {
        return toLong();//bits;
    }

    @Override
    public float floatValue() {
        return bits;//throw new UnsupportedOperationException();//
    }

    @Override
    public double doubleValue() {
        return bits;//throw new UnsupportedOperationException();//
    }

//    public static BytesLoader.Loadable<Bits> loader = new BytesLoader.Loadable<>(){
//        @Override
//        public Object load(BytesLoader bytesReader) {
//            return Bits.of(bytesReader.loadLong());
//        }
//    };

}
