package cn.gongler.util;

import cn.gongler.util.tuple.Tuple;
import cn.gongler.util.tuple.Tuple2;

import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.stream.Collectors;

/**
 * 工具类：保留最近的n条记录的容器。支持多线程的无锁工具类。效率优先。 代价：可能会读到错误的
 *
 * @param <E> E
 * @author gongler
 */
public class Recently<E> extends AbstractMap<LocalDateTime, E> {// implements Iterable<Recently.MyEntry<E>>{

    private static final long serialVersionUID = -3195148595268151000L;//Recently

    public static final Tuple2<AtomicInteger, AtomicLong> INSTANCE_CNT = Tuple.of(new AtomicInteger(), new AtomicLong());//gongler20181023add

    private static final Recently EMPTY = new Recently(1);//gongleradd20170401

    public static <T> Recently<T> empty() {//20170401
        return (Recently<T>) EMPTY;
    }

    private final Entry<Long, E>[] buf;//gongler20181030 private final Entry<LocalDateTime, E>[] buf;
    private final AtomicLong writePos = new AtomicLong(0L);//gongler20181009 AtomicInteger升级为AtomicLong //20131227gongler允许多线程使用 volatile int pos = 0;
    private final int capacity;
    private final int capacityAlign;
    private final int mask;//不适合容量1

    private final BiFunction<Long, E, Map.Entry<Long, E>> map;//gongler20181030内存压缩 //private final BiFunction<LocalDateTime, E, Map.Entry<LocalDateTime, E>> map;

    public Recently() {
        this(16);
    }

    /**
     * 使用2的幂作为容量，将更有效利用内存空间。
     *
     * @param capacity 容量
     */
    public Recently(int capacity) {
        this(capacity, null);
    }

    private static class RecentlyEntry<LocalDateTime, E> extends SimpleImmutableEntry<LocalDateTime, E> {//gongler20181025

        public RecentlyEntry(LocalDateTime key, E value) {
            super(key, value);
        }
    }

    /**
     * 探索：为了方便内存分析，允许标记
     *
     * @param capacity 容量
     * @param map      mapper
     * @since 2018.10.25
     */
    public Recently(int capacity, BiFunction<Long, E, Map.Entry<Long, E>> map) {//gongler20181025 实验性质，不保证永久支持。
        this.capacity = capacity;
        this.capacityAlign = AlignCapacity(capacity);
        this.mask = capacityAlign - 1;
        buf = new Entry[capacityAlign];
        INSTANCE_CNT.e1().incrementAndGet();//gongler20181023add
        INSTANCE_CNT.e2().addAndGet(capacity);//gongler20181023add
        this.map = map == null ? RecentlyEntry::new : map;
    }

    /**
     * @param e 不允许插入null
     * @return e
     */
    public E push(E e) {
        Objects.requireNonNull(e);//gongler20181009add
        buf[index(writePos.getAndIncrement())] = map.apply(System.currentTimeMillis(), e);//gongler20181025 new SimpleImmutableEntry<>(LocalDateTime.now(), e);
        return e;
    }

    private int index(long serial) {
        return ((int) serial) & mask;//gongler20181009形参改为long，由于mask为正，不会产生负值索引 //gongler20171220 return mask == 0 ? 0 : serial & mask;
    }

    private Entry<Long, E> get(long pos) {
        return buf[index(pos)];
    }

    public void forEach(long limit, BiConsumer<LocalDateTime, E> action) {//20160913add
        GonglerUtil.MapForEachLimit(this, limit, action);
    }

    public long pushTimes() {//gongler20181009规划
        return writePos.longValue();
    }

    public Entry<LocalDateTime, E>[] toArray() {//gongler20181009
        return this.entrySet().toArray(new Entry[0]);
    }

    @Override
    public String toString() {
        return toString(10L);//ElementsToString(this, 10);
    }

    public String toString(long limit) {
        return this.values().stream().limit(limit).map(String::valueOf).collect(Collectors.joining(", "));//ElementsToString(this, 10);
    }

    /**
     * 返回能容纳参数容量的最小2的整数幂。1=>1, 2=>2, 3,4=>4, 5~8=>8 8~16=>16 17~24=>24
     *
     * @param anyCapacity 原始值
     * @return 对齐后值
     */
    private static int AlignCapacity(int anyCapacity) {
        if (anyCapacity <= 1) {
            return 1;
        } else {
            return 1 << (HighBitIndex(anyCapacity - 1) + 1);
        }
    }

    private static int HighBitIndex(int val) {
        for (int i = 0; i < 32; i++) {
            final int moveBitCnt = 31 - i;
            if (((val >>> moveBitCnt) & 1) == 1) {
                return moveBitCnt;
            }
        }
        return 1;
    }

    @Override
    public Set<Entry<LocalDateTime, E>> entrySet() {
        return new AbstractSet<Entry<LocalDateTime, E>>() {

            @Override
            public Iterator<Entry<LocalDateTime, E>> iterator() {
                return new Iterator<Entry<LocalDateTime, E>>() {
                    final long firstReadPos = writePos.get() - 1L;
                    final Entry<Long, E> firstReadItem = get(firstReadPos);
                    long iteratorNextSerial = firstReadPos;

                    @Override
                    public boolean hasNext() {
                        long next = iteratorNextSerial;
                        Entry<Long, E> nextItem = get(next);
                        return next > firstReadPos - capacity
                                && nextItem != null
                                && nextItem.getKey() <= firstReadItem.getKey();//gongler20181030 && !nextItem.getKey().isAfter(firstReadItem.getKey());//如果被覆盖时，不再输出。
                    }

                    @Override
                    public Entry<LocalDateTime, E> next() {
                        //gongler20181030 return get(iteratorNextSerial--);
                        Entry<Long, E> entry = get(iteratorNextSerial--);
                        return new SimpleImmutableEntry(GonglerUtil.toLocalDateTime(entry.getKey()), entry.getValue());
                    }

                };
            }

            @Override
            public int size() {
                return buf.length;//(int)this.stream().count();
            }
        };
    }

//    public static void main(String[] args) {
//        Recently<Integer> r = new Recently<>(3);
//        r.push(1);
//        r.push(2);
//        r.push(3);
//        r.push(4);
//        r.push(4);
//        r.push(5);
//        System.out.println("a"+r);
//        System.out.println("a"+r.size());
//        r.forEach((t, v)->System.out.println("b"+t+", "+v));
//        
//    }

}
/* 
2018.10.30 为了节约内存，内部时间用long替代LocalDateTime
2018.09.13 新增方法：Recently.toArray()
2018.09.13 Recently增加pushTimes()方法；删除MyEntry类
2016.09.13 add forEach(long limit, BiConsumer<LocalDateTime, E> action)
2015.04.27 实现Map接口
 * 2013.12.27 改用AtomicInteger
 * 2013.05.03 gongler created 为了页面日志输出而建的工具类（由于ValHis效率无法达到该需求，而新建本类）
 */
