/*
 * create by gongler before 2014.07.
 *
 *
 */
package cn.gongler.util;

import cn.gongler.util.function.ExceptionConsumer;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedTransferQueue;
import java.util.concurrent.TransferQueue;
import java.util.function.Consumer;

/**
 * <pre>
 * QueueConsumer consumer = QueueConsumer.of("QueueConsumer");
 * consumer.accept(()-&gt;System.out.println("test1");
 * </pre>
 *
 * @author gongler
 * @since 2016-08-05
 */
public class QueueConsumer implements Consumer<ITask>, AutoCloseable {

    private static final long serialVersionUID = 4800910141074860566L;//QueueAutoConsumer @since 2016-08-05

    public static QueueConsumer of(String threadName) {
        return of(threadName, 10_0000);
    }

    @Deprecated
    public static QueueConsumer of(String threadName, BlockingQueue<ITask> queue) {
        return new QueueConsumer(threadName, queue);
    }

    public static QueueConsumer of(String threadName, int queueCapacity) {
        return of(threadName, new ArrayBlockingQueue<>(queueCapacity));
    }

    public static QueueConsumer of(String threadName, int queueCapacity, int threadCount) {
        return new QueueConsumer(threadName, queueCapacity, threadCount, 10);
    }

    private final String threadName;
    private int threadNum;

    private final BlockingQueue<ITask> taskQueue;
    private final List<Thread> backThreads = new ArrayList<>();
    private boolean cancel = false;

    private QueueConsumer(String threadName, BlockingQueue<ITask> queue) {
        this.threadName = threadName;
        this.taskQueue = queue;
        Thread backThread = GonglerUtil.StartDaemonThread(threadName, () -> {
            try {
                while (cancel == false) {
                    taskQueue.take().executeWithCatchAny();
                }
            } catch (InterruptedException ex) {
            } finally {
                System.out.println("Exception: Thread:" + threadName + " is exited.");
            }
        });
        backThreads.add(backThread);
    }

    private QueueConsumer(String threadName, int queueCapacity, int threadNum, int itemCountMax) {
        this.threadName = threadName;
        final TransferQueue<ITask> queue = new LinkedTransferQueue<>();//注意是无界队列，警惕内存溢出。
        MinuteTimer.of().add(() -> {
            if (queue.size() > queueCapacity) {
                System.out.println("WARN:" + LocalDateTime.now() + ", " + threadName + " clearQueue:" + queue.size());
                queue.clear();
            }
        });
        this.taskQueue = queue;
        this.threadNum = threadNum;
        //int itemCountMax = 100;
        for (int i = 0; i < threadNum; i++) {
            String name = threadName + i;
            Thread backThread = GonglerUtil.StartDaemonThread(name, () -> {
                try {
                    List<ITask> tasks = new ArrayList<>();
                    while (cancel == false) {
                        int count = queue.drainTo(tasks, itemCountMax);
                        //System.out.println("load:" + count);
                        if (count == 0) {
                            taskQueue.take().executeWithCatchAny();
                        } else {
                            for (ITask task : tasks) {
                                task.executeWithCatchAny();
                            }
                            tasks.clear();
                        }
                    }
                } catch (InterruptedException ex) {
                } finally {
                    System.out.println("Exception: Thread:" + name + " is exited.");
                }
            });
            backThreads.add(backThread);
        }
    }

    private long taskCount = 0L;//gongler20171031 极速不精确计数。

    @Override
    public void accept(ITask task) {
        taskCount++;
        taskQueue.offer(task);
    }

    public long acceptTaskCount() {
        return this.taskCount;
    }

    public int size() {
        return taskQueue.size();
    }

    /**
     * 如果不习惯于lambda，那么可以用本方法转换成同类型队列的视图形式使用。
     * <pre>
     * Consumer&lt;String&gt; consumer = QueueConsumer.of("QueueConsumer").toView(data -&gt; System.out.println(data));
     * consumer.accept("test1");
     * </pre>
     *
     * @param <D>         D
     * @param dataHandler handler
     * @return Consumer
     */
    public <D> Consumer<D> toView(ExceptionConsumer<D> dataHandler) {
        return data -> accept((() -> dataHandler.accept(data)));
    }

    /**
     * 提供原始队列，方便可以直接使用更多的方法。
     *
     * @return queue
     */
    public Collection<ITask> getQueue() {
        return Collections.unmodifiableCollection(taskQueue);
    }

    @Override
    public void close() {
        cancel = true;
        backThreads.stream().forEach(Thread::interrupt);
    }

    @Override
    public String toString() {
        return threadName + "_Thread:" + threadNum + "_queueSize:" + size();
    }

//    public static void main(String[] args) throws InterruptedException {
////        Consumer<String> consumer = QueueConsumer.of("abc").toView(data -> System.out.println(data));
////        consumer.accept("abc");
////        consumer.accept("abc1");
////        consumer.accept("abc2");
//        QueueConsumer q;
//        Consumer<String> consumer2 = (q = QueueConsumer.of("QueueConsumer", 100, 3)).toView(a -> {
//            Thread.sleep(1000);
//            System.out.println(LocalDateTime.now() + ", " + Thread.currentThread() + ", " + a);
//        });
//        for (int i = 0; i < 100; i++) {
//            consumer2.accept(",i=" + i);
//        }
//        Thread.sleep(1000L * 30L);
//
//        q.close();
//    }
}

/**
 * 2017.10.28 14:57 增加Builder，增加多线程
 */
