/*
 *
 *
 *
 */
package cn.gongler.util;

import java.util.*;
import java.util.function.Function;

import static cn.gongler.util.GonglerUtil.WithDefault;

/**
 * 分段式线路名比较器
 * 优先比较线路数字，然后再比较前后缀,不含数字则排最后: [1路, 快1路, 2, 2路, 2路1支线, 11路, 专线]
 * 补充知识：中文数字“一二三四五六七八九”在任何编码中都是乱序。"三分公司包车队", "四分公司包车队"无法通过字符排序解决，需要特殊处理。
 * 算法：XX路作为基准点，前后缀文字按按数字与否进行文字串切割，切割下来的文字段分数和字符串2种类型比较排序。
 * 依赖guava18.jar
 * <pre>
 * String msg = "1路,1环1路,1环2路,2路,2环1路,2环2路,3路,4路,5路,6路,7路,8路,9路,10路,11路,12路,13路,14路,15路,16路,17路,18路,19路,20路,21路,22路,23路,24路,25路,26路,27路,28路,29路,30路,32路,33路,34路,35路,36路,37路,38路,39路,40路,41路,42路,43路,45路,46路,48路,49路,50路,51路,52路,53路,55路,56路,57路,58路,59路,60路,61路,62路,63路,65路,66路,67路,68路,69路,70路,71路,72路,73路,75路,76路,77路,78路,79路,80路,81路,82路,84路,85路,86路,87路,88路,89路,90路,91路,92路,93路,96路,98路,99路,101路,102路,103路,106路,107路,108路,110路,111路,112路,113路,115路,116路,117路,118路,130路,131路,132路,133路,134路,135路,136路,138路,139路,142路,143路,145路,147路,148路,164路,171路,177路,192路,201路,202路,203路,204路,205路,206路,207路,210路,211路,212路,215路,216路,281路,290路,301路,302路,303路,304路,304班线,305路,306路,307路,308路,309路,311路,312路,313路,314路,315路,317路,318路,319路,320路,321路,322路,323路,325路,326路,327路,328路,329路,331路,333路,337路,344路,347路,357路,368路,501路,502路,503路,504路,505路,506路,507路,508路,511路,512路,513路,515路,516路,518路,519路,520路,522路,523路,525路,526路,527路,530路,536路,537路,553路,555路,572路,580路,微环1路,微环2路,微环3路,快1路,快2环1路,快2环2路,快32路,快35路,快107路,快516路,游1,游2,游3,游5,游6,游7,游11,游12,游13,游16,藁城环1路,藁城环2路,观光1路,观光2路,观光3路,旅游公司";
 * List<String > l = new ArrayList<>(Arrays.asList(msg.split(",")));
 * List<String> l2 = Arrays.asList("三分公司包车队", "四分公司包车队", "六分公司包车队", "九分公司包车队", "二分公司包车队", "一分公司包车队", "五分公司包车队", "定时班车", "304班线", "9路", "9路5支线", "8路5支线", "9路火车站线", "9路12支线", "9路62支线", "旅游公司", "藁城环11路", "藁城环2路", "观光11路", "观光2路", "游11", "游3", "快2环1路", "快1路", "微环3路", "131路", "1环2路", "");
 * l.addAll(l2);
 * Collections.sort(l);
 * //1. 如果不带数字，只有文字。则放在最后，按字符串比较。CCCCCCCCCCCCCCC
 * //2. 如果不含“路”且最后一个是数字。则补“路”，例如："游11"， 等价于"游11路"。
 * //3. 如果含有“路”，则切割成 前缀和后缀。前缀只含数字优先。例如：“12路”的“12”。 文字+数字+文字+“路”+字符串（兼容1位数）。
 * for (String name : l) {
 * System.out.println(name + ", " + new LineParser(name));
 * }
 * Collections.sort(l, Util.LineComparator());//ChineseLinenameComparator.of());
 * System.out.println("===========");
 * for (String name : l) {
 * System.out.println(name + ", " + new LineParser(name));
 * }
 * System.out.println(Util.BytesToHex("二九六三四五一".getBytes("GBK")));
 * System.out.println(Util.BytesToHex("一二三四五六七八九".getBytes("GBK")));
 * System.out.println(Util.BytesToHex("二九六三四五一".getBytes("UTF-8")));
 * System.out.println(Util.BytesToHex("一二三四五六七八九".getBytes("UTF-8")));
 * System.out.println("" + new String(Util.HexToBytes("D2BBB6FEC8FDCBC4CEE5C1F9C6DFB0CBBEC5"), "GBK"));
 * </pre>
 *
 * @param <T>
 * @author gongler
 * @since 2017.10.27
 */

class ChineseLinenameComparator<T> implements Comparator<T> {

    private static final long serialVersionUID = -1811684678923180894L;//2017-11-01 ChineseLinenameComparator

    //* 2017.10.26 Util.LineComparator() 如果没有传入比较器，则只允许比较字符串。规避误用。 private static final LinenameComparator<Object> DEFAULT_INSTANCE = new LinenameComparator<>(String::valueOf);
    private static final ChineseLinenameComparator<String> DEFAULT_INSTANCE = new ChineseLinenameComparator<>(Function.identity());

    /**
     * 如果你的线路类的toString就是线路名称，则可以直接用该工厂方法。将返回一个单例比较器。
     *
     * @return
     */
    public static ChineseLinenameComparator of() {
        return DEFAULT_INSTANCE;
    }

    /**
     * 如果你的线路类A含有一个返回线路名的方法例如method，则可以用该工厂方法获得比较器：LinenameComparator.of(LineParser::method);
     *
     * @param <T>        你的线路类
     * @param toLinename
     * @return
     */
    public static <T> ChineseLinenameComparator<T> of(Function<T, String> toLinename) {
        return new ChineseLinenameComparator<>(toLinename);
    }

    private final Function<T, String> toLinename;

    private ChineseLinenameComparator(Function<T, String> toLinename) {
        this.toLinename = a -> a == null ? "" : WithDefault(toLinename.apply(a), "");//线路名增加null容错处理，等同于“”
    }

    @Override
    public int compare(T o1, T o2) {
        LineParser t1 = new LineParser(toLinename.apply(o1));
        LineParser t2 = new LineParser(toLinename.apply(o2));
        ComparisonChain c = ComparisonChain.start()
                .compareTrueFirst(t1.hasDigit(), t2.hasDigit())//让全文字线路名“旅游公司”排在最后
                .compareTrueFirst(t1.leftIndex() < 0, t2.leftIndex() < 0);//“304班线”, 排在线路后面，排在全中文“旅游公司”之前
        for (int i = Math.min(t1.leftIndex(), t2.leftIndex()); i <= Math.max(t1.rightIndex(), t2.rightIndex()); i++) {
            c = c.compare(t1.get(i), t2.get(i));
        }
        return c.result();
    }

    private static class LineParser {

        private static final long serialVersionUID = 1L;

        static class Item implements Comparable<Item> {

            private static final long serialVersionUID = 1L;

            static Item NULL = new Item("0");

            Item(String str) {
                try {
                    this.intVal = Integer.parseInt(str, 10);
                } catch (Exception e) {
                    this.msgVal = str;
                }
            }

            int intVal = Integer.MAX_VALUE;
            String msgVal = "";

            @Override
            public int compareTo(Item o) {
                return ComparisonChain.start().compare(this.intVal, o.intVal).compare(this.msgVal, o.msgVal).result();
            }

            @Override
            public String toString() {
                return "(" + intVal + "," + msgVal + ")";
            }
        }

        private final List<Item> items = new LinkedList<>();//int or string 
        private int indexOfLu = -1;
        private final String originalLineName;
        private final Map<Integer, Item> map = new HashMap<>();

        /**
         *
         * @param original MUST NOT null
         */
        LineParser(String original) {
            this.originalLineName = original;
            boolean isDigit = true;
            StringBuilder itemBuf = new StringBuilder();
            for (char ch : original.toCharArray()) {
                if (itemBuf.length() == 0) {
                    itemBuf.append(changeChineseDigit(ch));
                    isDigit = Character.isDigit(ch);
                } else {
                    if (Character.isDigit(ch) != isDigit) {
                        items.add(new Item(itemBuf.toString()));
                        itemBuf.setLength(0);
                    } else {
                        if (itemBuf.toString().endsWith("路")) {
                            items.add(new Item(itemBuf.toString()));
                            itemBuf.setLength(0);
                        }
                    }
                    itemBuf.append(changeChineseDigit(ch));
                    isDigit = Character.isDigit(ch);
                    if ('路' == ch) {
                        indexOfLu = items.size();
                    }
                }
            }
            if (itemBuf.length() > 0) {
                items.add(new Item(itemBuf.toString()));
            }
            if (isDigit && indexOfLu == -1) {//如果没出现“路”且以数字结尾，则假定省略了“路”。例如：“游1”等价于“游1路”
                items.add(new Item("路"));
                indexOfLu = items.size() - 1;
            }

            for (int i = 0; i < items.size(); i++) {
                map.put(i - indexOfLu, items.get(i));
            }

        }

        Item get(int i) {
            return map.getOrDefault(i, Item.NULL);
        }

        int leftIndex() {
            return -indexOfLu;
        }

        int rightIndex() {
            return items.size() - 1 - indexOfLu;
        }

        boolean hasDigit() {
            for (char ch : originalLineName.toCharArray()) {
                if (Character.isDigit(ch)) {
                    return true;
                }
            }
            return false;
        }

        int leftLength() {
            if (leftIndex() < 0) {
                return -leftIndex();
            } else {
                return 0;
            }
        }

        int getLineNumber() {
            return (get(-1) == null) ? Integer.MAX_VALUE : get(-1).intVal;
        }

        private static char changeChineseDigit(char ch) {
            char ret;
            switch (ch) {
                case '一':
                    ret = '1';
                    break;
                case '二':
                    ret = '2';
                    break;
                case '三':
                    ret = '3';
                    break;
                case '四':
                    ret = '4';
                    break;
                case '五':
                    ret = '5';
                    break;
                case '六':
                    ret = '6';
                    break;
                case '七':
                    ret = '7';
                    break;
                case '八':
                    ret = '8';
                    break;
                case '九':
                    ret = '9';
                    break;
                case '零':
                    ret = '0';
                case '十':
                    ret = 'A';
                case '百':
                    ret = 'B';
                case '千':
                    ret = 'C';
                    break;
                default:
                    ret = ch;
            }
            return ret;
        }

        @Override
        public String toString() {
            StringBuilder buf = new StringBuilder();
            for (int i = 0; i < items.size(); i++) {
                buf.append("[").append(i - indexOfLu).append("]").append(items.get(i));
            }
            buf.append(" hasDigit:").append(this.hasDigit());
            buf.append(" LineNumber:").append(this.getLineNumber());
            buf.append(" leftLength():").append(this.leftLength());
            return buf.append(", ").append(leftIndex()).append("~").append(rightIndex()).toString();
        }
    }

    private static abstract class ComparisonChain {

        private static final long serialVersionUID = 1L;

        private ComparisonChain() {
        }

        public static ComparisonChain start() {
            return ACTIVE;
        }

        private static final ComparisonChain ACTIVE =
                new ComparisonChain() {
                    @SuppressWarnings("unchecked") // unsafe; see discussion on supertype
                    @Override
                    public ComparisonChain compare(Comparable<?> left, Comparable<?> right) {
                        return classify(((Comparable<Object>) left).compareTo(right));
                    }

                    @Override
                    public <T extends Object> ComparisonChain compare(
                            T left, T right, Comparator<T> comparator) {
                        return classify(comparator.compare(left, right));
                    }

                    @Override
                    public ComparisonChain compare(int left, int right) {
                        return classify(Integer.compare(left, right));
                    }

                    @Override
                    public ComparisonChain compare(long left, long right) {
                        return classify(Long.compare(left, right));
                    }

                    @Override
                    public ComparisonChain compare(float left, float right) {
                        return classify(Float.compare(left, right));
                    }

                    @Override
                    public ComparisonChain compare(double left, double right) {
                        return classify(Double.compare(left, right));
                    }

                    @Override
                    public ComparisonChain compareTrueFirst(boolean left, boolean right) {
                        return classify(Boolean.compare(right, left)); // reversed
                    }

                    @Override
                    public ComparisonChain compareFalseFirst(boolean left, boolean right) {
                        return classify(Boolean.compare(left, right));
                    }

                    private ComparisonChain classify(int result) {
                        return (result < 0) ? LESS : (result > 0) ? GREATER : ACTIVE;
                    }

                    @Override
                    public int result() {
                        return 0;
                    }
                };

        private static final ComparisonChain LESS = new ComparisonChain.InactiveComparisonChain(-1);

        private static final ComparisonChain GREATER = new ComparisonChain.InactiveComparisonChain(1);

        private static final class InactiveComparisonChain extends ComparisonChain {
            final int result;

            InactiveComparisonChain(int result) {
                this.result = result;
            }

            @Override
            public ComparisonChain compare(Comparable<?> left, Comparable<?> right) {
                return this;
            }

            @Override
            public <T extends Object> ComparisonChain compare(
                    T left, T right, Comparator<T> comparator) {
                return this;
            }

            @Override
            public ComparisonChain compare(int left, int right) {
                return this;
            }

            @Override
            public ComparisonChain compare(long left, long right) {
                return this;
            }

            @Override
            public ComparisonChain compare(float left, float right) {
                return this;
            }

            @Override
            public ComparisonChain compare(double left, double right) {
                return this;
            }

            @Override
            public ComparisonChain compareTrueFirst(boolean left, boolean right) {
                return this;
            }

            @Override
            public ComparisonChain compareFalseFirst(boolean left, boolean right) {
                return this;
            }

            @Override
            public int result() {
                return result;
            }
        }

        public abstract ComparisonChain compare(Comparable<?> left, Comparable<?> right);

        /**
         * Compares two objects using a comparator, <i>if</i> the result of this comparison chain has not
         * already been determined.
         */
        public abstract <T extends Object> ComparisonChain compare(T left, T right, Comparator<T> comparator);

        public abstract ComparisonChain compare(int left, int right);

        public abstract ComparisonChain compare(long left, long right);

        /**
         * Compares two {@code float} values as specified by {@link Float#compare}, <i>if</i> the result
         * of this comparison chain has not already been determined.
         */
        public abstract ComparisonChain compare(float left, float right);

        /**
         * Compares two {@code double} values as specified by {@link Double#compare}, <i>if</i> the result
         * of this comparison chain has not already been determined.
         */
        public abstract ComparisonChain compare(double left, double right);

        public abstract ComparisonChain compareTrueFirst(boolean left, boolean right);

        public abstract ComparisonChain compareFalseFirst(boolean left, boolean right);

        public abstract int result();
    }


//    public static void main(String[] args) throws UnsupportedEncodingException {
//        String msg = "1路,1环1路,1环2路,2路,2环1路,2环2路,3路,4路,5路,6路,7路,8路,9路,10路,11路,12路,13路,14路,15路,16路,17路,18路,19路,20路,21路,22路,23路,24路,25路,26路,27路,28路,29路,30路,32路,33路,34路,35路,36路,37路,38路,39路,40路,41路,42路,43路,45路,46路,48路,49路,50路,51路,52路,53路,55路,56路,57路,58路,59路,60路,61路,62路,63路,65路,66路,67路,68路,69路,70路,71路,72路,73路,75路,76路,77路,78路,79路,80路,81路,82路,84路,85路,86路,87路,88路,89路,90路,91路,92路,93路,96路,98路,99路,101路,102路,103路,106路,107路,108路,110路,111路,112路,113路,115路,116路,117路,118路,130路,131路,132路,133路,134路,135路,136路,138路,139路,142路,143路,145路,147路,148路,164路,171路,177路,192路,201路,202路,203路,204路,205路,206路,207路,210路,211路,212路,215路,216路,281路,290路,301路,302路,303路,304路,304班线,305路,306路,307路,308路,309路,311路,312路,313路,314路,315路,317路,318路,319路,320路,321路,322路,323路,325路,326路,327路,328路,329路,331路,333路,337路,344路,347路,357路,368路,501路,502路,503路,504路,505路,506路,507路,508路,511路,512路,513路,515路,516路,518路,519路,520路,522路,523路,525路,526路,527路,530路,536路,537路,553路,555路,572路,580路,微环1路,微环2路,微环3路,快1路,快2环1路,快2环2路,快32路,快35路,快107路,快516路,游1,游2,游3,游5,游6,游7,游11,游12,游13,游16,藁城环1路,藁城环2路,观光1路,观光2路,观光3路,旅游公司";
//        List<String > l = new ArrayList<>(Arrays.asList(msg.split(",")));
//        List<String> l2 = Arrays.asList("三分公司包车队", "四分公司包车队", "六分公司包车队", "九分公司包车队", "二分公司包车队", "一分公司包车队", "五分公司包车队", "定时班车", "304班线", "9路", "9路5支线", "8路5支线", "9路火车站线", "9路12支线", "9路62支线", "旅游公司", "藁城环11路", "藁城环2路", "观光11路", "观光2路", "游11", "游3", "快2环1路", "快1路", "微环3路", "131路", "1环2路", "");
//        l.addAll(l2);
//        Collections.sort(l);
//        /*
//        1. 如果不带数字，只有文字。则放在最后，按字符串比较。CCCCCCCCCCCCCCC
//        2. 如果不含“路”且最后一个是数字。则补“路”，例如："游11"， 等价于"游11路"。
//        3. 如果含有“路”，则切割成 前缀和后缀。前缀只含数字优先。例如：“12路”的“12”。 文字+数字+文字+“路”+字符串（兼容1位数）。
//         */
//        for (String name : l) {
//            System.out.println(name + ", " + new LineParser(name));
//        }
//        Collections.sort(l, Util.LineComparator());//ChineseLinenameComparator.of());
//        System.out.println("===========");
//        for (String name : l) {
//            System.out.println(name + ", " + new LineParser(name));
//        }
//        System.out.println(Util.BytesToHex("二九六三四五一".getBytes("GBK")));
//        System.out.println(Util.BytesToHex("一二三四五六七八九".getBytes("GBK")));
//        System.out.println(Util.BytesToHex("二九六三四五一".getBytes("UTF-8")));
//        System.out.println(Util.BytesToHex("一二三四五六七八九".getBytes("UTF-8")));
//        System.out.println("" + new String(Util.HexToBytes("D2BBB6FEC8FDCBC4CEE5C1F9C6DFB0CBBEC5"), "GBK"));
//    }
}
