package cn.gongler.util.text;

import cn.gongler.util.GonglerUtil;

import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.function.Function;

import static cn.gongler.util.GonglerUtil.WithDefault;

/**
 * String.split()增强。无需再判断是否达到边界；不用考虑类型转换；提供默认值
 *
 * @author honger
 * date 2022/10/16
 */

public class StringSpliter {
    private final String msg;
    private final String delimiter;
    private final int limit;
    private final String[] items;
    private int pos = 0;

    /**
     * 默认逗号分隔
     *
     * @param msg 字符串
     * @return StringSplitter
     */
    public static StringSpliter of(String msg) {
        return of(msg, ",");
    }

    public static StringSpliter of(String msg, String delimiter) {
        return of(msg, delimiter, 0);
    }

    public static StringSpliter of(String msg, String delimiter, int limit) {
        return new StringSpliter(msg, delimiter, limit);
    }

    /**
     * @param msg       允许null
     * @param delimiter 分隔符
     * @param limit     分隔个数
     */
    public StringSpliter(String msg, String delimiter, int limit) {
        this.msg = WithDefault(msg, "");
        this.delimiter = delimiter;
        this.limit = limit;
        this.items = this.msg.split(delimiter, limit);
    }

    private String next() {
        String ret = get(pos);
        pos++;
        return ret;
    }

    private String get(int index) {
        if (index >= items.length) return null;
        String item = items[index];
        if (item == null || "".equals(item)) return null;
        return item;
    }

    public int nextInt() {
        return nextInt(0);
    }

    public int nextInt(int defaultVal) {
        String next = next();
        if (next == null) return defaultVal;
        return GonglerUtil.DecodeInt(next.trim());
    }

    public Integer nextInteger() {
        return nextInteger(null);
    }

    public Integer nextInteger(Integer defaultVal) {
        String next = next();
        if (next == null) return defaultVal;
        return GonglerUtil.DecodeInt(next.trim());
    }

    public Long nextLong() {
        return nextLong(null);
    }

    public Long nextLong(Long defaultVal) {
        String next = next();
        if (next == null) return defaultVal;
        return GonglerUtil.DecodeLong(next.trim());
    }

    public Double nextDouble() {
        return nextDouble(null);
    }

    public Double nextDouble(Double defaultVal) {
        String next = next();
        if (next == null) return defaultVal;
        return Double.parseDouble(next.trim());
    }
    public Double getDouble(int index) {
        return getDouble(index, null);
    }

    public Double getDouble(int index, Double defaultVal) {
        String next = get(index);
        if (next == null) return defaultVal;
        return Double.parseDouble(next.trim());
    }

    public String nextString() {
        return nextString(null);
    }

    public String nextString(String defaultVal) {
        String next = next();
        if (next == null) return defaultVal;
        return next;
    }

    public <T> T nextObject(Function<String, T> decoder) {
        return nextObject(decoder, null);
    }

    public <T> T nextObject(Function<String, T> decoder, T defaultVal) {
        String next = next();
        if (next == null) return defaultVal;
        return decoder.apply(next);
    }

    public int getInt(int index) {
        return getInt(index, 0);
    }

    public int getInt(int index, int defaultVal) {
        String next = get(index);
        if (next == null) return defaultVal;
        return GonglerUtil.DecodeInt(next.trim());
    }

    public Integer getInteger(int index) {
        return getInteger(index, null);
    }

    public Integer getInteger(int index, Integer defaultVal) {
        String next = get(index);
        if (next == null) return defaultVal;
        return GonglerUtil.DecodeInt(next.trim());
    }

    public Long getLong(int index) {
        return getLong(index, null);
    }

    public Long getLong(int index, Long defaultVal) {
        String next = get(index);
        if (next == null) return defaultVal;
        return GonglerUtil.DecodeLong(next.trim());
    }

    public String getString(int index) {
        return getString(index, null);
    }

    public String getString(int index, String defaultVal) {
        String next = get(index);
        if (next == null) return defaultVal;
        return next;
    }

    public Boolean nextBoolean() {
        return nextBoolean(null);
    }

    public Boolean nextBoolean(Boolean defaultVal) {
        String next = next();
        if (next == null) return defaultVal;
        return Boolean.parseBoolean(next.trim());
    }

    public Boolean getBoolean(int index) {
        return getBoolean(index, null);
    }

    public Boolean getBoolean(int index, Boolean defaultVal) {
        String next = get(index);
        if (next == null) return defaultVal;
        return Boolean.parseBoolean(next.trim());
    }


    public Duration nextDuration() {
        return nextDuration(null);
    }

    public Duration nextDuration(Duration defaultVal) {
        String next = next();
        if (next == null) return defaultVal;
        return Duration.parse(next.trim());
    }

    public Duration getDuration(int index) {
        return getDuration(index, null);
    }

    public Duration getDuration(int index, Duration defaultVal) {
        String next = get(index);
        if (next == null) return defaultVal;
        return Duration.parse(next.trim());
    }


    public LocalDateTime nextLocalDateTime() {
        return nextLocalDateTime(null);
    }

    public LocalDateTime nextLocalDateTime(LocalDateTime defaultVal) {
        String next = next();
        if (next == null) return defaultVal;
        return LocalDateTime.parse(next.trim());

    }

    public LocalDateTime getLocalDateTime(int index) {
        return getLocalDateTime(index, null);
    }

    public LocalDateTime getLocalDateTime(int index, LocalDateTime defaultVal) {
        String next = get(index);
        if (next == null) return defaultVal;
        return LocalDateTime.parse(next.trim());
    }


    public LocalDate nextLocalDate() {
        return nextLocalDate(null);
    }

    public LocalDate nextLocalDate(LocalDate defaultVal) {
        String next = next();
        if (next == null) return defaultVal;
        return LocalDate.parse(next.trim());
    }

    public LocalDate getLocalDate(int index) {
        return getLocalDate(index, null);
    }

    public LocalDate getLocalDate(int index, LocalDate defaultVal) {
        String next = get(index);
        if (next == null) return defaultVal;
        return LocalDate.parse(next.trim());
    }

    public LocalTime nextLocalTime() {
        return nextLocalTime(null);
    }

    public LocalTime nextLocalTime(LocalTime defaultVal) {
        String next = next();
        if (next == null) return defaultVal;
        return LocalTime.parse(next.trim());
    }

    public LocalTime getLocalTime(int index) {
        return getLocalTime(index, null);
    }

    public LocalTime getLocalTime(int index, LocalTime defaultVal) {
        String next = get(index);
        if (next == null) return defaultVal;
        return LocalTime.parse(next.trim());
    }

    public <T> T getObject(int index, Function<String, T> decoder) {
        return getObject(index, decoder, null);
    }

    public <T> T getObject(int index, Function<String, T> decoder, T defaultVal) {
        String next = get(index);
        if (next == null) return defaultVal;
        return decoder.apply(next);
    }

    public StringSpliter nextAsSpliter(String delimiter) {
        return of(nextString(), delimiter);
    }

    public StringSpliter nextAsSpliter(String delimiter, int limit) {
        return of(nextString(), delimiter, limit);
    }

    public StringSpliter getAsSpliter(int index, String delimiter) {
        return getAsSpliter(index, delimiter, 0);
    }

    public StringSpliter getAsSpliter(int index, String delimiter, int limit) {
        return of(get(index), delimiter, limit);
    }

    public String toString() {
        return StringLinker.of("{", ",", "}").add("[", msg, "]").add("pos:", pos).toString();
    }

}
