/*
 *
 *
 *
 */
package cn.gongler.util.text;

import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.function.UnaryOperator;

import static cn.gongler.util.GonglerUtil.WithDefault;

/**
 * LED文字格式化（补齐或截断）工具
 *
 * @author gongler
 * @since 2017.03.07
 */
public class ColumnWidthFiller {

    private static final long serialVersionUID = -5957526185840744064L;//ColumnWidthFiller @since 2017-03-11

    private final Charset charset;
    private final StringBuilder buf = new StringBuilder();
    private int widthTotal = 0;

    public static ColumnWidthFiller of() {
        return GBK();
    }

    private static ColumnWidthFiller GBK() {
        return new ColumnWidthFiller(Charset.forName("GBK"));
    }

    //    private static ColumnWidthFiller UTF8() {
//        return new ColumnWidthFiller(UTF_8);
//    }
//
    private ColumnWidthFiller(Charset charset) {
        this.charset = charset;
    }

    /**
     * 得到文字占用屏幕的宽度（以ASC码为单位）
     *
     * @param msg
     * @return
     */
    private int showWidth(String msg) {
        return msg.getBytes(charset).length;
    }

    public ColumnWidthFiller add(int width, Object msg) {
        return left(width, msg);
    }

    private ColumnWidthFiller left(int width, Object obj) {
        this.widthTotal += width;
        String msg = String.valueOf(WithDefault(obj, ""));
        buf.append(msg);
        int cha = width - showWidth(msg);
        if (cha > 0) {
            for (int i = 0; i < cha; i++) {
                buf.append(' ');
            }
        }

        return this;
    }

    private ColumnWidthFiller right(int width, Object obj) {
        this.widthTotal += width;
        String msg = String.valueOf(WithDefault(obj, ""));
        int cha = width - showWidth(msg);
        if (cha > 0) {
            for (int i = 0; i < cha; i++) {
                buf.append(' ');
            }
        }
        buf.append(msg);

        return this;
    }

    public ColumnWidthFiller skip(int count) {
        for (int i = 0; i < count; i++) {
            buf.append(' ');
        }
        this.widthTotal += count;
        return this;
    }

    /**
     * 自己的宽度就是应该占用的宽度。通常是最后一项
     *
     * @param val 添加的项
     * @return result
     */
    public ColumnWidthFiller add(Object val) {
        if (val != null) {
            String msg = String.valueOf(val);
            buf.append(msg);
            this.widthTotal += showWidth(msg);
        }
        return this;
    }

    /**
     * 高级用法：可以先定义后几项的添加项。
     *
     * @param val 添加所有项
     * @return result
     */
    public ColumnWidthFiller addAll(UnaryOperator<ColumnWidthFiller> val) {
        return val.apply(this);
    }

    /**
     * 默认允许长度溢出，如果需要强制截断，请执行本方法。
     *
     * @param ledWidth 截断用的长度
     * @return 截断后的长度
     */
    public ColumnWidthFiller forceWidth(int ledWidth) {
        byte[] original = buf.toString().getBytes(charset);
        if (original.length == ledWidth) {
            return this;
        }

        byte[] copy = new byte[ledWidth];
        Arrays.fill(copy, (byte) ' ');
        System.arraycopy(original, 0, copy, 0, Math.min(original.length, ledWidth));
        String m = new String(copy, charset);
        return ColumnWidthFiller.of().left(ledWidth, m);
    }

    /**
     * 默认允许长度溢出，如果需要用单项理论长度和强制截断，请执行本方法。
     *
     * @return 截断后的长度
     */
    public ColumnWidthFiller forceWidth() {
        return forceWidth(widthTotal);
    }

    public String build() {
        return buf.toString();
    }

    @Override
    public String toString() {
        return buf.toString();
    }

//    public static void main(String[] args) {
//        String msg = ColumnWidthFiller.of().add(3, "a").add(4, "12345").forceWidth().build();
//        System.out.println("" + msg);
//        //System.out.println("bytes:" + msg.getBytes(GBK).length);
//    }

}
