/*
 */
package cn.gongler.util.sgeo.geo;

import cn.gongler.util.db.ConnectionFactory;
import cn.gongler.util.db.DbUtil;
import cn.gongler.util.tuple.Tuple;
import cn.gongler.util.tuple.Tuple3;
import cn.gongler.util.tuple.Tuple5;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.SQLException;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static cn.gongler.util.sgeo.geo.ScopeGroup.SCOPE_QUICK_MAP;

/**
 * @author gongler
 * @since 2017.09.06
 */
public class ScopeGroupFactory {

    private static final long serialVersionUID = 1L;//since 2017-09-06

    private static final ScopeGroupFactory INSTANCE = new ScopeGroupFactory();

    public static ScopeGroupFactory of() {
        return INSTANCE;
    }

    private final Map<Long, ScopeGroup> GROUP_MAP = new ConcurrentSkipListMap();

    private ScopeGroupFactory() {
        register(ScopeGroup.CHANGZHANS_SCOPEGROUP);
        register(ScopeGroup.BUSSTOPS_SCOPEGROUP);
        register(ScopeGroup.LINESCOPE_SCOPEGROUP);
        register(ScopeGroup.OVERSPEED_SCOPEGROUP);
        register(ScopeGroup.CITYSCOPE_SCOPEGROUP);
    }

    private void register(ScopeGroup group) {
        GROUP_MAP.put(group.id(), group);
    }

    public Map<Long, ScopeGroup> groupMap() {
        return Collections.unmodifiableMap(GROUP_MAP);
    }

    public ScopeGroup group(long groupId) {
        return GROUP_MAP.computeIfAbsent(groupId, id -> new ScopeGroup(id, String.valueOf(id), null));
    }

    public Scope scope(long scopeId) {//20170907gongleradd 适配直接用围栏号获取围栏
        //return GROUP_MAP.values().stream().flatMap(a -> a.scopes().stream()).filter(scope -> scope.id() == scopeId).findAny().orElse(null);//适配
        return ScopeGroup.QuickScope(scopeId);
    }

    public Map<Long, Scope> scopeQuickMap() {//20170908gongler 仅用于开发阶段的调试监测
        return Collections.unmodifiableMap(SCOPE_QUICK_MAP);
    }

    private static Scope LoadScope(Connection conn, Long scopeId) throws SQLException {
        Scope[] ret = new Scope[1];
        DbUtil.ExecuteQuery(conn, "SELECT * FROM GPS_SCOPE_HEADER WHERE SCOPE_DISABLE=0 AND SCOPE_NO=?", (rs, rowIndex) -> {
            Scope.Builder scopeBuilder = new Scope.Builder(rs.getLong("SCOPE_NO")).name(rs.getString("SCOPE_NAME"))
                    .num1(rs.getLong("EXT_NUM1")).num2(rs.getLong("EXT_NUM2")).group(rs.getLong("SCOPE_GROUP"));

            DbUtil.ExecuteQuery(conn, "SELECT GEO_L, GEO_B FROM GPS_SCOPE_DETAIL WHERE SCOPE_NO=? ORDER BY SERIAL ", (rs2, rowIndex2) -> {
                double geoL = rs2.getDouble("GEO_L");
                double geoB = rs2.getDouble("GEO_B");
                scopeBuilder.add(geoL, geoB);
            }, scopeId);
            Scope scope = scopeBuilder.build();
            ret[0] = scope;
        }, scopeId);

        return ret[0];
    }

    private static List<Scope> LoadGroup(Connection conn, Long groupId) throws SQLException {
        final List<Scope> list = new LinkedList();
        DbUtil.ExecuteQuery(conn, "SELECT SCOPE_NO FROM GPS_SCOPE_HEADER WHERE SCOPE_DISABLE=0 and scope_group =? ORDER BY SCOPE_NO", (rs, rowIndex) -> {//rownum<20 and
            long scopeId = rs.getLong("SCOPE_NO");
            Scope scope = LoadScope(conn, scopeId);
            //System.out.println("scope" + scopeId + ": " + scope);
            list.add(scope);
        }, groupId);
        return list;
    }

    private void loadGroup(Connection conn, Long groupId) throws SQLException {
        final List<Scope> list = LoadGroup(conn, groupId);
        group(groupId).addAllScope(list);

        //日志输出
        //System.out.println("ScopeGroupFactory.of(" + groupId + ")" + ScopeGroupFactory.of().group(groupId));
        for (Scope scope : group(groupId).scopes()) {
            //System.out.println("ScopeGroup" + groupId + ".scope[i]:" + scope);
        }
    }

    private void loadAll(Connection conn) throws SQLException {
        DbUtil.ExecuteQuery(conn, "SELECT SCOPE_GROUP FROM GPS_SCOPE_HEADER WHERE SCOPE_DISABLE=0 group by SCOPE_GROUP ORDER BY SCOPE_GROUP",
                (rs, rowIndex) -> loadGroup(conn, rs.getLong("SCOPE_GROUP")));
    }

    private void loadAllFast(Connection conn) throws SQLException {
        List<Tuple3<Long, Double, Double>> detailList = new ArrayList<>();//1.scopeId, 2.geoL, 3.geoB
        DbUtil.ExecuteQuery(conn, "SELECT * FROM GPS_SCOPE_DETAIL ORDER BY SCOPE_NO, SERIAL ", (rs2, rowIndex2) -> {// where rownum<1000
            long scopeId = rs2.getLong("SCOPE_NO");
            double geoL = rs2.getDouble("GEO_L");
            double geoB = rs2.getDouble("GEO_B");
            //System.out.println(""+scopeId);
            detailList.add(Tuple.of(scopeId, geoL, geoB));
        });
        //System.out.println("GPS_SCOPE_DETAIL records:" + detailList.size());
        List<Tuple5<Long, Long, String, Long, Long>> headerList = new ArrayList<>();//1.scopeGroup, 2.scopeId, 3.scopeName, 4.scopeNum1, 5.scopeNum2
        DbUtil.ExecuteQuery(conn, "SELECT * FROM GPS_SCOPE_HEADER WHERE SCOPE_DISABLE=0 ORDER BY SCOPE_GROUP,SCOPE_NO ", (rs, rowIndex) -> {//  and scope_group>1
            headerList.add(Tuple.of(rs.getLong("SCOPE_GROUP"),
                    rs.getLong("SCOPE_NO"),
                    rs.getString("SCOPE_NAME"),
                    rs.getLong("EXT_NUM1"),
                    rs.getLong("EXT_NUM2")));

        });
        //System.out.println("GPS_SCOPE_HEADER records:" + headerList.size());
        headerList.stream().collect(Collectors.groupingBy(Tuple5::first)).forEach((groupId, scopeInfoList) -> {
            List<Scope> scopesInGroup = new ArrayList<>();
            scopeInfoList.stream().forEach(record -> {
                long scopeId = record.second();
                Scope.Builder scopeBuilder = new Scope.Builder(scopeId).group(groupId).name(record.third()).num1(record.forth()).num2(record.fifth());
                detailList.stream().filter(t -> t.first() == scopeId).forEach(t -> scopeBuilder.add(t.second(), t.third()));
                Scope scope = scopeBuilder.build();
                scopesInGroup.add(scope);
            });
            group(groupId).addAllScope(scopesInGroup);
        });

    }

    public static final LocalDateTime DEFAULT_LOCALDATETIME = LocalDateTime.of(2000, 1, 1, 1, 1);
    public LocalDateTime allLoadTime = DEFAULT_LOCALDATETIME;
    public LocalDateTime allLoadFinishedTime = DEFAULT_LOCALDATETIME;
    public LocalDateTime allLoadSucessTime = DEFAULT_LOCALDATETIME;

    ////////////////////////////////////////////////////////////////////////////
    //给外部调用的接口
    public void loadAllFromDb(Connection conn, boolean fastMode) throws SQLException {
        //System.out.println("loadAllFromDb...");
        allLoadTime = LocalDateTime.now();
        try {
            if (fastMode) {
                loadAllFast(conn);
            } else {
                loadAll(conn);
            }
            allLoadSucessTime = LocalDateTime.now();
        } finally {
            allLoadFinishedTime = LocalDateTime.now();
            int allScopeCount = GROUP_MAP.values().stream().mapToInt(a -> a.scopes().size()).sum();
            //System.out.println("LoadAllScopesFromDb:" + (fastMode ? "fastMode" : "normalMode") + ", " + allScopeCount + ", " + allLoadTime + "" + allLoadFinishedTime + ", " + allLoadSucessTime);
        }
    }

    public void removeScope(long scopeId) {
        GROUP_MAP.values().forEach(group -> group.removeScope(scopeId));
    }

    public void reloadScope(Connection conn, long scopeId) throws SQLException {
        Scope scope = LoadScope(conn, scopeId);
        if (scope != null) {//避免数据库无法访问造成旧的也被删除。
            removeScope(scopeId);
            group(scope.groupId()).addScope(scope);
        }
    }

    ////////////////////////////////////////////////////////////////////////////
    //初始化数据库
    ConnectionFactory connFactory;

    public ScopeGroupFactory connectionFactory(ConnectionFactory connFactory) {//临时兼容
        this.connFactory = connFactory;
        //System.out.println("ScopeGroupFactory.connFactory=" + this.connFactory);//gongler20200521add 排查connFactory空指针
        //数据库初始化
        synchronized (scopeFirstLoadLock) {
            scopeFirstLoadLock.notify();
        }
        return this;
    }

    public ScopeGroupFactory connectionFactory(DataSource ds) {//临时兼容
        return connectionFactory(ds::getConnection);
    }

    ////////////////////////////////////////////////////////////////////////////
    //载入数据
    //private boolean needReload = true;
    //private LocalDateTime lastTryLoadTime = LocalDateTime.of(2000, 1, 1, 1, 1);
    private final Object scopeFirstLoadLock = new Object();

    Thread dbLoadThread = new Thread(() -> {
        try {
            synchronized (scopeFirstLoadLock) {
                scopeFirstLoadLock.wait(TimeUnit.MINUTES.toMillis(30));//一旦外部就绪就立即载入围栏
            }
            while (true) {
                //System.out.println("dbLoadThread.loadAllFromDb()...");
                try (Connection conn = connFactory.getConnection()) {//gongler20200521 会抛出空指针，待处理
                    loadAllFromDb(conn, true);
                } catch (Throwable e) {
                    e.printStackTrace();
                }
                Thread.sleep(TimeUnit.MINUTES.toMillis(30));
            }
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
    });

    {
        dbLoadThread.setDaemon(true);
        dbLoadThread.start();
    }

    public boolean inside(IGeoPoint gps, long scopeId) {
        Scope scope = scope(scopeId);
        if (scope != null) {
            return scope.inside(gps);
        } else {
            return false;
        }
    }


    @Override
    public String toString() {
        return "{ScopeGroupFactory,groups:" + this.groupMap().size() + ",allLoadTime:" + this.allLoadTime + "}";
    }

//    public static void main(String[] args) throws SQLException, InterruptedException {
////        2, scope11,11 num1:0 num2:11 pnt:5 area:0.0
////	GeoPoint(11.0,114.556011)
////	GeoPoint(11.0,114.556357)
////	GeoPoint(11.0,114.39844)
////	GeoPoint(11.0,114.398785)
////	GeoPoint(11.0,114.556703)
//
//        try (Connection conn = DriverManager.getConnection("jdbc:oracle:thin:@121.28.98.12:11522:sjzgps2", "gjgps", "sjzgjgps001");
//                Statement statement = conn.createStatement();) {
//            ResultSet rs = statement.executeQuery("select * from dual ");
//            while (rs.next()) {
//                System.out.println("." + rs.getString(1));//System.out.println(".");
//            }
//            //LoadScope(conn, 123L);
//            ScopeGroupFactory factory = ScopeGroupFactory.of();
////            factory.beforeAccessDb(conn);//factory.loadAllFast(conn);
////            factory.beforeAccessDb(conn);
//            for (int i = 0; i < 100; i++) {
//                Thread.sleep(1000 * 10);
//                //factory.beforeAccessDb(conn);
//                System.out.println("" + factory);
//                factory.GROUP_MAP.forEach((groupId, group) -> {
//                    System.out.println("//group" + groupId);
//                    for (Scope scope : group.scopes()) {
//                        System.out.println("" + groupId + ", " + scope);
//                        if (scope.pointCount() > 0 && scope.polygonArea() == 0) {
//                            for (IGeoPoint point : scope.geoPoints()) {
//                                System.out.println("\t" + point);
//                            }
//                        }
//                    }
//                });
//            }
//        }
//    }

}
