/*
 */
package cn.gongler.util.sgeo.geo;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * @author gongler
 * @since 2017.09.06
 */
public class ScopeGroup implements Iterable<Scope> {

    private static final long serialVersionUID = 1L;//since 2017-09-06

    public final static long BUSSTOP_GROUPID = 1;
    public final static long LINESCOPE_GROUPID = 2;
    public final static long CHANG_GROUPID = 3;
    public final static long OVERSPEED_GROUPID = 4;
    public final static long CITYSCOPE_GROUPID = 5;//20130725
    public final static long OIL_LEAVELINE_GROUPID = 6;//20140920

    public final static Comparator<Scope> AREA_CMP = Comparator.comparingDouble(Scope::polygonArea).thenComparingLong(Scope::id);//按围栏面积排序
    public final static ScopeGroup CHANGZHANS_SCOPEGROUP = new ScopeGroup(CHANG_GROUPID, "场站", AREA_CMP);//按围栏面积排序
    public final static ScopeGroup BUSSTOPS_SCOPEGROUP = new ScopeGroup(BUSSTOP_GROUPID, "站点", null);
    public final static ScopeGroup LINESCOPE_SCOPEGROUP = new ScopeGroup(LINESCOPE_GROUPID, "线路围栏", null);
    public final static ScopeGroup OVERSPEED_SCOPEGROUP = new ScopeGroup(OVERSPEED_GROUPID, "限速区域", AREA_CMP);//按围栏面积排序
    public final static ScopeGroup CITYSCOPE_SCOPEGROUP = new ScopeGroup(CITYSCOPE_GROUPID, "城区区域", null);//襄阳出租车

    private final long id;
    private final String name;
    private static final Comparator<Scope> DEFAULT_CMP = Comparator.comparingLong(Scope::id);
    private final SortedSet<Scope> scopes;

    ScopeGroup(long id, String name, Comparator<Scope> cmp) {
        this.id = id;
        this.name = name;
        scopes = new ConcurrentSkipListSet(cmp != null ? cmp : DEFAULT_CMP);
    }

    public long id() {
        return id;
    }

    public String name() {
        return name;
    }

    public Scope firstInside(IGeoPoint gps) {
        for (Scope scope : scopes()) {
            if (scope.inside(gps)) {
                return scope;
            }
        }
        return null;
    }

    public Scope firstInside(IGeoPoint gps, Predicate<Scope> filter) {//20170912gongleradd
        for (Scope scope : scopes()) {
            if (filter.test(scope) && scope.inside(gps)) {
                return scope;
            }
        }
        return null;
    }

    public Collection<Scope> allInside(IGeoPoint gps) {
        return scopes().stream().filter(scope -> scope.inside(gps)).collect(Collectors.toList());
    }

    ////////////////////////////////////////////////////////////////////////////
    //

    public Collection<Scope> scopes() {//只读
        return Collections.unmodifiableCollection(scopes);
    }

    static final Map<Long, Scope> SCOPE_QUICK_MAP = new ConcurrentHashMap<>();

    static Scope QuickScope(long scopeId) {//不允许外部包直接访问。
        return SCOPE_QUICK_MAP.get(scopeId);
    }

    void removeScope(long scopeId) {
        scopes.stream().filter(scope -> scope.id() == scopeId).collect(Collectors.toList()).forEach(scopes::remove);
        SCOPE_QUICK_MAP.remove(scopeId);
    }

    void addScope(Scope scope) {
        scopes.add(scope);
        SCOPE_QUICK_MAP.put(scope.id(), scope);
    }

    void flushScope(Scope scope) {
        removeScope(scope.id());
        addScope(scope);
    }

    void addAllScope(Collection<Scope> scopes) {
        this.scopes.stream().mapToLong(Scope::id).forEach(SCOPE_QUICK_MAP::remove);
        this.scopes.clear();
        this.scopes.addAll(scopes);
        SCOPE_QUICK_MAP.putAll(scopes.stream().collect(Collectors.toMap(Scope::id, Function.identity())));
    }

    public Scope scope(long scopeId) {
        return scopes.stream().filter(s -> s.id() == scopeId).findAny().orElse(null);//注意：在id输入错误时，不能用QuickScope错误取得其他group的scope。
    }

    ////////////////////////////////////////////////////////////////////////////
    @Override
    public Iterator<Scope> iterator() {
        return this.scopes().iterator();
    }

    @Override
    public String toString() {
        return "{scopeGroup" + id() + "," + name + " scopes:" + scopes.size() + "}";
    }

}
