/*
 */
package cn.gongler.util.sgeo.geo;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

/**
 * 折线插值：根据限定间隔插入中间点（配合双流正先线路站点调整自适应方案的正先需求设计）
 * 2017.09.30注：【限制】目前是一个简易实现，精度1毫米。不适宜用于日期变更线附近的插值。
 * <pre>
 * IGeoPoint point1 = IGeoPoint.of(117.976043D, 39.638393D);
 * IGeoPoint point2 = IGeoPoint.of(117.968462D, 39.639923D);
 * IGeoPoint point3 = IGeoPoint.of(118.115075D, 39.624328D);
 * List&lt;IGeoPoint&gt; points = EquidistancePolylineBuilder.of(50).add(point1).add(point2).add(point3).build();
 * System.out.println(points);
 * </pre>
 *
 * @author gongler
 * @since 2017.09.30
 */
public class EquidistancePolylineBuilder {

    private static final long serialVersionUID = 11369096263413918L;//2017-09-30 Util

    /**
     * @param meter 间隔，单位米
     * @return instance
     */
    public static EquidistancePolylineBuilder of(int meter) {
        return new EquidistancePolylineBuilder(meter);
    }

    private final int meter;
    LinkedList<IGeoPoint> points = new LinkedList<>();

    private EquidistancePolylineBuilder(int meter) {
        this.meter = meter;
    }

    private static IGeoPoint CreateNext(IGeoPoint previous, IGeoPoint targetPoint, int meters) {
        double distanceMeters = previous.metersTo(targetPoint);
        if (distanceMeters < meters) {//过近，无需插点。
            //System.out.println("过近不再插值" + distanceMeters);
            return null;
        } else {
            //先快速实现近似算法
            double persent = meters / distanceMeters;
            return IGeoPoint.of(previous.gpsLng() + (targetPoint.gpsLng() - previous.gpsLng()) * persent, previous.gpsLat() + (targetPoint.gpsLat() - previous.gpsLat()) * persent);
        }
    }

    private List<IGeoPoint> createMiddlePoints(IGeoPoint point1, IGeoPoint point2) {
        List<IGeoPoint> middlePoints = new ArrayList<>();
        IGeoPoint previous = point1;
        while (true) {
            IGeoPoint next = CreateNext(previous, point2, meter);
            if (next == null) {
                break;
            } else {
                middlePoints.add(next);
                previous = next;
            }
        }
        return middlePoints;
    }

    /**
     * 添加一个节点
     *
     * @param point 定位点
     * @return instance
     */
    public EquidistancePolylineBuilder add(IGeoPoint point) {
        if (!points.isEmpty()) {
            IGeoPoint previous = points.getLast();
            List<IGeoPoint> middlePoints = createMiddlePoints(previous, point);
            points.addAll(middlePoints);
        }
        points.add(point);
        return this;
    }

    /**
     * 添加多个节点
     *
     * @param points points
     * @return this
     */
    public EquidistancePolylineBuilder addAll(List<IGeoPoint> points) {
        for (IGeoPoint point : points) {
            add(point);
        }
        return this;
    }

    /**
     * 返回按指定间隔米数的插值的坐标序列（含原坐标）的列表。
     *
     * @return points
     */
    public List<IGeoPoint> build() {
        return points;
    }

//    public static void main(String[] args) {
////17:29:55/192.168.163.252:34012 BusGpsPack(c2) [0]LINE=1161 [1]BUS=300 [2]时间戳=20170929173017 [3]纬度=39.638393 [4]经度=117.976043 [5]时速=9 [6]方位角=104 [7]高度=0 [8]车辆状态=3 [9]当前站点序号=16 [10]车厢内温度=0 [11]GPRS信号强度=13 [12]当天行驶里程=180277 [13]车外温度=0 [14]GPS卫星数（取反）=0 [15]GPS属性=16 [16]累计油量=235 [17]站内标记=0 [18]6102保留=0 [19]设定温度=0 [20]保留=0
////17:27:54/192.168.163.252:34012 BusGpsPack(c2) [0]LINE=1161 [1]BUS=300 [2]时间戳=20170929172817 [3]纬度=39.639923 [4]经度=117.968462 [5]时速=35 [6]方位角=104 [7]高度=0 [8]车辆状态=3 [9]当前站点序号=14 [10]车厢内温度=0 [11]GPRS信号强度=13 [12]当天行驶里程=179520 [13]车外温度=0 [14]GPS卫星数（取反）=0 [15]GPS属性=16 [16]累计油量=161 [17]站内标记=0 [18]6102保留=0 [19]设定温度=0 [20]保留=0
////08:49:40/192.168.163.252:41941 BusGpsPack(c2) [0]LINE=1161 [1]BUS=300 [2]时间戳=20170930084958 [3]纬度=39.624328 [4]经度=118.115075 [5]时速=27 [6]方位角=180 [7]高度=0 [8]车辆状态=83 [9]当前站点序号=32 [10]车厢内温度=0 [11]GPRS信号强度=5 [12]当天行驶里程=48667 [13]车外温度=0 [14]GPS卫星数（取反）=0 [15]GPS属性=16 [16]累计油量=245 [17]站内标记=0 [18]6102保留=0 [19]设定温度=0 [20]保留=0
//        IGeoPoint point1 = IGeoPoint.of(117.976043D, 39.638393D);
//        IGeoPoint point2 = IGeoPoint.of(117.968462D, 39.639923D);
//        IGeoPoint point3 = IGeoPoint.of(118.115075D, 39.624328D);
//        double dis = point1.metersTo(point2) + point2.metersTo(point3);
//        System.out.println("原始距离（米）：" + dis);
//        List<IGeoPoint> points = EquidistancePolylineBuilder.of(50).add(point1).add(point2).add(point3).build();
//        System.out.println(points);
//        double sum = 0;
//        for (int i = 1; i < points.size(); i++) {
//            double meter = points.get(i - 1).metersTo(points.get(i));
//            sum += meter;
//            System.out.println("sum=" + sum + ", add:" + meter + "//" + points.get(i));
//        }
//        System.out.println("插值后累计距离（米）：" + sum);
//    }

}
