/*
 * 2013/02~2013/03 created for Sender(ver2).
 * @author gongler
 */
package cn.gongler.util.resend.db;

import cn.gongler.util.GonglerUtil;
import cn.gongler.util.QueueConsumer;
import cn.gongler.util.db.ConnectionFactory;
import cn.gongler.util.db.DbUtil;
import cn.gongler.util.db.IDbTask;
import cn.gongler.util.function.ExceptionBiFunction;
import cn.gongler.util.function.ExceptionConsumer;
import cn.gongler.util.resend.ISender;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.RowId;
import java.time.LocalDateTime;
import java.util.Objects;
import java.util.function.BiPredicate;
import java.util.function.Consumer;

import static cn.gongler.util.GonglerUtil.Commit;

/**
 * 2013.03.07 增加提醒重新检测发送表机制。 2013.03.01
 * 未来：增加连接事件，触发重新开始发送待发送的数据。前提：网关保存车辆在线状态，并产生连线断线事件（含IP端口信息） 2014.03.27 16:58
 * 减负：取消多处低于1分钟（10秒）的定时监测；整理代码;
 */

/**
 * 该发送机制仅适用于支持RowId的系统。
 *
 * @param <Pack> pack
 * @author gongler
 */
public class RowIdDbSendPackScanner<Pack extends ISendPack, AckPack extends IAckPack> {

    private static final long serialVersionUID = 1L;

    protected String resetLoadTimeSql = "UPDATE EXT_PACK_SEND_QUEUE SET LOAD_TIME=NULL WHERE LOAD_TIME IS NOT NULL AND AND ((EXT_WAIT_ACK=1 AND ACK_CODE IS NULL) OR (EXT_WAIT_ACK=0 AND SENT_TIME IS NULL)) AND SYSDATE<SEND_EXPIRED_TIME";
    protected String loadSql = "SELECT t.ROWID, t.* FROM EXT_PACK_SEND_QUEUE t WHERE LOAD_TIME IS NULL ORDER BY ISSUE_TIME";//rowid必须写成t.rowid, 如果没有t.则数据库执行错误。疑似规定如此。
    protected String loadedSql = "UPDATE EXT_PACK_SEND_QUEUE SET LOAD_TIME=SYSTIMESTAMP WHERE ROWID=?";
    protected String sendSql = "UPDATE EXT_PACK_SEND_QUEUE SET SENT_TIME=SYSTIMESTAMP, TRY_COUNT=TRY_COUNT+1 WHERE ROWID=?";
    protected String ackSql = "UPDATE EXT_PACK_SEND_QUEUE SET ACK_TIME=SYSTIMESTAMP, ACK_CODE=? WHERE ROWID=?";
    protected String expiredSql = "UPDATE EXT_PACK_SEND_QUEUE SET ACK_TIME=SYSTIMESTAMP, ACK_CODE=-2 WHERE ROWID=?";

    final Runnable initTask;
    final ExceptionConsumer<SendPackScanner.ISendPackRegister<Pack, RowId>> checker;
    final ISender<Pack> realSender;

    BiPredicate<Pack, AckPack> ackChecker = (pack, ack) -> ack.isAckOf(pack);

    SendPackScanner<Pack, RowId, AckPack> poolingBusResender;
    private final QueueConsumer queueAutoConsumer;
    Consumer<IDbTask> dbTaskQueueAutoConsumerView;
    ConnectionFactory connectionFactory;
    private static final String DEFAULT_SEND_TABLE = "EXT_PACK_SEND_QUEUE";
    private String sendTableName = DEFAULT_SEND_TABLE;

//    public SendPackScanner<Pack, RowId, AckPack> ref() {
//        return poolingBusResender;
//    }

    /**
     * @param connectionFactory 获取连接
     * @param packFactory       ResultSet转换成Pack
     * @param realSender        触发重发动作
     */
    public RowIdDbSendPackScanner(ConnectionFactory connectionFactory, ExceptionBiFunction<String, ResultSet, Pack> packFactory, ISender<Pack> realSender) {
        Objects.requireNonNull(connectionFactory);
        Objects.requireNonNull(packFactory);
        Objects.requireNonNull(realSender);
        this.connectionFactory = connectionFactory;
        queueAutoConsumer = QueueConsumer.of("DbSender.queueAutoConsumer");
        dbTaskQueueAutoConsumerView = queueAutoConsumer.toView(connectionFactory::acceptWithCatchAny);
        this.realSender = realSender;

        initTask = () -> {
            System.out.println("sendTable reset status//" + resetLoadTimeSql);
            connectionFactory.acceptWithCatchAny(conn -> {
                DbUtil.ExecuteUpdate(conn, resetLoadTimeSql);//对未过期但尚未发送成功的记录，要清除已读标记，以便后继重新加载。
                GonglerUtil.Commit(conn);//conn.commit();
            });
        };

        checker = lis -> {
            try (Connection conn = connectionFactory.getConnection();
                 PreparedStatement statement = conn.prepareStatement(loadSql);
                 PreparedStatement updateLoadTimePstm = conn.prepareStatement(loadedSql)) {
                ResultSet rs = statement.executeQuery();
                while (rs.next()) {
                    DbUtil.ExecuteUpdate(conn, updateLoadTimePstm, rs.getRowId("ROWID"));//内部有自动回滚机制//无论数据是否合法，都要在数据库中标记已经读过该数据库。避免重复读入。
                    Commit(conn);//conn.commit();//及时提交，避免其他修改动作死锁。

                    final long busId = rs.getLong("TARGET_NO");
                    final boolean waitAck = rs.getInt("EXT_WAIT_ACK") != 0;//群发是置为no ack
                    LocalDateTime expiredTime = rs.getTimestamp("SEND_EXPIRED_TIME").toLocalDateTime();
                    final RowId rowId = rs.getRowId("ROWID");
                    String packBody = rs.getString("PACK_BODY");

                    final Pack sendPack = packFactory.applyWithCatchAny(packBody, rs);//packFactory.apply(rs);
                    if (sendPack != null) {
                        lis.add(busId, sendPack, expiredTime, waitAck, rowId);
                    }
                }//while
            }
        };

    }

    /**
     * @param sendTable 如果出现多个发送表，可以通过本方法修改发送表名
     * @return result
     */
    public RowIdDbSendPackScanner<Pack, AckPack> sendTable(String sendTable) {
        Objects.requireNonNull(sendTable);
        String oldSendTable = this.sendTableName;
        this.sendTableName = sendTable;
        resetLoadTimeSql = resetLoadTimeSql.replace(oldSendTable, sendTable);
        loadSql = loadSql.replace(oldSendTable, sendTable);
        loadedSql = loadedSql.replace(oldSendTable, sendTable);
        sendSql = sendSql.replace(oldSendTable, sendTable);
        ackSql = ackSql.replace(oldSendTable, sendTable);
        expiredSql = expiredSql.replace(oldSendTable, sendTable);
        return this;
    }

//    /**
//     * 如果存在复杂答应（虽然目前还没有发现这样的情况），可以定制应答包识别逻辑。
//     *
//     * @param ackChecker
//     * @return
//     */
//    public RowIdDbSendPackScanner<Pack, AckPack> ackChecker(BiPredicate<Pack, AckPack> ackChecker) {
//        Objects.requireNonNull(ackChecker);
//        this.ackChecker = ackChecker;
//        return this;
//    }

    /**
     * 启动发送
     */
    public void start() {
        this.poolingBusResender = new SendPackScanner<>(initTask, checker, ackChecker, realSender);
        poolingBusResender.busResender()
                .addSendEventListener((params, sendTimes) -> pushDbTask(conn -> DbUtil.ExecuteUpdate(conn, sendSql, params.param())))
                .addSendAckEventListener((params, ack, sendContext) -> pushDbTask(conn -> DbUtil.ExecuteUpdate(conn, ackSql, ack.ackCode(), params.param())))
                .addSendExpiredEventListener((params, sendTimes, sendContext) -> pushDbTask(conn -> DbUtil.ExecuteUpdate(conn, expiredSql, params.param())));
    }

    /**
     * 检查应答包是否是正在发送的指令应答。
     *
     * @param busPackAck busPackAck
     */
    public void acceptAck(AckPack busPackAck) {
        poolingBusResender.ack(busPackAck.busId(), busPackAck);
    }

    /**
     * 该方法会触发一次主动检查发送表，例如收到司机刷卡指令后主动触发，可以使得下发回应缩短半秒钟。
     */
    public void notifyRecheck() {
        poolingBusResender.notifyRecheck();//2013.03.11 如果速度慢，可以考虑转入后台线程处理。
    }

    private void pushDbTask(IDbTask dataHandler) {
        dbTaskQueueAutoConsumerView.accept(dataHandler);
    }

//    public static void main(String[] args) {
//        RowIdDbSendPackScanner<String, IAck> rowIdDbSender = new RowIdDbSendPackScanner<String, IAck>(
//                ConnectionFactory.of("jdbc:oracle:thin:@192.168.90.196:1521:zndd1962", "GJGPS_PACK", "123456"),
//                (packBody, rs) -> {
//                    System.out.println("" + packBody);
//                    return packBody;
//                }, (busId, msg) -> System.out.println("send: " + busId + ", " + msg));
//        IAck ack = new IAck() {
//            @Override
//            public int ackCode() {
//                return 0;
//            }
//        };
//        rowIdDbSender.acceptAck(123L, ack);
//
//    }
}
