/*
 * 2013/02~2013/03 created for Sender(ver2).
 * 2013/04/03 强制捕捉发送过期用户事件的异常抛出，避免因用户使用不当造成重发定时器停止。
 * @author gongler
 */
package cn.gongler.util.resend;

import cn.gongler.util.GonglerUtil;

import java.io.Closeable;
import java.time.LocalDateTime;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.IntConsumer;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * 这是个不对外开放的内部类。
 *
 * @author gongler
 */
class ResenderEngineer implements Closeable {

    private static final long serialVersionUID = -7775477028134507369L;//ResenderEngineer @since 2016-08-08

    private final long TICK_TIME = 100L;
    private final AtomicLong resendSeconds = new AtomicLong(20L);
    private volatile boolean cancel = false;
    private final BlockingQueue<TaskWrapper> sendingQueue = new LinkedBlockingQueue();//supported for JDK6 LinkedTransferQueue();

    public ResenderEngineer resendSeconds(long resendSeconds) {
        this.resendSeconds.set(resendSeconds);
        return this;
    }

    private long resendSeconds() {
        return this.resendSeconds.get();
    }

    public interface ISendTaskRef {

    }

    public ISendTaskRef addSendTask(IntConsumer send, LocalDateTime expiredTime, IntConsumer expiredEventListener) {
        TaskWrapper task = new TaskWrapper(send, expiredTime, expiredEventListener);
        sendingQueue.offer(task);
        return task;
    }

    public void removeSendTask(ISendTaskRef taskRef) {
        if (taskRef instanceof TaskWrapper) {
            //延迟删除。
            final TaskWrapper wrapper = (TaskWrapper) taskRef;
            wrapper.cancel();
        }
    }

    private final Thread worker = GonglerUtil.StartDaemonThread("Resend.worker",
            () -> {
                int frame = 0;
                while (!cancel) {
                    frame++;//frame = ((frame + 1) & 0xFF);
                    LocalDateTime currentTime = LocalDateTime.now();//System.currentTimeMillis();
                    while (true) {
                        final TaskWrapper taskWrapper = sendingQueue.peek();
                        if (taskWrapper != null) {
                            final LocalDateTime jobPlanTime = taskWrapper.getTimeoutTime();
                            //优化：如果条件不符，重新获取一下当前时间，再看看是否符合。减少每次后后去当前时间。
                            if (jobPlanTime.isBefore(currentTime) || jobPlanTime.isBefore(LocalDateTime.now())) {//if (jobPlanTime <= System.currentTimeMillis()) {
                                sendingQueue.poll();//remove it
                                if (taskWrapper.isCancel()) {//被外部中止（通常为收到回应）
                                } else if (taskWrapper.expiredTime().isAfter(currentTime)) {//还没有过期
                                    try {
                                        taskWrapper.notifyResendEvent(currentTime);//触发重发动作
                                    } catch (Exception e) {//20130403确保不会因用户的过期处理抛出异常，造成定时器退出。
                                        e.printStackTrace();
                                    }
                                    final LocalDateTime nextPlanTime = currentTime.plusSeconds(resendSeconds());
                                    taskWrapper.initTimeoutTime(nextPlanTime);
                                    sendingQueue.offer(taskWrapper);//20140327避免抛出异常 sendingQueue.add(taskWrapper);//登记到单次超时重发计时器。
                                } else {// when taskWrapper.expiredTime() <= currentTime
                                    try {
                                        taskWrapper.expired(currentTime);
                                    } catch (Exception e) {//20130403确保不会因用户的过期处理抛出异常，造成定时器退出。
                                        e.printStackTrace();
                                    }
                                }
                            } else {
                                break;//还没到时间，等待下一滴答
                            }
                        } else {//null
                            break;//队列空，等待下次滴答再检查。
                        }
                    }//while
                    if (frame % 0xFFF == 0) {
                        //System.out.println("resender.queueSize=" + sendingQueue.size());
                    }
                    try {
                        Thread.sleep(TICK_TIME);
                    } catch (InterruptedException ex) {
                        Logger.getLogger(ResenderEngineer.class.getName()).log(Level.SEVERE, null, ex);
                    }
                }//loop clock tick
            });

    public int getQueueSize() {
        return sendingQueue.size();
    }

    @Override
    public void close() {
        cancel = true;
        worker.interrupt();
    }

    private class TaskWrapper implements ISendTaskRef {

        private static final long serialVersionUID = 1L;

        private final LocalDateTime createTime = LocalDateTime.now();
        private final IntConsumer task;
        private final LocalDateTime expiredTime;
        private final IntConsumer expiredEventListener;

        private int sendTimes = 0;
        private LocalDateTime timeoutTime;
        private volatile boolean cancel = false;

        TaskWrapper(IntConsumer task, LocalDateTime expiredTime, IntConsumer expiredEventListener) {
            this.task = task;
            this.expiredTime = expiredTime;
            this.expiredEventListener = expiredEventListener;
            this.timeoutTime = LocalDateTime.now().plusSeconds(resendSeconds());//timeoutTime;
        }

        public void notifyResendEvent(LocalDateTime currentTime) {
            sendTimes++;
            GonglerUtil.ExecuteWithCatchAny(() -> task.accept(1 + sendTimes));//task.run();
        }

        public LocalDateTime expiredTime() {
            return expiredTime;
        }

        /**
         * @return the createTime
         */
        public LocalDateTime getCreateTime() {
            return createTime;
        }

        /**
         * @return the timeoutTime
         */
        public LocalDateTime getTimeoutTime() {
            return timeoutTime;
        }

        /**
         * @param timeoutTime the timeoutTime to set
         */
        public void initTimeoutTime(LocalDateTime timeoutTime) {
            this.timeoutTime = timeoutTime;
        }

        private void expired(LocalDateTime currentTime) {
            expiredEventListener.accept(1 + sendTimes);//.run();
        }

        /**
         * @return the cancel
         */
        public boolean isCancel() {
            return cancel;
        }

        public void cancel() {
            this.cancel = true;
        }

    }
}
