/*
 * 2013/02~2013/03 created for Sender(ver2).
 * 2013.04.22 debug：修正ignoredAckEvent()中空指针。
 * @author gongler
 */
package cn.gongler.util.resend;

import cn.gongler.util.QueueConsumer;
import cn.gongler.util.Recently;
import cn.gongler.util.function.ExceptionBiConsumer;
import cn.gongler.util.resend.ResenderEngineer.ISendTaskRef;
import cn.gongler.util.tuple.Tuple;
import cn.gongler.util.tuple.Tuple7;

import java.io.Closeable;
import java.io.IOException;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.*;
import java.util.function.BiPredicate;
import java.util.function.IntConsumer;
import java.util.stream.Collectors;

import static cn.gongler.util.GonglerUtil.ExecuteWithCatchAny;
import static cn.gongler.util.resend.ISendContext.SendState.*;

/*
 * 2013.04.03 gongler SendBus 增加synchronized, dbCheckThread(new), resendWorkThread(expired), receiverThread(ack)
 */

/**
 * 等重发间隔重发服务器。
 *
 * @param <Pack>  Pack
 * @param <Param> Parameter
 * @param <Ack>   Ack Pack
 * @author gongler
 * @since 2016-08-08
 */
public class Resender<Pack, Param, Ack> implements Closeable {

    private static final long serialVersionUID = -1227835320183986648L;//Resender @since 2016-08-08

    private final ResenderEngineer resendEngineer;
    private final Map<Long, SendBus> busSendingMap = new ConcurrentSkipListMap();
    private final BiPredicate<Pack, Ack> ackChecker;
    private final ISender<Pack> realSender;//ExceptionTriConsumer<Long, Pack, Integer> realSender;//BusId, pack,  sendTimes
    private final QueueConsumer queueConsumer = QueueConsumer.of("BusResender,queueAutoConsumer");

    private final Set<ISendEventListener<Pack, Param, Ack>> sendEventListenerSet = ConcurrentHashMap.newKeySet();
    private final Set<IAckEventListener<Pack, Param, Ack>> sendAckEventListenerSet = ConcurrentHashMap.newKeySet();
    private final Set<IExpiredEventListener<Pack, Param, Ack>> sendExpiredEventListenerSet = ConcurrentHashMap.newKeySet();

    public Resender(BiPredicate<Pack, Ack> ackChecker, ISender<Pack> realSender) {//ExceptionTriConsumer<Long, Pack, Integer> realSender) {
        this.resendEngineer = new ResenderEngineer();
        this.ackChecker = ackChecker;
        this.realSender = realSender::send;
    }

    ResenderEngineer ref() {
        return resendEngineer;
    }

    public Resender<Pack, Param, Ack> resendSeconds(long resendSeconds) {
        this.resendEngineer.resendSeconds(resendSeconds);
        return this;
    }

    static class SendContext implements ISendContext {

        private static final long serialVersionUID = 1L;

        int sendTimes = 0;
        LocalDateTime createTime = LocalDateTime.now();
        LocalDateTime firstSendTime;
        LocalDateTime lastSendTime;
        LocalDateTime finishTime;
        SendState sendState = WAITING;

        public void sendTimeIncrement() {
            sendTimes++;
            LocalDateTime now = LocalDateTime.now();
            if (firstSendTime == null) {
                firstSendTime = now;
                sendState = SENDING;
            }
            lastSendTime = now;
        }

        public void sendFinished(boolean sucess) {
            finishTime = LocalDateTime.now();
            sendState = sucess ? SENT_SUCESS : SENT_FAILTURE;
        }

        @Override
        public int sendTimes() {
            return sendTimes;
        }

        @Override
        public LocalDateTime createTime() {
            return createTime;
        }

        @Override
        public LocalDateTime firstSendTime() {
            return firstSendTime;
        }

        @Override
        public LocalDateTime lastSendTime() {
            return lastSendTime;
        }

        @Override
        public LocalDateTime finishTime() {
            return finishTime;
        }

        @Override
        public SendState sendState() {
            return sendState;
        }

        @Override
        public String toString() {
            return sendState() +
                    ",sendTimes:" + sendTimes() +
                    ",duration:" + duration() +
                    ",create:" + createTime() +
                    ",firstSendTime:" + firstSendTime() +
                    ",lastSendTime:" + lastSendTime() +
                    ",finishTime:" + finishTime();
        }
    }

    /**
     * 注册一条新发送数据
     *
     * @param busId       busId
     * @param pack        指令
     * @param expiredTime 过期时间
     * @param waitAck     需要应答
     * @param param       可以添加一个附件，在重发、过期、ack回应事件触发时，可以利用。例如RowId等。也可不用。
     */
    public void add(Long busId, Pack pack, LocalDateTime expiredTime, boolean waitAck, Param param) {
        add(busId, pack, expiredTime, waitAck, param, null);
    }

    protected void add(Long busId, Pack pack, LocalDateTime expiredTime, boolean waitAck, Param param, ExceptionBiConsumer<Ack, ISendContext> sendResultHandler) {//调试中，暂不对外开放。
        SendBus bus = busSendingMap.computeIfAbsent(busId, SendBus::new);
        ISendPackParams2<Pack, Param, Ack> params = toSendPackParams(busId, pack, expiredTime, waitAck, param, realSender, sendResultHandler);
        bus.addNewSendData(params);//bus.handleNewSendData(Tuple5.of(pack, expiredTime, waitAck, sender, obj));
    }

    /**
     * 发送成功，则清除再发数据，并自动拉出下一条待发送数据。
     *
     * @param busId   终端唯一标识，可能是Long，也可能是String
     * @param ackPack 应答
     */
    public void ack(Long busId, Ack ackPack) {
        final SendBus bus = busSendingMap.get(busId);
        if (bus != null) {
            bus.handleAck(ackPack);
        }
    }

    /**
     * @return 正在发送的车辆数
     */
    public long sendingBusCount() {
        return busSendingMap.values().stream().filter(SendBus::isSending).count();
    }

    /**
     * @return 每辆车的正在发送指令数Map
     */
    public Map<Long, Long> busPendingPackCountMap() {//20130719
        return busSendingMap.values().stream().filter(SendBus::isSending).collect(Collectors.toMap(SendBus::busId, SendBus::getQueueSize));
    }

//    public QueueAutoConsumer queueAutoConsumer() {
//        return queueAutoConsumer;
//    }

    /**
     * 注册发送事件监听器
     *
     * @param lis 监听器
     * @return this
     */
    public Resender<Pack, Param, Ack> addSendEventListener(ISendEventListener<Pack, Param, Ack> lis) {
        this.sendEventListenerSet.add(lis);
        return this;
    }

    /**
     * 注册发送应答事件监听器
     *
     * @param lis 监听器
     * @return this
     */
    public Resender<Pack, Param, Ack> addSendAckEventListener(IAckEventListener<Pack, Param, Ack> lis) {
        this.sendAckEventListenerSet.add(lis);
        return this;
    }

    /**
     * 注册发送过期事件监听器
     *
     * @param lis 监听器
     * @return this
     */
    public Resender<Pack, Param, Ack> addSendExpiredEventListener(IExpiredEventListener<Pack, Param, Ack> lis) {
        this.sendExpiredEventListenerSet.add(lis);
        return this;
    }

    private void notifySendEvent(ISendPackParams2<Pack, Param, Ack> params, int sendTimes) {
        sendEventListenerSet.stream().forEach(lis -> queueConsumer.accept(() -> lis.sendEvent(params, sendTimes)));
    }

    Recently<Long> sendMillsRecently = new Recently<>(4);

    long ackMillnsAvg() {
        OptionalDouble ret = sendMillsRecently.values().stream().mapToLong(a -> a).average();
        return ret.isPresent() ? (long) ret.getAsDouble() : -1L;
    }

    private void notifySendAckEvent(ISendPackParams2<Pack, Param, Ack> params, Ack ack) {
        ISendContext sendContext = params.sendContext();
        if (sendContext.sendTimes() == 1) {
            sendMillsRecently.push(sendContext.duration().toMillis());
        }

        sendAckEventListenerSet.stream().forEach(lis -> queueConsumer.accept(() -> lis.sendAckEvent(params, ack, params.sendContext())));
    }

    private void notifySendExpiredEvent(ISendPackParams2<Pack, Param, Ack> params, int sendTimes) {
        sendExpiredEventListenerSet.stream().forEach(lis -> queueConsumer.accept(() -> lis.sendExpiredEvent(params, sendTimes, params.sendContext())));
    }

    private void callback(Ack ackPack, ISendPackParams2<Pack, Param, Ack> sendEvent) {
        ExceptionBiConsumer<Ack, ISendContext> resultHandler = sendEvent.sendResultHandler();
        if (resultHandler != null) {
            queueConsumer.accept(() -> resultHandler.accept(ackPack, sendEvent.sendContext()));
        }
    }

    @Override
    public void close() throws IOException {
        resendEngineer.close();
    }

    @Override
    public String toString() {
        String buf = this.getClass().getSimpleName() +
                //.append("pending:").append(pendingPackCount())
                "_sendingBusCount:" + sendingBusCount() +
                "_queueAutoConsumer:" + queueConsumer +
                "_ackMillnsAvg:" + ackMillnsAvg();
        return buf;
    }

    ////////////////////////////////////////////////////////////////////////////

    /**
     * 保存车辆的发送状态
     */
    private class SendBus {

        private static final long serialVersionUID = 1L;


        final Long busId;
        private final ConcurrentLinkedQueue<ISendPackParams2<Pack, Param, Ack>> queue = new ConcurrentLinkedQueue<>();
        private long activeTime = System.currentTimeMillis();
        volatile ISendTaskRef ref;

        SendBus(Long busId) {
            this.busId = busId;
        }

        public Long busId() {
            return busId;
        }

        //private int sendTimes = 0;

        /**
         * 获得当前正在发送的数据
         *
         * @return
         */
        private ISendPackParams2<Pack, Param, Ack> getCurrent() {
            return queue.peek();//return this.sendEvent;
        }

        public long getQueueSize() {
            return queue.size();
        }

        boolean isSending() {
            return !queue.isEmpty();
        }

        /**
         * 处理新的一条待发数据
         *
         * @param sendEvent 事件
         */
        synchronized void addNewSendData(ISendPackParams2<Pack, Param, Ack> sendEvent) {//Tuple5 sendEvent) {
            boolean idle = (this.getCurrent() == null);//为了逻辑清晰，修改写法。queue.isEmpty()?true:false;

//立即更新，故不存在重复的可能性。
//            if(queue.contains(sendEvent)){
//                //忽略重复读入的数据
//                return;
//            }
            boolean sucess = queue.offer(sendEvent);//20160804 queue.add(sendEvent);
            if (sucess) {
                if (idle) {
                    notifyChangedSendingObjectEvent();//dbCheckWorkThread
                }
            }
            activeTime = System.currentTimeMillis();
        }

        /**
         * 处理回应。如果正确，则结束当前重发机制（并自动拉出下一条）。
         *
         * @param ackPack 应答指令
         */
        synchronized ISendPackParams2<Pack, Param, Ack> handleAck(Ack ackPack) {
            final ISendPackParams2<Pack, Param, Ack> sendEvent = this.getCurrent();
            if (sendEvent != null) {//if(sendEvent!=null && sendEvent.verifyAck(ackPack)){
                if (ackChecker.test(sendEvent.pack(), ackPack)) {
                    sendEvent.sendContext().sendFinished(true);
                    notifySendAckEvent(sendEvent, ackPack);//sendEventListener.sendAckEvent(data, ackPack);
                    callback(ackPack, sendEvent);
                    resendEngineer.removeSendTask(ref);
                    sendFinished();//ack
                    return sendEvent;
                }
            }
            return null;
        }

        /**
         * 提醒发送事件。可用于日志记录，监控等扩展。
         */
        private void notifyChangedSendingObjectEvent() {
            final ISendPackParams2<Pack, Param, Ack> sendEvent = this.getCurrent();//bus.next();
            if (sendEvent != null) {
                IntConsumer sendTask = times -> {
                    ExecuteWithCatchAny(() -> sendEvent.sender().send(busId, sendEvent.pack()));
                    sendEvent.sendContext().sendTimeIncrement();
                    notifySendEvent(sendEvent, times);
                };//first send//该包的首次发送，随后发送通过定时器出发。sendEventListener.sendEvent(sendEvent, 0);
                sendTask.accept(1);//first send
                if (!sendEvent.waitAck()) {//无回应包，发送一次后就可以直接发送下一个。
                    sendFinished();//sendOneTimeOnly dbCheckThread or follow previous sendFinished()
                } else {
                    ref = resendEngineer.addSendTask(sendTask, sendEvent.expiredTime(), sendTimes -> expiredEvent(sendEvent, sendTimes));
                }
            }

        }

        private void expiredEvent(ISendPackParams2<Pack, Param, Ack> sendEvent, int sendTimes) {
            sendEvent.sendContext().sendFinished(false);
            notifySendExpiredEvent(sendEvent, sendTimes);
            callback(null, sendEvent);
            sendFinished();
        }

        private boolean isInactived() {
            return System.currentTimeMillis() - activeTime > TimeUnit.HOURS.toMillis(24);
        }

        synchronized void sendFinished() {//by resendWorkThread when expired, or self
            ISendPackParams2<Pack, Param, Ack> skipNextExpiredPack = null;//
            final LocalDateTime now = LocalDateTime.now();

            //用循环语句替代嵌套方法调用，用于规避极端情况下潜在的嵌套层数过多错误。
            do {
                if (skipNextExpiredPack != null) {//这是要直接跳过的已经过期的未发送数据。
                    //System.out.println("ignore expired nosend data: " + skipNextExpiredPack);
                    try {
                        notifySendExpiredEvent(skipNextExpiredPack, 0);
                        callback(null, skipNextExpiredPack);
                    } catch (Throwable e) {
                        e.printStackTrace();
                    }
                }
                queue.poll();//
                skipNextExpiredPack = this.getCurrent();//20130318DEBUG while判断前，还不一定是需要跳过的过期包
            } while (skipNextExpiredPack != null && skipNextExpiredPack.expiredTime().isBefore(now));
            notifyChangedSendingObjectEvent();//at sendFinished() 
        }
    }

    ////////////////////////////////////////////////////////////////////////////
    private final ScheduledExecutorService scheduledSerivce = Executors.newSingleThreadScheduledExecutor();

    ////////////////////////////////////////////////////////////////////////////
    {
        ScheduledFuture<?> scheduleWithFixedDelay = scheduledSerivce.scheduleWithFixedDelay(() -> {
                    List<Long> removeList = new ArrayList();
                    for (Map.Entry<Long, SendBus> entry : busSendingMap.entrySet()) {
                        final Long key = entry.getKey();
                        final SendBus bus = entry.getValue();
                        if (bus.isInactived()) {
                            removeList.add(key);
                        }
                    }
                    for (Long key : removeList) {
                        busSendingMap.remove(key);
                    }
                } //自动清除过期不活跃车辆
                , 1, 1, TimeUnit.HOURS);
    }

    private interface ISendPackParams2<Pack, Param, Ack> extends ISendPackParams<Pack, Param, Ack> {

        SendContext sendContext();
    }

    private static <Pack, Param, Ack> ISendPackParams2<Pack, Param, Ack> toSendPackParams(Long busId, Pack pack, LocalDateTime expiredTime, boolean waitAck, Param param, ISender<Pack> sender, ExceptionBiConsumer<Ack, ISendContext> sendResultHandler) {
        Tuple7<Long, Pack, LocalDateTime, Boolean, Param, ISender<Pack>, ExceptionBiConsumer<Ack, ISendContext>> tuple = Tuple.of(busId, pack, expiredTime, waitAck, param, sender, sendResultHandler);
        return new ISendPackParams2<Pack, Param, Ack>() {
            SendContext sendContext = new SendContext();

            @Override
            public Long busId() {
                return tuple.first();
            }

            @Override
            public Pack pack() {
                return tuple.second();
            }

            @Override
            public LocalDateTime expiredTime() {
                return tuple.third();
            }

            @Override
            public boolean waitAck() {
                return tuple.forth();
            }

            @Override
            public Param param() {
                return tuple.fifth();
            }

            //            @Override
//            public ExceptionTriConsumer<Long, SendPack, Integer> sender() {
//                return tuple.sixth();
//            }
            @Override
            public ISender<Pack> sender() {
                return tuple.sixth();
            }

            @Override
            public ExceptionBiConsumer<Ack, ISendContext> sendResultHandler() {
                return tuple.seventh();
            }

            @Override
            public String toString() {
                return tuple.toString();
            }

            @Override
            public SendContext sendContext() {
                return sendContext;
            }
        };
    }

//    public static void main(String[] args) throws InterruptedException {
//        Resender<String, String, String> sender = new Resender<>((pack, ack) -> {
//            return Objects.equals(pack, ack);
//        }, (busId, pack) -> System.out.println("" + LocalDateTime.now() + ", " + busId + ", " + pack));
//        sender.resendSeconds(3)
//                .addSendAckEventListener((pack, ack, ctx) -> System.out.println("ack: " + LocalDateTime.now() + ", " + pack + ", " + ack))
//                .addSendEventListener((params, times) -> System.out.println("send: " + LocalDateTime.now() + ", " + params + ". " + times))
//                .addSendExpiredEventListener((params, times, ctx) -> System.out.println("expired: " + LocalDateTime.now() + ", " + params));
//        sender.add(123L, "pack1", LocalDateTime.now().plusSeconds(10), true, "abc", (ack, cxt) -> {
//            System.out.println("haha:" + ack + ", " + cxt.sendState() + ", " + cxt.duration());
//        });
//        sender.add(123L, "pack2", LocalDateTime.now().plusSeconds(20), false, "abc2", (ack, cxt) -> {
//            System.out.println("haha2:" + ack + ", " + cxt.sendState() + ", " + cxt.duration());
//        });
//        sender.add(123L, "pack3", LocalDateTime.now().plusSeconds(30), true, "abc3", null);
//        sender.add(123L, "pack4", LocalDateTime.now().plusSeconds(40), true, "abc4", (ack, cxt) -> {
//            System.out.println("haha4:" + ack + ", " + cxt.sendState() + ", " + cxt.duration());
//        });
//        sender.add(123L, "pack5", LocalDateTime.now().plusSeconds(30), true, "abc5", (ack, cxt) -> {
//            System.out.println("haha5:" + ack + ", " + cxt.sendState() + ", " + cxt.duration());
//        });
//        Thread.sleep(1000 * 5L);
//        sender.ack(123L, "pack12");
//        sender.ack(123L, "pack1");
//        Thread.sleep(1000 * 10L);
//        System.out.println("" + sender);
//    }
}
