/*
 * Tcps Protocol808_2015 Server
 * created by gongler at 2015.04.08
 *
 */
package cn.gongler.util.protocol.itemtype;


import cn.gongler.util.bytes.BytesBuilder;
import cn.gongler.util.bytes.BytesLoader;

import java.util.Arrays;

/**
 * 定长字符串或含长度头的字符串。
 *
 * @author gongler
 */
public class StringWithLengthItemType extends StringItemType<StringWithLengthItemType> {

    private static final long serialVersionUID = 1L;

    int bytes;//正值表示定长字符串；负值表示长度头字节数。

    public static StringWithLengthItemType withFixLength(int bytes, String name) {
        StringWithLengthItemType instance = new StringWithLengthItemType(bytes, name);
        return instance;
    }

    public static StringWithLengthItemType withLengthHeader(int headerBytes, String name) {
        StringWithLengthItemType instance = new StringWithLengthItemType(-headerBytes, name);
        return instance;
    }

    private StringWithLengthItemType(int bytes, String name) {
        super(name);//super("STRING");
        this.bytes = bytes;
    }

    /**
     * @param in loader
     * @return item
     */
    @Override
    public Item load(BytesLoader in) {
        int bodyLen = bytes < 0 ? in.loadInt(Math.abs(bytes)) : bytes;//定长或长度头获取长度
        byte[] bodyBytes = bodyLen == 0 ? in.loadRemainBytes() : in.loadBytes(bodyLen);//自如字符串字节。如果长度是0，则载入剩余字节。
        return _newItem(newString(bodyBytes));
    }

    private byte fillByte = ' ';

    public byte fillByte() {
        return fillByte;
    }

    /**
     * 允许定制定长字符串的填充字符。默认是空格
     *
     * @param newValue newValue
     * @return itemType
     */
    public StringWithLengthItemType fillByte(byte newValue) {
        this.fillByte = newValue;
        return this;
    }

    /**
     * @param itemValue itemValue
     * @param build     builder
     */
    @Override
    public void toBytes(Object itemValue, BytesBuilder build) {
        int len = 0;
        if (itemValue instanceof String) {
            String val = (String) itemValue;
            byte[] strBytes = val.getBytes(CHARSET);

            if (isFixLength()) {//由于是定长，将截断或填充空格。
                final int orgLen = strBytes.length;
                final byte[] fixLengthBytes = Arrays.copyOf(strBytes, this.bytes);//定长自动截断或补
                if (orgLen < fixLengthBytes.length) {
                    Arrays.fill(fixLengthBytes, orgLen, fixLengthBytes.length, fillByte);
                }
                build.addBytes(fixLengthBytes);
            } else {//含头字符串。header + body
                int headLen = Math.abs(this.bytes);
                build.addNum(headLen, strBytes.length);
                build.addBytes(strBytes);
            }
            //build.addByte(0);//没有字符串结束标记
        }
    }

    @Override
    public String toString() {
        return super.toString() + '_' + this.bytes;
    }

    private boolean isFixLength() {
        return this.bytes > 0;
    }

//    /**
//     * @param args
//     */
//    public static void main(String[] args) {
//        StringWithLengthItemType type = new StringWithLengthItemType(10, "name");
//        BytesBuilder build = new BytesBuilder();
//        type.toBytes("string", build);
//        System.out.println("" + build);
//    }

}
