/*
 * gongler 设备模拟器（仅做接收和应答指令）
 * 2018.08.28
 */
package cn.gongler.util.net.client;

import cn.gongler.util.GonglerUtil;
import cn.gongler.util.ITask;
import cn.gongler.util.QueueConsumer;
import cn.gongler.util.function.ExceptionBiConsumer;
import cn.gongler.util.function.ExceptionFunction;

import java.io.DataInputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.time.LocalDateTime;
import java.util.Timer;
import java.util.function.Supplier;

/**
 * TCP 客户端简易引擎
 *
 * @param <Pack> pack
 * @author gongler
 * @since 2018.09.05
 */
public class TcpClient<Pack> {

    private static final long serialVersionUID = 1L;

    Socket socket = null;
    Timer timer;
    InetSocketAddress svrAddr;
    QueueConsumer sendConsumer;
    QueueConsumer recvConsumer;
    DataInputStream in = null;
    OutputStream out = null;
    private LocalDateTime connectTime;

    private class ReceiveTask implements ITask {

        volatile boolean cancel = false;
        private final ITask task;

        ReceiveTask(ITask task) {
            this.task = task;
        }

        public void stop() {
            this.cancel = true;
        }

        @Override
        public void run() throws Exception {
            try {
                task.run();
            } catch (IOException e) {//gongler20220708debug
                GonglerUtil.Close(socket);
                socket = null;
                throw e;
            }
        }

    }

    private ReceiveTask lastRecvTask = null;
    private ExceptionFunction<Pack, byte[]> packToBytes;
    private ExceptionBiConsumer<ITcpClientContext, IClientSender<Pack>> heartHandler = (cxt, s) -> {
    };
    ITcpClientContext cxt;

    /**
     * @param host             主机
     * @param port             端口
     * @param recvFactory      接收工厂
     * @param packToBytes      序列化工具
     * @param connPackFactory  连接处理工厂
     * @param heartPackFactory 心跳工厂
     */
    public TcpClient(String host, int port, Supplier<IReceiveHandler<Pack>> recvFactory, ExceptionFunction<Pack, byte[]> packToBytes, ExceptionFunction<ITcpClientContext, Pack> connPackFactory, ExceptionFunction<ITcpClientContext, Pack> heartPackFactory) {//ToIntFunction<byte[]> receiver){
        this.svrAddr = new InetSocketAddress(host, port);
        this.sendConsumer = QueueConsumer.of("sender");
        this.recvConsumer = QueueConsumer.of("receiver");
        this.packToBytes = packToBytes;

        this.timer = new Timer();
        timer.schedule(GonglerUtil.TimerTask(() -> {
            if (socket == null) {
                Socket socket = new Socket();
                socket.connect(svrAddr);
                this.socket = socket;
                this.in = new DataInputStream(socket.getInputStream());
                this.out = socket.getOutputStream();
                this.connectTime = LocalDateTime.now();
                this.cxt = new ITcpClientContext() {
                    @Override
                    public LocalDateTime connectedTime() {
                        return connectTime;
                    }

                    @Override
                    public InetSocketAddress svrAddr() {
                        return (InetSocketAddress) svrAddr;
                    }

                    @Override
                    public InetSocketAddress clientAddr() {
                        return (InetSocketAddress) socket.getLocalSocketAddress();
                    }
                };
                Pack connPack = connPackFactory.apply(cxt);//gongleradd2021年11月10日15:10:51
                if (connPack != null) {
                    this.send(connPack);
                }
                this.lastRecvTask = new ReceiveTask(() -> recvFactory.get().accept(this.cxt, this.in, this::send));
                this.recvConsumer.accept(lastRecvTask);
                //System.out.println("connected.");
            } else {
                send(heartPackFactory.apply(cxt));
                this.heartHandler.accept(this.cxt, this::send);
            }

        }), 1000, 10000);

    }

    public void heartEvent(ExceptionBiConsumer<ITcpClientContext, IClientSender<Pack>> heartHandler) {
        this.heartHandler = heartHandler;
    }

    /**
     * @param pack pack
     */
    public void send(Pack pack) {
        this.sendConsumer.accept(() -> {
            try {
                byte[] bytes = packToBytes.apply(pack);
                this.out.write(bytes);
                this.out.flush();
                //System.out.println("sent:" + Util.BytesToHex(bytes) + ", " + socket);
            } catch (IOException e) {
                if (this.lastRecvTask != null) {
                    this.lastRecvTask.stop();
                }
                GonglerUtil.Close(this.out);
                GonglerUtil.Close(this.in);
                GonglerUtil.Close(socket);
                socket = null;
                //System.out.println("disconnected.");
            }
        });
    }

}
