/*
 */
package cn.gongler.util.math;

import java.lang.annotation.Native;
import java.math.BigInteger;

/**
 * @author gongler
 */
class UnsignedLong extends Number implements Comparable<UnsignedLong> {

    private static final long serialVersionUID = -1L;
    /**
     * A constant holding the minimum value a {@code long} can have,
     * -2<sup>63</sup>.
     */
    @Native
    public static final long MIN_VALUE = 0x8000000000000000L;

    /**
     * A constant holding the maximum value a {@code long} can have,
     * 2<sup>63</sup>-1.
     */
    @Native
    public static final long MAX_VALUE = 0x7fffffffffffffffL;

    /**
     * The {@code Class} instance representing the primitive type {@code long}.
     *
     * @since JDK1.1
     */
    @SuppressWarnings("unchecked")
    public static final Class<UnsignedLong> TYPE = UnsignedLong.class;//(Class<Long>) Class.getPrimitiveClass("long");

    /**
     * Return a BigInteger equal to the unsigned value of the argument.
     */
    private static BigInteger toUnsignedBigInteger(long i) {
        if (i >= 0L) {
            return BigInteger.valueOf(i);
        } else {
            int upper = (int) (i >>> 32);
            int lower = (int) i;

            // return (upper << 32) + lower
            return (BigInteger.valueOf(Integer.toUnsignedLong(upper))).shiftLeft(32).
                    add(BigInteger.valueOf(Integer.toUnsignedLong(lower)));
        }
    }

    public static String toHexString(long i) {
        return Long.toUnsignedString(i, 16);
    }

    public static String toOctalString(long i) {
        return Long.toOctalString(i);
    }

    public static String toBinaryString(long i) {
        return Long.toBinaryString(i);
    }

    public static String toString(long i) {
        return Long.toUnsignedString(i);
    }

    /**
     * Parses the string argument as a signed {@code long} in the radix
     * specified by the second argument. The characters in the string must all
     * be digits of the specified radix (as determined by whether
     * {@link java.lang.Character#digit(char, int)} returns a nonnegative
     * value), except that the first character may be an ASCII minus sign
     * {@code '-'} ({@code '\u005Cu002D'}) to indicate a negative value or an
     * ASCII plus sign {@code '+'}
     * ({@code '\u005Cu002B'}) to indicate a positive value. The resulting
     * {@code long} value is returned.
     *
     * <p>
     * Note that neither the character {@code L} ({@code '\u005Cu004C'}) nor
     * {@code l} ({@code '\u005Cu006C'}) is permitted to appear at the end of
     * the string as a type indicator, as would be permitted in Java programming
     * language source code - except that either {@code L} or {@code l} may
     * appear as a digit for a radix greater than or equal to 22.
     *
     * <p>
     * An exception of type {@code NumberFormatException} is thrown if any of
     * the following situations occurs:
     * <ul>
     *
     * <li>The first argument is {@code null} or is a string of length zero.
     *
     * <li>The {@code radix} is either smaller than {@link
     * java.lang.Character#MIN_RADIX} or larger than {@link
     * java.lang.Character#MAX_RADIX}.
     *
     * <li>Any character of the string is not a digit of the specified radix,
     * except that the first character may be a minus sign
     * {@code '-'} ({@code '\u005Cu002d'}) or plus sign {@code
     * '+'} ({@code '\u005Cu002B'}) provided that the string is longer than
     * length 1.
     *
     * <li>The value represented by the string is not a value of type
     * {@code long}.
     * </ul>
     *
     * <p>
     * Examples:
     * <blockquote><pre>
     * parse("0", 10) returns 0L
     * parse("473", 10) returns 473L
     * parse("+42", 10) returns 42L
     * parse("-0", 10) returns 0L
     * parse("-FF", 16) returns -255L
     * parse("1100110", 2) returns 102L
     * parse("99", 8) throws a NumberFormatException
     * parse("Hazelnut", 10) throws a NumberFormatException
     * parse("Hazelnut", 36) returns 1356099454469L
     * </pre></blockquote>
     *
     * @param s     the {@code String} containing the {@code long} representation to
     *              be parsed.
     * @param radix the radix to be used while parsing {@code s}.
     * @return the {@code long} represented by the string argument in the
     * specified radix.
     * @throws NumberFormatException if the string does not contain a parsable
     *                               {@code long}.
     */
    public static long parse(String s, int radix) throws NumberFormatException {
        return Long.parseUnsignedLong(s, radix);
    }

    /**
     * Parses the string argument as a signed decimal {@code long}. The
     * characters in the string must all be decimal digits, except that the
     * first character may be an ASCII minus sign {@code '-'}
     * ({@code \u005Cu002D'}) to indicate a negative value or an ASCII plus sign
     * {@code '+'} ({@code '\u005Cu002B'}) to indicate a positive value. The
     * resulting {@code long} value is returned, exactly as if the argument and
     * the radix {@code 10} were given as arguments to the {@link
     * #parse(java.lang.String, int)} method.
     *
     * <p>
     * Note that neither the character {@code L} ({@code '\u005Cu004C'}) nor
     * {@code l} ({@code '\u005Cu006C'}) is permitted to appear at the end of
     * the string as a type indicator, as would be permitted in Java programming
     * language source code.
     *
     * @param s a {@code String} containing the {@code long} representation to
     *          be parsed
     * @return the {@code long} represented by the argument in decimal.
     * @throws NumberFormatException if the string does not contain a parsable
     *                               {@code long}.
     */
    public static long parse(String s) throws NumberFormatException {
        return UnsignedLong.parse(s, 10);
    }

    public static long parseUnsignedLong(String s) throws NumberFormatException {
        return Long.parseUnsignedLong(s);
    }

    /**
     * Returns a {@code Long} object holding the value extracted from the
     * specified {@code String} when parsed with the radix given by the second
     * argument. The first argument is interpreted as representing a signed
     * {@code long} in the radix specified by the second argument, exactly as if
     * the arguments were given to the {@link
     * #parse(java.lang.String, int)} method. The result is a {@code Long}
     * object that represents the {@code long} value specified by the string.
     *
     * <p>
     * In other words, this method returns a {@code Long} object equal to the
     * value of:
     *
     * <blockquote> {@code new Long(Long.parse(s, radix))}
     * </blockquote>
     *
     * @param s     the string to be parsed
     * @param radix the radix to be used in interpreting {@code s}
     * @return a {@code Long} object holding the value represented by the string
     * argument in the specified radix.
     * @throws NumberFormatException If the {@code String} does not contain a
     *                               parsable {@code long}.
     */
    public static Long valueOf(String s, int radix) throws NumberFormatException {
        return Long.valueOf(UnsignedLong.parse(s, radix));
    }

    /**
     * Returns a {@code Long} object holding the value of the specified
     * {@code String}. The argument is interpreted as representing a signed
     * decimal {@code long}, exactly as if the argument were given to the {@link
     * #parse(java.lang.String)} method. The result is a {@code Long} object
     * that represents the integer value specified by the string.
     *
     * <p>
     * In other words, this method returns a {@code Long} object equal to the
     * value of:
     *
     * <blockquote> {@code new Long(Long.parse(s))}
     * </blockquote>
     *
     * @param s the string to be parsed.
     * @return a {@code Long} object holding the value represented by the string
     * argument.
     * @throws NumberFormatException If the string cannot be parsed as a
     *                               {@code long}.
     */
    public static Long valueOf(String s) throws NumberFormatException {
        return Long.valueOf(UnsignedLong.parse(s, 10));
    }

    private static class LongCache {

        private LongCache() {
        }

        static final Long[] cache = new Long[-(-128) + 127 + 1];

        static {
            for (int i = 0; i < cache.length; i++) {
                cache[i] = new Long(i - 128);
            }
        }
    }


    public static Long valueOf(long l) {
        final int offset = 128;
        if (l >= -128 && l <= 127) { // will cache
            return LongCache.cache[(int) l + offset];
        }
        return new Long(l);
    }

    /**
     * Decodes a {@code String} into a {@code Long}. Accepts decimal,
     * hexadecimal, and octal numbers given by the following grammar:
     *
     * <blockquote>
     * <dl>
     * <dt><i>DecodableString:</i>
     * <dd><i>Sign<sub>opt</sub> DecimalNumeral</i>
     * <dd><i>Sign<sub>opt</sub></i> {@code 0x} <i>HexDigits</i>
     * <dd><i>Sign<sub>opt</sub></i> {@code 0X} <i>HexDigits</i>
     * <dd><i>Sign<sub>opt</sub></i> {@code #} <i>HexDigits</i>
     * <dd><i>Sign<sub>opt</sub></i> {@code 0} <i>OctalDigits</i>
     *
     * <dt><i>Sign:</i>
     * <dd>{@code -}
     * <dd>{@code +}
     * </dl>
     * </blockquote>
     *
     * <i>DecimalNumeral</i>, <i>HexDigits</i>, and <i>OctalDigits</i>
     * are as defined in section 3.10.1 of
     * <cite>The Java&trade; Language Specification</cite>, except that
     * underscores are not accepted between digits.
     *
     * <p>
     * The sequence of characters following an optional sign and/or radix
     * specifier ("{@code 0x}", "{@code 0X}", "{@code #}", or leading zero) is
     * parsed as by the {@code
     * Long.parse} method with the indicated radix (10, 16, or 8). This sequence
     * of characters must represent a positive value or a
     * {@link NumberFormatException} will be thrown. The result is negated if
     * first character of the specified {@code String} is the minus sign. No
     * whitespace characters are permitted in the {@code String}.
     *
     * @param nm the {@code String} to decode.
     * @return a {@code Long} object holding the {@code long} value represented
     * by {@code nm}
     * @throws NumberFormatException if the {@code String} does not contain a
     *                               parsable {@code long}.
     * @see java.lang.Long#parseLong(String, int)
     * @since 1.2
     */
    public static Long decode(String nm) throws NumberFormatException {
        int radix = 10;
        int index = 0;
        boolean negative = false;
        Long result;

        if (nm.length() == 0) {
            throw new NumberFormatException("Zero length string");
        }
        char firstChar = nm.charAt(0);
        // Handle sign, if present
        if (firstChar == '-') {
            negative = true;
            index++;
        } else if (firstChar == '+') {
            index++;
        }

        // Handle radix specifier, if present
        if (nm.startsWith("0x", index) || nm.startsWith("0X", index)) {
            index += 2;
            radix = 16;
        } else if (nm.startsWith("#", index)) {
            index++;
            radix = 16;
        } else if (nm.startsWith("0", index) && nm.length() > 1 + index) {
            index++;
            radix = 8;
        }

        if (nm.startsWith("-", index) || nm.startsWith("+", index)) {
            throw new NumberFormatException("Sign character in wrong position");
        }

        try {
            result = Long.valueOf(nm.substring(index), radix);
            result = negative ? Long.valueOf(-result.longValue()) : result;
        } catch (NumberFormatException e) {
            // If number is Long.MIN_VALUE, we'll end up here. The next line
            // handles this case, and causes any genuine format error to be
            // rethrown.
            String constant = negative ? ("-" + nm.substring(index))
                    : nm.substring(index);
            result = Long.valueOf(constant, radix);
        }
        return result;
    }

    /**
     * The value of the {@code Long}.
     *
     * @serial
     */
    private final long value;

    /**
     * Constructs a newly allocated {@code Long} object that represents the
     * specified {@code long} argument.
     *
     * @param value the value to be represented by the {@code Long} object.
     */
    public UnsignedLong(long value) {
        this.value = value;
    }

    /**
     * Returns the value of this {@code Long} as a {@code byte} after a
     * narrowing primitive conversion.
     *
     * @jls 5.1.3 Narrowing Primitive Conversions
     */
    @Override
    public byte byteValue() {
        return (byte) value;
    }

    /**
     * Returns the value of this {@code Long} as a {@code short} after a
     * narrowing primitive conversion.
     *
     * @jls 5.1.3 Narrowing Primitive Conversions
     */
    @Override
    public short shortValue() {
        return (short) value;
    }

    /**
     * Returns the value of this {@code Long} as an {@code int} after a
     * narrowing primitive conversion.
     *
     * @jls 5.1.3 Narrowing Primitive Conversions
     */
    @Override
    public int intValue() {
        return (int) value;
    }

    /**
     * Returns the value of this {@code Long} as a {@code long} value.
     */
    @Override
    public long longValue() {
        return value;
    }

    /**
     * Returns the value of this {@code Long} as a {@code float} after a
     * widening primitive conversion.
     *
     * @jls 5.1.2 Widening Primitive Conversions
     */
    @Override
    public float floatValue() {
        return (float) value;
    }

    /**
     * Returns the value of this {@code Long} as a {@code double} after a
     * widening primitive conversion.
     *
     * @jls 5.1.2 Widening Primitive Conversions
     */
    @Override
    public double doubleValue() {
        return (double) value;
    }

    /**
     * Returns a {@code String} object representing this {@code Long}'s value.
     * The value is converted to signed decimal representation and returned as a
     * string, exactly as if the {@code long} value were given as an argument to
     * the {@link java.lang.Long#toString(long)} method.
     *
     * @return a string representation of the value of this object in
     * base&nbsp;10.
     */
    @Override
    public String toString() {
        return toString(value);
    }

    /**
     * Returns a hash code for this {@code Long}. The result is the exclusive OR
     * of the two halves of the primitive {@code long} value held by this
     * {@code Long} object. That is, the hashcode is the value of the
     * expression:
     *
     * <blockquote> {@code (int)(this.longValue()^(this.longValue()>>>32))}
     * </blockquote>
     *
     * @return a hash code value for this object.
     */
    @Override
    public int hashCode() {
        return Long.hashCode(value);
    }

    /**
     * Returns a hash code for a {@code long} value; compatible with
     * {@code Long.hashCode()}.
     *
     * @param value the value to hash
     * @return a hash code value for a {@code long} value.
     * @since 1.8
     */
    public static int hashCode(long value) {
        return (int) (value ^ (value >>> 32));
    }

    /**
     * Compares this object to the specified object. The result is {@code true}
     * if and only if the argument is not {@code null} and is a {@code Long}
     * object that contains the same {@code long} value as this object.
     *
     * @param obj the object to compare with.
     * @return {@code true} if the objects are the same; {@code false}
     * otherwise.
     */
    @Override
    public boolean equals(Object obj) {
        if (obj instanceof UnsignedLong) {
            return value == ((UnsignedLong) obj).value;
        }
        return false;
    }

    /**
     * Returns the {@code long} value of the system property with the specified
     * name. The first argument is treated as the name of a system property.
     * System properties are accessible through the
     * {@link java.lang.System#getProperty(java.lang.String)} method. The string
     * value of this property is then interpreted as a {@code long} value, as
     * per the {@link Long#decode decode} method, and a {@code Long} object
     * representing this value is returned; in summary:
     *
     * <ul>
     * <li>If the property value begins with the two ASCII characters {@code 0x}
     * or the ASCII character {@code #}, not followed by a minus sign, then the
     * rest of it is parsed as a hexadecimal integer exactly as for the method
     * {@link #valueOf(java.lang.String, int)} with radix 16.
     * <li>If the property value begins with the ASCII character {@code 0}
     * followed by another character, it is parsed as an octal integer exactly
     * as by the method {@link
     * #valueOf(java.lang.String, int)} with radix 8.
     * <li>Otherwise the property value is parsed as a decimal integer exactly
     * as by the method {@link #valueOf(java.lang.String, int)} with radix 10.
     * </ul>
     *
     * <p>
     * Note that, in every case, neither {@code L} ({@code '\u005Cu004C'}) nor
     * {@code l} ({@code '\u005Cu006C'}) is permitted to appear at the end of
     * the property value as a type indicator, as would be permitted in Java
     * programming language source code.
     *
     * <p>
     * The second argument is the default value. The default value is returned
     * if there is no property of the specified name, if the property does not
     * have the correct numeric format, or if the specified name is empty or
     * {@code null}.
     *
     * @param nm  property name.
     * @param val default value.
     * @return the {@code Long} value of the property.
     * @throws SecurityException for the same reasons as
     *                           {@link System#getProperty(String) System.getProperty}
     * @see System#getProperty(java.lang.String)
     * @see System#getProperty(java.lang.String, java.lang.String)
     */
    public static Long getLong(String nm, Long val) {
        String v = null;
        try {
            v = System.getProperty(nm);
        } catch (IllegalArgumentException | NullPointerException e) {
        }
        if (v != null) {
            try {
                return Long.decode(v);
            } catch (NumberFormatException e) {
            }
        }
        return val;
    }

    /**
     * Compares two {@code Long} objects numerically.
     *
     * @param anotherLong the {@code Long} to be compared.
     * @return the value {@code 0} if this {@code Long} is equal to the argument
     * {@code Long}; a value less than {@code 0} if this {@code Long} is
     * numerically less than the argument {@code Long}; and a value greater than
     * {@code 0} if this {@code Long} is numerically greater than the argument
     * {@code Long} (signed comparison).
     * @since 1.2
     */
    @Override
    public int compareTo(UnsignedLong anotherLong) {
        return Long.compareUnsigned(this.value, anotherLong.value);
    }

    /**
     * Compares two {@code long} values numerically. The value returned is
     * identical to what would be returned by:
     * <pre>
     *    Long.valueOf(x).compareTo(Long.valueOf(y))
     * </pre>
     *
     * @param x the first {@code long} to compare
     * @param y the second {@code long} to compare
     * @return the value {@code 0} if {@code x == y}; a value less than
     * {@code 0} if {@code x < y}; and a value greater than {@code 0} if
     * {@code x > y}
     * @since 1.7
     */
    public static int compare(long x, long y) {
        return Long.compareUnsigned(x, y);
    }

    /**
     * Compares two {@code long} values numerically treating the values as
     * unsigned.
     *
     * @param x the first {@code long} to compare
     * @param y the second {@code long} to compare
     * @return the value {@code 0} if {@code x == y}; a value less than
     * {@code 0} if {@code x < y} as unsigned values; and a value greater than
     * {@code 0} if {@code x > y} as unsigned values
     * @since 1.8
     */
    public static int compareUnsigned(long x, long y) {
        return compare(x + MIN_VALUE, y + MIN_VALUE);
    }

    /**
     * Returns the unsigned quotient of dividing the first argument by the
     * second where each argument and the result is interpreted as an unsigned
     * value.
     *
     * <p>
     * Note that in two's complement arithmetic, the three other basic
     * arithmetic operations of add, subtract, and multiply are bit-wise
     * identical if the two operands are regarded as both being signed or both
     * being unsigned. Therefore separate {@code
     * addUnsigned}, etc. methods are not provided.
     *
     * @param dividend the value to be divided
     * @param divisor  the value doing the dividing
     * @return the unsigned quotient of the first argument divided by the second
     * argument
     * @see #remainderUnsigned
     * @since 1.8
     */
    public static long divideUnsigned(long dividend, long divisor) {
        if (divisor < 0L) { // signed comparison
            // Answer must be 0 or 1 depending on relative magnitude
            // of dividend and divisor.
            return (compareUnsigned(dividend, divisor)) < 0 ? 0L : 1L;
        }

        if (dividend > 0) //  Both inputs non-negative
        {
            return dividend / divisor;
        } else {
            /*
             * For simple code, leveraging BigInteger.  Longer and faster
             * code written directly in terms of operations on longs is
             * possible; see "Hacker's Delight" for divide and remainder
             * algorithms.
             */
            return toUnsignedBigInteger(dividend).
                    divide(toUnsignedBigInteger(divisor)).longValue();
        }
    }

    /**
     * Returns the unsigned remainder from dividing the first argument by the
     * second where each argument and the result is interpreted as an unsigned
     * value.
     *
     * @param dividend the value to be divided
     * @param divisor  the value doing the dividing
     * @return the unsigned remainder of the first argument divided by the
     * second argument
     * @see #divideUnsigned
     * @since 1.8
     */
    public static long remainderUnsigned(long dividend, long divisor) {
        if (dividend > 0 && divisor > 0) { // signed comparisons
            return dividend % divisor;
        } else {
            if (compareUnsigned(dividend, divisor) < 0) // Avoid explicit check for 0 divisor
            {
                return dividend;
            } else {
                return toUnsignedBigInteger(dividend).
                        remainder(toUnsignedBigInteger(divisor)).longValue();
            }
        }
    }

    // Bit Twiddling
    /**
     * The number of bits used to represent a {@code long} value in two's
     * complement binary form.
     *
     * @since 1.5
     */
    @Native
    public static final int SIZE = 64;

    /**
     * The number of bytes used to represent a {@code long} value in two's
     * complement binary form.
     *
     * @since 1.8
     */
    public static final int BYTES = SIZE / Byte.SIZE;

    /**
     * Returns a {@code long} value with at most a single one-bit, in the
     * position of the highest-order ("leftmost") one-bit in the specified
     * {@code long} value. Returns zero if the specified value has no one-bits
     * in its two's complement binary representation, that is, if it is equal to
     * zero.
     *
     * @param i the value whose highest one bit is to be computed
     * @return a {@code long} value with a single one-bit, in the position of
     * the highest-order one-bit in the specified value, or zero if the
     * specified value is itself equal to zero.
     * @since 1.5
     */
    public static long highestOneBit(long i) {
        // HD, Figure 3-1
        i |= (i >> 1);
        i |= (i >> 2);
        i |= (i >> 4);
        i |= (i >> 8);
        i |= (i >> 16);
        i |= (i >> 32);
        return i - (i >>> 1);
    }

    /**
     * Returns a {@code long} value with at most a single one-bit, in the
     * position of the lowest-order ("rightmost") one-bit in the specified
     * {@code long} value. Returns zero if the specified value has no one-bits
     * in its two's complement binary representation, that is, if it is equal to
     * zero.
     *
     * @param i the value whose lowest one bit is to be computed
     * @return a {@code long} value with a single one-bit, in the position of
     * the lowest-order one-bit in the specified value, or zero if the specified
     * value is itself equal to zero.
     * @since 1.5
     */
    public static long lowestOneBit(long i) {
        // HD, Section 2-1
        return i & -i;
    }

    /**
     * Returns the number of zero bits preceding the highest-order ("leftmost")
     * one-bit in the two's complement binary representation of the specified
     * {@code long} value. Returns 64 if the specified value has no one-bits in
     * its two's complement representation, in other words if it is equal to
     * zero.
     *
     * <p>
     * Note that this method is closely related to the logarithm base 2. For all
     * positive {@code long} values x:
     * <ul>
     * <li>floor(log<sub>2</sub>(x)) = {@code 63 - numberOfLeadingZeros(x)}
     * <li>ceil(log<sub>2</sub>(x)) = {@code 64 - numberOfLeadingZeros(x - 1)}
     * </ul>
     *
     * @param i the value whose number of leading zeros is to be computed
     * @return the number of zero bits preceding the highest-order ("leftmost")
     * one-bit in the two's complement binary representation of the specified
     * {@code long} value, or 64 if the value is equal to zero.
     * @since 1.5
     */
    public static int numberOfLeadingZeros(long i) {
        // HD, Figure 5-6
        if (i == 0) {
            return 64;
        }
        int n = 1;
        int x = (int) (i >>> 32);
        if (x == 0) {
            n += 32;
            x = (int) i;
        }
        if (x >>> 16 == 0) {
            n += 16;
            x <<= 16;
        }
        if (x >>> 24 == 0) {
            n += 8;
            x <<= 8;
        }
        if (x >>> 28 == 0) {
            n += 4;
            x <<= 4;
        }
        if (x >>> 30 == 0) {
            n += 2;
            x <<= 2;
        }
        n -= x >>> 31;
        return n;
    }

    /**
     * Returns the number of zero bits following the lowest-order ("rightmost")
     * one-bit in the two's complement binary representation of the specified
     * {@code long} value. Returns 64 if the specified value has no one-bits in
     * its two's complement representation, in other words if it is equal to
     * zero.
     *
     * @param i the value whose number of trailing zeros is to be computed
     * @return the number of zero bits following the lowest-order ("rightmost")
     * one-bit in the two's complement binary representation of the specified
     * {@code long} value, or 64 if the value is equal to zero.
     * @since 1.5
     */
    public static int numberOfTrailingZeros(long i) {
        // HD, Figure 5-14
        int x, y;
        if (i == 0) {
            return 64;
        }
        int n = 63;
        y = (int) i;
        if (y != 0) {
            n = n - 32;
            x = y;
        } else {
            x = (int) (i >>> 32);
        }
        y = x << 16;
        if (y != 0) {
            n = n - 16;
            x = y;
        }
        y = x << 8;
        if (y != 0) {
            n = n - 8;
            x = y;
        }
        y = x << 4;
        if (y != 0) {
            n = n - 4;
            x = y;
        }
        y = x << 2;
        if (y != 0) {
            n = n - 2;
            x = y;
        }
        return n - ((x << 1) >>> 31);
    }

    /**
     * Returns the number of one-bits in the two's complement binary
     * representation of the specified {@code long} value. This function is
     * sometimes referred to as the <i>population count</i>.
     *
     * @param i the value whose bits are to be counted
     * @return the number of one-bits in the two's complement binary
     * representation of the specified {@code long} value.
     * @since 1.5
     */
    public static int bitCount(long i) {
        // HD, Figure 5-14
        i = i - ((i >>> 1) & 0x5555555555555555L);
        i = (i & 0x3333333333333333L) + ((i >>> 2) & 0x3333333333333333L);
        i = (i + (i >>> 4)) & 0x0f0f0f0f0f0f0f0fL;
        i = i + (i >>> 8);
        i = i + (i >>> 16);
        i = i + (i >>> 32);
        return (int) i & 0x7f;
    }

    /**
     * Returns the value obtained by rotating the two's complement binary
     * representation of the specified {@code long} value left by the specified
     * number of bits. (Bits shifted out of the left hand, or high-order, side
     * reenter on the right, or low-order.)
     *
     * <p>
     * Note that left rotation with a negative distance is equivalent to right
     * rotation: {@code rotateLeft(val, -distance) == rotateRight(val,
     * distance)}. Note also that rotation by any multiple of 64 is a no-op, so
     * all but the last six bits of the rotation distance can be ignored, even
     * if the distance is negative: {@code rotateLeft(val,
     * distance) == rotateLeft(val, distance & 0x3F)}.
     *
     * @param i        the value whose bits are to be rotated left
     * @param distance the number of bit positions to rotate left
     * @return the value obtained by rotating the two's complement binary
     * representation of the specified {@code long} value left by the specified
     * number of bits.
     * @since 1.5
     */
    public static long rotateLeft(long i, int distance) {
        return (i << distance) | (i >>> -distance);
    }

    /**
     * Returns the value obtained by rotating the two's complement binary
     * representation of the specified {@code long} value right by the specified
     * number of bits. (Bits shifted out of the right hand, or low-order, side
     * reenter on the left, or high-order.)
     *
     * <p>
     * Note that right rotation with a negative distance is equivalent to left
     * rotation: {@code rotateRight(val, -distance) == rotateLeft(val,
     * distance)}. Note also that rotation by any multiple of 64 is a no-op, so
     * all but the last six bits of the rotation distance can be ignored, even
     * if the distance is negative: {@code rotateRight(val,
     * distance) == rotateRight(val, distance & 0x3F)}.
     *
     * @param i        the value whose bits are to be rotated right
     * @param distance the number of bit positions to rotate right
     * @return the value obtained by rotating the two's complement binary
     * representation of the specified {@code long} value right by the specified
     * number of bits.
     * @since 1.5
     */
    public static long rotateRight(long i, int distance) {
        return (i >>> distance) | (i << -distance);
    }

    /**
     * Returns the value obtained by reversing the order of the bits in the
     * two's complement binary representation of the specified {@code long}
     * value.
     *
     * @param i the value to be reversed
     * @return the value obtained by reversing order of the bits in the
     * specified {@code long} value.
     * @since 1.5
     */
    public static long reverse(long i) {
        // HD, Figure 7-1
        i = (i & 0x5555555555555555L) << 1 | (i >>> 1) & 0x5555555555555555L;
        i = (i & 0x3333333333333333L) << 2 | (i >>> 2) & 0x3333333333333333L;
        i = (i & 0x0f0f0f0f0f0f0f0fL) << 4 | (i >>> 4) & 0x0f0f0f0f0f0f0f0fL;
        i = (i & 0x00ff00ff00ff00ffL) << 8 | (i >>> 8) & 0x00ff00ff00ff00ffL;
        i = (i << 48) | ((i & 0xffff0000L) << 16)
                | ((i >>> 16) & 0xffff0000L) | (i >>> 48);
        return i;
    }

    /**
     * Returns the signum function of the specified {@code long} value. (The
     * return value is -1 if the specified value is negative; 0 if the specified
     * value is zero; and 1 if the specified value is positive.)
     *
     * @param i the value whose signum is to be computed
     * @return the signum function of the specified {@code long} value.
     * @since 1.5
     */
    public static int signum(long i) {
        // HD, Section 2-7
        return (int) ((i >> 63) | (-i >>> 63));
    }

    /**
     * Returns the value obtained by reversing the order of the bytes in the
     * two's complement representation of the specified {@code long} value.
     *
     * @param i the value whose bytes are to be reversed
     * @return the value obtained by reversing the bytes in the specified
     * {@code long} value.
     * @since 1.5
     */
    public static long reverseBytes(long i) {
        i = (i & 0x00ff00ff00ff00ffL) << 8 | (i >>> 8) & 0x00ff00ff00ff00ffL;
        return (i << 48) | ((i & 0xffff0000L) << 16)
                | ((i >>> 16) & 0xffff0000L) | (i >>> 48);
    }

    /**
     * Adds two {@code long} values together as per the + operator.
     *
     * @param a the first operand
     * @param b the second operand
     * @return the sum of {@code a} and {@code b}
     * @see java.util.function.BinaryOperator
     * @since 1.8
     */
    public static long sum(long a, long b) {
        return a + b;
    }

    /**
     * Returns the greater of two {@code long} values as if by calling
     * {@link Math#max(long, long) Math.max}.
     *
     * @param a the first operand
     * @param b the second operand
     * @return the greater of {@code a} and {@code b}
     * @see java.util.function.BinaryOperator
     * @since 1.8
     */
    public static long max(long a, long b) {
        return Math.max(a, b);
    }

    /**
     * Returns the smaller of two {@code long} values as if by calling
     * {@link Math#min(long, long) Math.min}.
     *
     * @param a the first operand
     * @param b the second operand
     * @return the smaller of {@code a} and {@code b}
     * @see java.util.function.BinaryOperator
     * @since 1.8
     */
    public static long min(long a, long b) {
        return Math.min(a, b);
    }

}
