/*
 2014.03.06 最近bytes。
 */
package cn.gongler.util.io;

import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 2014.03.07 为了缓存最近的控制台输出，创建该循环字节缓存类。用于存放最近数据。 实现了OutputStream接口。但不保证将来还会支持该特性。
 *
 * @author gongler
 */
public class RingBytesOutputStream extends FilterOutputStream {//OutputStream{

    private static final long serialVersionUID = 1L;


    private final byte[] buf;//final E[] buf;
    private final AtomicInteger writePos = new AtomicInteger(0);//20131227gongler允许多线程使用 volatile int pos = 0;
    private final int capacity;
    private final int capacityAlign;
    private final int mask;//不适合容量1

    public RingBytesOutputStream() {
        this(null);
    }

    public RingBytesOutputStream(OutputStream out) {
        this(out, 1024 * 1024);
    }

    /**
     * 使用2的幂作为容量，将更有效利用内存空间。
     *
     * @param out      outputstream
     * @param capacity 实际采用对对齐后的大小
     */
    public RingBytesOutputStream(OutputStream out, int capacity) {
        super(out == null ? NULL_OUTPUT_STREAM : out);
        this.capacity = capacity;
        this.capacityAlign = AlignCapacity(capacity);
        this.mask = capacityAlign - 1;
        buf = new byte[capacityAlign];
        Arrays.fill(buf, (byte) ' ');
    }

    /**
     * @param e a byte value
     * @throws IOException IOException
     */
    @Override
    public void write(int e) throws IOException {//如果要进一步提升性能，建议实现方法：write(byte b[], int off, int len)
        buf[wrapIndex(writePos.getAndIncrement())] = (byte) e;
        out.write(e);
    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        int pos = wrapIndex(writePos.getAndAdd(len));
        int firstCopySize = Math.min(len, buf.length - pos);
        System.arraycopy(b, off, buf, pos, firstCopySize);
        int remainSize = len - firstCopySize;
        if (remainSize > 0) {//如果需要回绕
            System.arraycopy(b, off + firstCopySize, buf, 0, Math.min(remainSize, buf.length));//瑕疵：如果len大于缓存大小时，输出非最理想部分。
        }
        out.write(b, off, len);
    }

    private int wrapIndex(int serial) {
        return mask == 0 ? 0 : serial & mask;
    }

    private int currentIndex() {
        return wrapIndex(writePos.get());
    }

    public byte[] toBytes() {
        return RingBufferToBytes(buf, currentIndex());
    }

    public byte[] toBytes(int lastLines) {
        byte[] bytes = toBytes();

        //从尾部输出指定的行数。要避免最大输出时出现半行开始，要避免头一遍覆盖前，头部输出0值。
        int line = 0;
        int pos = 0;//如果信息中没有换行符，则从头输出；如果不够指定行，则从最旧一个换行符开始输出，尽量避免半行（半行可能引发半个汉字乱码）
        for (int i = bytes.length - 1; i >= 0 && bytes[i] != 0; i--) {
            if (bytes[i] == '\n') {
                line++;
                pos = i + 1;
                if (line == lastLines) {
                    break;
                }
            }
        }
        return Arrays.copyOfRange(bytes, pos, bytes.length);
    }

    private static byte[] RingBufferToBytes(byte[] buf, int pos) {
        long from = System.nanoTime();
        byte[] tmp = (byte[]) buf.clone();//快照
        byte[] ret = new byte[tmp.length];
        System.arraycopy(tmp, pos, ret, 0, tmp.length - pos);
        System.arraycopy(tmp, 0, ret, tmp.length - pos, pos);
        long to = System.nanoTime();
        //System.out.println("" + (to - from));
        return ret;
    }

//    @Override
//    public String toString() {
//        return ElementsToString(this, 10);
//    }

    /**
     * 返回能容纳参数容量的最小2的整数幂。1=>1, 2=>2, 3,4=>4, 5~8=>8 8~16=>16 17~24=>24
     *
     * @param anyCapacity
     * @return
     */
    private static int AlignCapacity(int anyCapacity) {
        if (anyCapacity <= 1) {
            return 1;
        } else {
            return 1 << (HighBitIndex(anyCapacity - 1) + 1);
        }
    }

    private static int HighBitIndex(int val) {
        for (int i = 0; i < 32; i++) {
            final int moveBitCnt = 31 - i;
            if (((val >>> moveBitCnt) & 1) == 1) {
                return moveBitCnt;
            }
        }
        return 1;
    }

    private static final OutputStream NULL_OUTPUT_STREAM = new OutputStream() {

        @Override
        public void write(int b) throws IOException {
        }
    };

}
