package cn.gongler.util.io;

import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.function.IntConsumer;
import java.util.function.IntUnaryOperator;

public class DelegateOutputStream extends FilterOutputStream {

    public static DelegateOutputStream filter(OutputStream out, IntUnaryOperator mapper) {
        return new DelegateOutputStream(out, mapper);
    }

    public static DelegateOutputStream consumer(OutputStream out, IntConsumer consumer) {
        return new DelegateOutputStream(out, a -> {
            consumer.accept(a);
            return a;
        });
    }

    public static DelegateOutputStream consoleHex(OutputStream out) {
        return consumer(out, a -> {
            if (a >= 0) System.out.printf("%2X", a & 0xFF);
        });
    }

    public static DelegateOutputStream console(OutputStream out) {
        return consumer(out, a -> {
            if (a >= 0) System.out.printf("%c", (char) a);
        });
    }

    public static DelegateOutputStream consumer(IntConsumer consumer) {
        return new DelegateOutputStream(new OutputStream() {
            @Override
            public void write(int b) throws IOException {
            }
        }, a -> {
            consumer.accept(a);
            return a;
        });
    }

    final IntUnaryOperator mapper;


    /**
     * Creates an output stream filter built on top of the specified
     * underlying output stream.
     *
     * @param out the underlying output stream to be assigned to
     *            the field <tt>this.out</tt> for later use, or
     *            <code>null</code> if this instance is to be
     *            created without an underlying stream.
     */
    DelegateOutputStream(OutputStream out, IntUnaryOperator mapper) {
        super(out);
        this.mapper = mapper;
    }


    public void write(int b) throws IOException {
        out.write(mapper.applyAsInt(b));
    }

}
