/*
 *
 *
 *
 */
package cn.gongler.util.concurrent;

import java.util.AbstractCollection;
import java.util.Iterator;
import java.util.Objects;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * 链表。 为了保证无锁，牺牲了顺序性（总是在头部插入，而不是尾部。当然可以找到尾部，并插入）
 *
 * @param <E>
 * @author gongler
 */
class LinkedTable<E> extends AbstractCollection<E> implements Iterable<E> {

    private static final long serialVersionUID = 1500038516065571125L;//LinkedTable
    private final LinkedNode HeadNode;// = new LinkedNode<>(null);//该节点是虚拟节点,不存放内容。

    public LinkedTable(Class<E> aClass) {
        HeadNode = new LinkedNode(aClass);
    }

    public LinkedNode headNode() {//指向第一个普通节点的虚拟节点。
        return HeadNode;
    }

    public LinkedNode lastNormalNode() {//线程安全
        LinkedNode firstNormalNode = headNode().next;
        return firstNormalNode == null ? null : firstNormalNode.loopFindNextNormalNode(node -> node.next == null);
    }

    public LinkedNode addNode(E e) {//线程安全//从效率考虑，仅有限支持从头部插入新节点。也就是放弃顺序性，而是逆序。
        Objects.requireNonNull(e);
        return HeadNode.append(e);
    }

    @Override
    public boolean add(E e) {//线程安全
        addNode(e);
        return true;
    }

    public LinkedNode findNode(Predicate<E> condition) {//线程安全
        LinkedNode cur = HeadNode;
        while ((cur = cur.next) != null) {
            if (condition.test(cur.get())) {
                return cur;
            }
        }
        return null;
    }

    public LinkedNode getNode(E e) {//线程安全
        return findNode(val -> Objects.equals(e, val));
    }

    @Override
    public int size() {//线程安全
        LinkedNode cur = HeadNode;
        int cnt = 0;
        while ((cur = cur.next) != null) {
            cnt++;
        }
        return cnt;
    }

    @Override
    public boolean isEmpty() {
        return headNode().next == null;//return first().next() == first();
    }

    @Override
    public String toString() {
        return toDetailString();//"LinkedTable" + size();
    }

    public String toDetailString() {
        StringBuilder buf = new StringBuilder();
        buf.append("LinkedTable.size:").append(size());
        String itemsToString = this.stream().map(String::valueOf).collect(Collectors.joining(", ", "{", "}"));
        buf.append(itemsToString);
        return buf.toString();
    }

    @Override
    public Iterator<E> iterator() {//线程安全改造完毕 //线程不安全：其他线程可能会修改了Node.next，可能会报出空指针。
        return new Iterator<E>() {
            LinkedNode cur = LinkedTable.this.HeadNode;
            LinkedNode nextModifiedWhenCheckNext;

            @Override
            public boolean hasNext() {
                return (nextModifiedWhenCheckNext = cur.next) != null;//return cur.next != null;
            }

            @Override
            public E next() {
                cur = nextModifiedWhenCheckNext;//cur.next();
                return cur.get();//线程不安全
            }
        };
    }

    public class LinkedNode {

        private final E val;
        private LinkedNode next = null;//链尾指针为null

        public LinkedNode(E e) {
            this.val = e;
        }

        public LinkedNode(Class<E> aClass) {
            this.val = null;
        }

        public LinkedNode append(E e) {//线程安全
            LinkedNode newNext = new LinkedNode(e);
            LinkedNode oldNext = this.next;
            newNext.next = oldNext;
            this.next = newNext;
            return newNext;
        }

        public LinkedNode next() {
            return this.next;
        }

        public LinkedNode loopNext() {//线程安全//循环下一个, 调用时注意不要死循环。
            LinkedNode next2 = this.next;
            if (next2 == null) {
                next2 = ((LinkedNode) headNode()).next;
            }
            return next2;
        }

        public E get() {
            return val;
        }

        public boolean isTail() {
            return next() == null;
        }

        @Override
        public String toString() {
            return val + (next == null ? "_nextIsNull" : "") + "_tail:" + isTail();
        }

        public Iterable<LinkedNode> nextTurnAfterCurrentNode() {//线程不安全。如果需要可以改成线程安全，但是会增加限定，不能重复调用（虽然forEach语句不会发生）。
            return () -> new Iterator<LinkedNode>() {

                LinkedNode cur = LinkedNode.this;
                LinkedNode nextModifiedWhenCheckNext;
                int selfCount = 0;

                private LinkedNode _next() {
                    if (LinkedNode.this == headNode()) {
                        return cur.next;
                    } else {
                        if (cur == LinkedNode.this && selfCount == 1) {
                            return null;
                        }
                        return cur.loopNext();
                    }
                }

                @Override
                public boolean hasNext() {
                    return (nextModifiedWhenCheckNext = _next()) != null;//线程安全改造 return _next() != null;
                }

                @Override
                public LinkedNode next() {
                    LinkedNode next = nextModifiedWhenCheckNext;//线程安全改造 _next();
                    if (next == LinkedNode.this) {
                        selfCount++;
                    }
                    cur = next;
                    return next;
                }
            };
        }

        public LinkedNode loopFindNextNormalNode(Predicate<LinkedNode> test) {
            for (LinkedNode node : nextTurnAfterCurrentNode()) {
                if (test.test(node)) {
                    return node;
                }
            }
            return null;
        }

    }

//    public static void main(String[] args) {
//        LinkedTable<Long> LinkedTable = new LinkedTable<>(Long.class);
//        System.out.println("" + LinkedTable);
//        LinkedTable.addNode(123L);
//        System.out.println("" + LinkedTable);
//        LinkedTable.addNode(1234L);
//        System.out.println("" + LinkedTable);
//        LinkedTable.addNode(12345L);
//        System.out.println("" + LinkedTable);
//        LinkedTable.addNode(123456L);
//        System.out.println("" + LinkedTable);
//        //System.out.println("" + LinkedTable.first);
//
//        for (Object obj : LinkedTable.HeadNode.nextTurnAfterCurrentNode()) {
//            System.out.println("obj:" + obj);
//        }
//        for (Object obj : LinkedTable.HeadNode.next.nextTurnAfterCurrentNode()) {
//            System.out.println("obj2:" + obj);
//        }
//        for (Object obj : LinkedTable.lastNormalNode().nextTurnAfterCurrentNode()) {
//            System.out.println("obj3:" + obj);
//        }
//
//        for (Long val : LinkedTable) {
//            System.out.println("val:" + val);
//        }
//        System.out.println("LinkedTable.toDetailString()//" + LinkedTable.toDetailString());
//        System.out.println("getNode:" + LinkedTable.getNode(1234L));
//        System.out.println("" + LinkedTable.findNode(v -> v != 1234L));
//
//        LinkedTable<String> busLinkedTable = new LinkedTable<>(String.class);
//        String a = busLinkedTable.headNode().get();
//        //LinkedTable.LinkedNode<String> a = busLinkedTable.addNode("abc");
//    }
}
