package cn.gongler.util.bytes;

import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.StringJoiner;
import java.util.regex.Pattern;

import static cn.gongler.util.GonglerUtil.*;

/**
 * @author honger
 * @since 3.1
 */

public class HexUtil {

    private static final long serialVersionUID = 1L;

    private HexUtil() {
    }

    public static String StringToHex(CharSequence msg, Charset charset){
        return BytesToHex(msg.toString().getBytes(charset));
    }

    public static String HexToString(CharSequence hex, Charset charset){
        return new String(HexToBytes(hex), charset);
    }

    public static String StringToHex(CharSequence msg){
        return BytesToHex(msg.toString().getBytes(StandardCharsets.UTF_8));
    }

    public static String HexToString(CharSequence hex){
        return new String(HexToBytes(hex), StandardCharsets.UTF_8);
    }

    public static String GbkToHex(CharSequence msg){
        return BytesToHex(msg.toString().getBytes(GB18030()));
    }

    public static String HexToGbk(CharSequence hex){
        return new String(HexToBytes(hex), GB18030());
    }

    /**
     * 把byte[]转换成十六进制字符串。常用场景是日志输出。
     *
     * @param buf 字节数组
     * @return hex
     */
    public static String BytesToHex(byte[] buf) {
        if (buf == null) {
            return null;
        }
        return BytesToHex(buf, 0, buf.length);
    }

    public static String BytesToHex(byte[] buf, int start, int size) {
        StringBuilder out = new StringBuilder();
        for (int i = start; i < start + size; i++) {
            out.append(HEX_TABLES[Int(buf[i])]);//out.append(String.format("%02X", buf[i]));
        }
        return out.toString();
    }

    public static String BytesToHex(byte[] buf, CharSequence delimiter) {
        if (buf == null) return null;
        return BytesToHex(buf, 0, buf.length, delimiter);
    }

    public static String BytesToHex(byte[] buf, int start, int size, CharSequence delimiter) {
        StringJoiner out = new StringJoiner(delimiter);
        for (int i = start; i < start + size; i++) {
            out.add(HEX_TABLES[Int(buf[i])]);
        }
        return out.toString();
    }

    public static String BytesToHex(IBytesRange bytesRange) {//2022年10月7日
        return BytesToHex(bytesRange, "");
    }

    public static String BytesToHex(IBytesRange bytesRange, CharSequence delimiter) {//2022年10月7日
        return BytesToHex(bytesRange.buf(), bytesRange.from(), bytesRange.size(), delimiter);
    }


    /**
     * 输出包含字节数和HEX间空格的形式，适合人眼确认或在日志输出。
     *
     * @param buf 字节数组
     * @return hex描述
     */
    public static String BytesToMessage(byte[] buf) {
        return BytesToMessage(buf, 0, buf.length);
    }

    public static String BytesToMessage(byte[] buf, int start, int size) {
        return "(size:" + size + ")" + BytesToHex(buf, start, size, " ");
    }

    private static int _hexCharToVal(char ch) {
        if (ch >= 'a' && ch <= 'f') {
            return ch - 'a' + 10;
        } else if (ch >= 'A' && ch <= 'F') {
            return ch - 'A' + 10;
        } else if (ch >= '0' && ch <= '9') {
            return ch - '0';
        } else {
            throw new IllegalArgumentException("not hex char:" + ch);//return 0;
        }
    }

    public static int hexCode(char ch) {//2022年10月15日
        if (ch >= 'a' && ch <= 'f') {
            return ch - 'a' + 10;
        } else if (ch >= 'A' && ch <= 'F') {
            return ch - 'A' + 10;
        } else if (ch >= '0' && ch <= '9') {
            return ch - '0';
        } else if (ch == '#') {
            return -2;
        } else if (ch == '\n') {
            return -3;
        } else {
            return -1;
        }
    }

    private static final String charTable = "0123456789ABCDEF";

    /**
     * @param num 0~15的数值
     * @return HEX字符
     */
    public static char hexChar(int num) {//2022年10月21日
        return charTable.charAt(num);
    }

    /**
     * @param byteVal 字节值
     * @param isLow   0输出高4位二进制位的HEX字符；1输出低4位对应HEX字符
     * @return 对应半个字节的HEX字符
     */
    public static char hexCharInByte(int byteVal, int isLow) {//2022年10月21日
        return isLow != 0 ? hexChar(byteVal & 0x0F) : hexChar((byteVal >>> 4) & 0x0F);
    }


    /**
     * 把十六进制字符串HEX转换成byte[]
     *
     * @param hex HEX字符串
     * @return 字节数组
     */
    public static byte[] HexToBytes(CharSequence hex) {
        if (hex == null) return null;
        String hexStr = hex.toString().replaceAll("\\s+", "");//.replace(" ", "");//
        if (!isStrictHex(hexStr)) {//发现奇数字符数BUG
            throw new IllegalArgumentException("hex不合法字符或非偶数长度:" + hexStr.length() + ", " + hexStr);
        }
        int byteCnt = (hexStr.length() + 1) / 2;
        byte[] buf = new byte[byteCnt];
        for (int i = 0; i < byteCnt; i++) {
            int high = _hexCharToVal(hexStr.charAt(i * 2));
            int low = (i == byteCnt - 1 && hexStr.length() + 1 == byteCnt * 2) ? 0 : _hexCharToVal(hexStr.charAt(i * 2 + 1));
            buf[i] = (byte) ((high << 4) | (low & 0x0f));
        }
        return buf;//new HexBytes(hexString).getBytes();
    }

    private static final Pattern STRICT_HEX_PATTERN = Pattern.compile("([\\da-fA-F]{2})*");

    /**
     * @param hex 只含有成对HEX字符的字符串。可以直接转换成byte[]
     * @return 是否符合
     */
    public static boolean isStrictHex(CharSequence hex) {//2022年10月5日
        return STRICT_HEX_PATTERN.matcher(hex).matches();
    }

    private static final Pattern HEX_PATTERN = Pattern.compile("([\\da-fA-F]{2}|\\s)*");

    /**
     * @param hex 只含有空白字符与成对HEX字符的字符串。需要过滤空白字符才能转byte[]
     * @return 是否符合
     */
    public static boolean isHex(CharSequence hex) {//2022年10月5日
        return HEX_PATTERN.matcher(hex).matches();
    }

    public static String ByteToHex(int byteVal) {
        return HEX_TABLES[byteVal & 0xFF];
    }

    public static String forceToByteSize(int byteCnt, CharSequence hex) {//2022年10月11日
        if (hex == null) hex = "";
        int needByteSize = byteCnt * 2 - hex.length();
        if (needByteSize > 0) {
            return Repeat(needByteSize, "0") + hex;
        } else if (needByteSize == 0) {
            return hex.toString();
        } else {
            return hex.subSequence(0, byteCnt * 2).toString();
        }
    }

    private static final String[] HEX_TABLES = {
            "00", "01", "02", "03", "04", "05", "06", "07", "08", "09", "0A", "0B", "0C", "0D", "0E", "0F",
            "10", "11", "12", "13", "14", "15", "16", "17", "18", "19", "1A", "1B", "1C", "1D", "1E", "1F",
            "20", "21", "22", "23", "24", "25", "26", "27", "28", "29", "2A", "2B", "2C", "2D", "2E", "2F",
            "30", "31", "32", "33", "34", "35", "36", "37", "38", "39", "3A", "3B", "3C", "3D", "3E", "3F",
            "40", "41", "42", "43", "44", "45", "46", "47", "48", "49", "4A", "4B", "4C", "4D", "4E", "4F",
            "50", "51", "52", "53", "54", "55", "56", "57", "58", "59", "5A", "5B", "5C", "5D", "5E", "5F",
            "60", "61", "62", "63", "64", "65", "66", "67", "68", "69", "6A", "6B", "6C", "6D", "6E", "6F",
            "70", "71", "72", "73", "74", "75", "76", "77", "78", "79", "7A", "7B", "7C", "7D", "7E", "7F",
            "80", "81", "82", "83", "84", "85", "86", "87", "88", "89", "8A", "8B", "8C", "8D", "8E", "8F",
            "90", "91", "92", "93", "94", "95", "96", "97", "98", "99", "9A", "9B", "9C", "9D", "9E", "9F",
            "A0", "A1", "A2", "A3", "A4", "A5", "A6", "A7", "A8", "A9", "AA", "AB", "AC", "AD", "AE", "AF",
            "B0", "B1", "B2", "B3", "B4", "B5", "B6", "B7", "B8", "B9", "BA", "BB", "BC", "BD", "BE", "BF",
            "C0", "C1", "C2", "C3", "C4", "C5", "C6", "C7", "C8", "C9", "CA", "CB", "CC", "CD", "CE", "CF",
            "D0", "D1", "D2", "D3", "D4", "D5", "D6", "D7", "D8", "D9", "DA", "DB", "DC", "DD", "DE", "DF",
            "E0", "E1", "E2", "E3", "E4", "E5", "E6", "E7", "E8", "E9", "EA", "EB", "EC", "ED", "EE", "EF",
            "F0", "F1", "F2", "F3", "F4", "F5", "F6", "F7", "F8", "F9", "FA", "FB", "FC", "FD", "FE", "FF"};

    public static void main(String[] args) {
        byte[] buf = HexToBytes("00112233445566778899AABBCCddeeff");
        String hex = BytesToHex(buf);
        System.out.println(hex);
        System.out.println(BytesToHex(buf, " "));
    }
}
