/*
 * created by gongler at 2015.04.08
 *
 */
package cn.gongler.util.bytes;

import cn.gongler.util.annotation.Unstable;
import cn.gongler.util.text.StringLinker;

import java.io.InputStream;
import java.math.BigInteger;
import java.nio.ByteOrder;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import static cn.gongler.util.GonglerUtil.requireEquals;
import static cn.gongler.util.GonglerUtil.requireLessThenOrEqual;
import static cn.gongler.util.bytes.HexUtil.HexToBytes;

/**
 * 如果数据已经读完，则获取到默认字节值0。数字默认是大端字节顺序，但也支持直接指定字节顺序。
 *
 * @author gongler
 * @since 3.1
 */
public class BytesLoader {
    //TODO 增加其他输入源支持：InputStream等

    private static final long serialVersionUID = -7939956045118259253L;//BytesLoader @since 2016-08-19

    /**
     * 大端字节顺序
     *
     * @param buf 待解析字节数组
     * @return 实例
     */
    public static BytesLoader of(byte[] buf) {
        return BigEndian(buf);
    }

    public static BytesLoader LittleEndian(byte[] buf) {
        return new BytesLoader(buf).littleEndian();
    }

    public static BytesLoader BigEndian(byte[] buf) {
        return new BytesLoader(buf).bigEndian();
    }

    public static BytesLoader of(CharSequence hex) {
        return of(HexToBytes(hex));
    }

    public static BytesLoader of(byte[] buf, int from, int size) {
        return of(Arrays.copyOfRange(buf, from, from + size));
    }

    public static BytesLoader of(IntStream stream) {
        return of(BytesBuilder.of().addByteStream(stream).toBytes());
    }

    public static BytesLoader of(List<Byte> list) {
        return of(list.stream());
    }

    public static BytesLoader of(Stream<Byte> stream) {
        return of(BytesBuilder.of().addByteStream(stream).toBytes());
    }

    public static BytesLoader of(Byte[] buf) {
        return of(buf, 0, buf.length);
    }

    public static BytesLoader of(Byte[] buf, int from, int size) {
        return of(BytesBuilder.of().addBytes(buf, from, size).toBytes());
    }

    public static BytesLoader of(IBytesRange bytesRange) {
        return of(bytesRange.buf(), bytesRange.from(), bytesRange.size());
    }

    public static BytesLoader of(ToBytes toBytes) {
        byte[] bytes = toBytes.toBytes();
        return of(bytes, 0, bytes.length);
    }

    public static BytesLoader of(InputStream in) {
        return of(BytesBuilder.of().addByteStream(in).toBytes());
    }

    public static BytesLoader ofFile(Path binFile) {
        return of(BytesBuilder.of().addFile(binFile).toBytes());
    }

    public static BytesLoader ofHexFile(Path hexFile) {
        return of(BytesBuilder.of().addHexFile(hexFile).toBytes());
    }

    private boolean defaultIsBigEndian = true;
    private final byte[] buf;
    private int pos = 0;
    /**
     * 最多缓存8个bit值
     */
    private byte bitBuf = 0;
    private int bitCnt = 0;

    /**
     * @param buf 待解析字节数组
     */
    protected BytesLoader(byte[] buf) {
        this.buf = buf;
        //System.out.println("" + Arrays.toString(buf));
    }

    /**
     * 默认是大端字节顺序，本方法可以转换成默认小端顺序。
     *
     * @return this
     */
    private BytesLoader littleEndian() {
        this.defaultIsBigEndian = false;
        return this;
    }

    private BytesLoader bigEndian() {
        this.defaultIsBigEndian = true;
        return this;
    }

    public ByteOrder defaultByteOrder() {
        return defaultIsBigEndian ? ByteOrder.BIG_ENDIAN : ByteOrder.LITTLE_ENDIAN;
    }

    public BytesLoader defaultByteOrder(ByteOrder defaultByteOrder) {
        defaultIsBigEndian = ByteOrder.BIG_ENDIAN.equals(defaultByteOrder);
        return this;
    }

    /**
     * @return value
     */
    public int loadUnsignedByte() {
        return loadByte() & 0xFF;
    }

    /**
     * (大端字节顺序)
     *
     * @param bytes 字节数
     * @return 值
     */
    public Long loadLong(int bytes) {
        return loadLong(bytes, null);
    }

    public Long loadLong(int bytes, Long defaultVal) {
        if (remainBytesCount() < bytes) return defaultVal;
        return this.defaultIsBigEndian ? loadLongBigEndian(bytes) : loadLongLittleEndian(bytes);
    }

    public double loadDouble(int bytes, int scale) {//2022年10月19日
        return ((double) loadLong(bytes)) / Math.pow(10, scale);
    }

    public int loadInt(int bytes) {
        return loadLong(bytes, 0L).intValue();
    }

    public int loadInt(int bytes, int defaultVal) {
        if (remainBytesCount() < bytes) return defaultVal;
        return loadLong(bytes, 0L).intValue();
    }

    public int loadSignedInt(int bytes) {//20160822温度等。
        return loadSignedInt(bytes, 0);
    }

    /**
     * 温度等0是有效值，不适合作为默认值。故提供指定非0的默认值。
     *
     * @param bytes      字节数
     * @param defaultVal 默认值
     * @return 值
     */
    public int loadSignedInt(int bytes, int defaultVal) {//20160822温度等。
        if (remainBytesCount() < bytes) return defaultVal;
        int ret = loadInt(bytes);
        if (((ret >>> (bytes * 8 - 1)) & 1) != 0 && bytes < 4) {//20151111：尽管晦涩但逻辑没问题
            ret -= 1 << (bytes * 8);
        }
        return ret;
    }

    public Integer loadInteger(int bytes) {//2022年10月13日
        return loadInteger(bytes, null);
    }

    public Integer loadInteger(int bytes, Integer defaultVal) {//2022年10月13日
        if (remainBytesCount() < bytes) return defaultVal;
        return loadLong(bytes, 0L).intValue();
    }

    public BigInteger loadBigInteger(int bytes, BigInteger defaultVal) {//2022年10月13日
        if (remainBytesCount() < bytes) return defaultVal;
        byte[] buf = loadBytes(bytes);
        return new BigInteger(buf);
    }

    public BigInteger loadBigInteger(int bytes) {//2022年10月13日
        return loadBigInteger(bytes, null);
    }

    public int loadBcdInt(int bytes) {//20160819add
        return loadBcdInt(bytes, 0);
    }

    public int loadBcdInt(int bytes, int defaultVal) {
        if (remainBytesCount() < bytes) return defaultVal;
        return loadBcdInteger(bytes, 0);
    }

    public Integer loadBcdInteger(int bytes) {
        return loadBcdInteger(bytes, null);
    }

    public Integer loadBcdInteger(int bytes, Integer defaultVal) {
        if (remainBytesCount() < bytes) return defaultVal;
        return loadBcd(bytes, 0L).intValue();
    }

    public Long loadBcd(int bytes) {//20160819add
        return loadBcd(bytes, null);
    }

    public Long loadBcd(int bytes, Long defaultVal) {
        if (remainBytesCount() < bytes) return defaultVal;
        long ret = 0;
        for (int i = 0; i < bytes; i++) {
            int b = this.loadUnsignedByte();
            int high = (b >> 4) & 0x0F;
            int low = b & 0x0F;
            if (high > 9 || low > 9) {
                throw new NumberFormatException("BCD: 0x" + Integer.toHexString(b));
            }
            ret = ret * 100 + (high * 10 + low);
        }
        return ret;
    }

    /**
     * (大端字节顺序)
     *
     * @param bytes 字节数
     * @return 返回值
     */
    public Long loadLongBigEndian(int bytes) {
        return loadLongBigEndian(bytes, null);
    }

    public Long loadLongBigEndian(int bytes, Long defaultVal) {
        if (remainBytesCount() < bytes) return defaultVal;
        long ret = 0;
        for (int i = 0; i < bytes; i++) {
            ret <<= 8;
            ret |= loadUnsignedByte();
        }
        return ret;
    }

    /**
     * (小端字节顺序)
     *
     * @param bytes 字节数
     * @return 返回值
     */
    public Long loadLongLittleEndian(int bytes) {
        return loadLongLittleEndian(bytes, null);
    }

    public Long loadLongLittleEndian(int bytes, Long defaultVal) {
        if (remainBytesCount() < bytes) return defaultVal;
        long ret = 0;
        for (int i = 0; i < bytes; i++) {
            long currentByte = loadUnsignedByte();
            currentByte <<= i * 8L;
            ret |= currentByte;
        }
        return ret;
    }

    public Integer loadIntBigEndian(int bytes) {
        return loadIntBigEndian(bytes, null);
    }

    public Integer loadIntBigEndian(int bytes, Integer defaultVal) {
        if (remainBytesCount() < bytes) return defaultVal;
        return loadLongBigEndian(bytes).intValue();
    }

    public Integer loadIntLittleEndian(int bytes) {
        return loadIntLittleEndian(bytes, null);
    }

    public Integer loadIntLittleEndian(int bytes, Integer defaultVal) {
        if (remainBytesCount() < bytes) return defaultVal;
        return loadLongLittleEndian(bytes).intValue();
    }

    /**
     * 允许超出边界，按0对待
     *
     * @return 返回字节
     */
    public byte loadByte() {
        requireEquals(0, bitCnt);
        int now = this.pos;
        byte ret = now >= buf.length ? 0 : buf[now];
        pos++;
        return ret;
    }

    public BytesLoader skip(int len) {//20160908add
        for (int i = 0; i < len; i++) {
            loadByte();
        }
        return this;
    }

    public byte[] loadBytes(int bodyLen) {
        byte[] ret = new byte[bodyLen];
        for (int i = 0; i < bodyLen; i++) {
            ret[i] = loadByte();
        }
        return ret;
    }

    public IBytesRange loadBytesRange(int bodyLen) {
        if (remainBytesCount() > bodyLen) {
            return IBytesRange.of(buf, pos, bodyLen);
        } else {
            return IBytesRange.of(Arrays.copyOfRange(buf, pos, pos + bodyLen));
        }
    }

    public byte[] loadRemainBytes() {
        if (pos < buf.length) {
            int oldPos = pos;
            pos += buf.length - pos;
            return Arrays.copyOfRange(buf, pos, buf.length);
        } else {
            return Bytes.EMPTY_BYTES;
        }
    }

    /**
     * @param bytes 负值：绝对值是头的长度；正值：定长数据块的字节数。0：剩余
     * @return loader
     */
    public BytesLoader loadBytesLoader(int bytes) {
        int blockBytes;
        if (bytes < 0) {
            blockBytes = this.loadInt(Math.abs(bytes));
        } else if (bytes == 0) {
            blockBytes = this.remainBytesCount();
        } else {
            blockBytes = bytes;
        }
        byte[] valueBytes = this.loadBytes(blockBytes);
        return new BytesLoader(valueBytes);
    }

    public int remainBytesCount() {
        return Math.max(0, buf.length - pos);
    }

    /**
     * @return ret
     */
    public int loadBit() {
        if (bitCnt == 0) {
            bitBuf = loadByte();
            bitCnt = 8;
        }
        int ret = (bitBuf >>> (bitCnt - 1)) & 0x01;
        bitCnt--;
        return ret;
    }

    /**
     * @param bitCnt 不超过32
     * @return 数值
     */
    public int loadBitsAsInt(int bitCnt) {
        requireLessThenOrEqual(bitCnt, 32);
        int ret = 0;
        for (int i = 0; i < bitCnt; i++) {
            ret <<= 1;
            ret |= loadBit();
        }
        return ret;
    }


    /**
     * @param bitCnt 超过32时使用。
     * @return long value
     */
    public long loadBitsAsLong(int bitCnt) {//2022年10月6日
        requireLessThenOrEqual(bitCnt, 64);
        long ret = 0;
        for (int i = 0; i < bitCnt; i++) {
            ret <<= 1;
            ret |= loadBit();
        }
        return ret;
    }

    public Bits loadBits(int bitCnt) {
        requireLessThenOrEqual(bitCnt, 64);
        long ret = 0;
        for (int i = 0; i < bitCnt; i++) {
            ret <<= 1;
            ret |= loadBit();
        }
        return Bits.of(ret);
    }

    @FunctionalInterface
    public interface Loadable<T> {
        T load(BytesLoader bytesReader);
    }

    public <T> T load(Loadable<T> loadable) {
        return loadable.load(this);
    }

    @FunctionalInterface
    public interface LoadableBySize<T> {
        T load(int bytes, BytesLoader bytesReader);
    }

    public <T> T load(int bytes, LoadableBySize<T> loadable) {
        return loadable.load(bytes, this);
    }


    ////////////////////////////////////////////////////////////////////////////
    private static final Map<Class, Function<BytesLoader, ?>> map = new HashMap<>();

    @Unstable
    public static <T> void register(Class<T> aClass, Function<BytesLoader, ? super T> factory) {
        map.put(aClass, factory);
    }

    @Unstable
    public <T> T load(Class<T> aClass) {
        return (T) map.get(aClass).apply(this);
    }
    ////////////////////////////////////////////////////////////////////////////

    public int loadRemainBits() {
        return loadBitsAsInt(this.bitCnt);
    }

    public IBytesRange toBytesRange() {
        return SimpleByteArrayRange.of(() -> buf, 0, pos);
    }

    @Override
    public String toString() {
        StringLinker build = StringLinker.of(" ");
        build.merge(StringLinker.of().add(pos).add('/').add(buf.length).add(':'));
        for (int i = 0; i < buf.length; i++) {
            if (i == pos) {
                build.add(">");
            }
            build.add(HexUtil.ByteToHex(buf[i]));
        }
        return build.toString();
    }

}
