package cn.godmao.utils;

import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;
import com.esotericsoftware.kryo.serializers.DefaultSerializers;
import com.esotericsoftware.kryo.serializers.JavaSerializer;
import com.esotericsoftware.kryo.util.DefaultInstantiatorStrategy;
import com.esotericsoftware.kryo.util.Pool;
import org.objenesis.strategy.StdInstantiatorStrategy;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.URI;
import java.util.UUID;
import java.util.regex.Pattern;

public class KryoUtil {
    static final Pool<Kryo> kryoPool = new Pool<Kryo>(true, false, 1024) {
        protected Kryo create() {
            final Kryo kryo = new Kryo();
            kryo.setRegistrationRequired(false);
            kryo.setReferences(false);
            kryo.setInstantiatorStrategy(new DefaultInstantiatorStrategy(new StdInstantiatorStrategy()));
            kryo.addDefaultSerializer(Throwable.class, new JavaSerializer());
            kryo.addDefaultSerializer(UUID.class, new DefaultSerializers.UUIDSerializer());
            kryo.addDefaultSerializer(URI.class, new DefaultSerializers.URISerializer());
            kryo.addDefaultSerializer(Pattern.class, new DefaultSerializers.PatternSerializer());
            return kryo;
        }
    };

    static final Pool<Output> outputPool = new Pool<Output>(true, false, 512) {
        protected Output create() {
            return new Output(8192, -1);
        }
    };

    static final Pool<Input> inputPool = new Pool<Input>(true, false, 512) {
        protected Input create() {
            return new Input(8192);
        }
    };


    public static <T> byte[] serialize(T obj) {
        Kryo kryo = kryoPool.obtain();
        Output output = outputPool.obtain();
        try (final ByteArrayOutputStream stream = new ByteArrayOutputStream()) {
            output.setOutputStream(stream);
            kryo.writeClassAndObject(output, obj);
            output.flush();
            return stream.toByteArray();
        } catch (IOException e) {
//            e.printStackTrace();
            return null;
        } finally {
            output.reset();
            kryoPool.free(kryo);
            outputPool.free(output);
        }
    }

    public static Object deserialize(byte[] data) {
        Kryo kryo = kryoPool.obtain();
        Input input = inputPool.obtain();
        try {
            input.setBuffer(data);
            return kryo.readClassAndObject(input);
        } catch (Exception e) {
//            e.printStackTrace();
            return null;
        } finally {
            input.reset();
            kryoPool.free(kryo);
            inputPool.free(input);
        }
    }
}
