package cn.godmao.utils;


import org.apache.commons.lang3.time.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.text.ParseException;
import java.time.*;
import java.time.format.DateTimeFormatter;
import java.util.*;

public class DateUtil extends cn.hutool.core.date.DateUtil {
    private static final Logger log = LoggerFactory.getLogger(DateUtil.class);

    /**
     * 默认 zoneId
     */
    private static final ZoneId     DEFAULT_ZONE_ID     = ZoneId.systemDefault();
    private static final ZoneOffset DEFAULT_ZONE_OFFSET = DEFAULT_ZONE_ID.getRules().getOffset(Instant.now());

    private static final Map<String, DateTimeFormatter> DATE_TIME_FORMATTER_MAP = new HashMap<>();

    /**
     * 时间格式
     */
    public static final String DATE_PATTERN1  = "yyyy-MM-dd";
    public static final String DATE_PATTERN2  = "yyyy-MM-dd HH:mm:ss";
    public static final String DATE_PATTERN3  = "yyyy/MM/dd HH:mm";
    public static final String DATE_PATTERN4  = "yyyyMMdd";
    public static final String DATE_PATTERN5  = "yyyy/MM/dd";
    public static final String DATE_PATTERN6  = "yyyy年MM月dd日";
    public static final String DATE_PATTERN7  = "yyyy-MM-dd HH:mm";
    public static final String DATE_PATTERN8  = "yyyy/MM/dd/HH/mm/ss";
    public static final String DATE_PATTERN9  = "yyyy年MM月dd日HH时mm分ss秒";
    public static final String DATE_PATTERN10 = "yyyy_MM_dd";
    public static final String DATE_PATTERN11 = "yyyy-MM";

    static {
        DATE_TIME_FORMATTER_MAP.put(DATE_PATTERN1, DateTimeFormatter.ofPattern(DATE_PATTERN1));
        DATE_TIME_FORMATTER_MAP.put(DATE_PATTERN2, DateTimeFormatter.ofPattern(DATE_PATTERN2));
        DATE_TIME_FORMATTER_MAP.put(DATE_PATTERN3, DateTimeFormatter.ofPattern(DATE_PATTERN3));
        DATE_TIME_FORMATTER_MAP.put(DATE_PATTERN4, DateTimeFormatter.ofPattern(DATE_PATTERN4));
        DATE_TIME_FORMATTER_MAP.put(DATE_PATTERN5, DateTimeFormatter.ofPattern(DATE_PATTERN5));
        DATE_TIME_FORMATTER_MAP.put(DATE_PATTERN6, DateTimeFormatter.ofPattern(DATE_PATTERN6));
        DATE_TIME_FORMATTER_MAP.put(DATE_PATTERN7, DateTimeFormatter.ofPattern(DATE_PATTERN7));
        DATE_TIME_FORMATTER_MAP.put(DATE_PATTERN8, DateTimeFormatter.ofPattern(DATE_PATTERN8));
        DATE_TIME_FORMATTER_MAP.put(DATE_PATTERN9, DateTimeFormatter.ofPattern(DATE_PATTERN9));
        DATE_TIME_FORMATTER_MAP.put(DATE_PATTERN10, DateTimeFormatter.ofPattern(DATE_PATTERN10));
        DATE_TIME_FORMATTER_MAP.put(DATE_PATTERN11, DateTimeFormatter.ofPattern(DATE_PATTERN11));
    }


    /**
     * 从全局缓存中拿 pattern 对应的 formatter 或者新建
     *
     * @param pattern pattern
     * @return pattern 对应的 formatter
     */
    private static DateTimeFormatter getFormatter(String pattern) {
        return DATE_TIME_FORMATTER_MAP.getOrDefault(pattern, DateTimeFormatter.ofPattern(pattern));
    }

    /**
     * 日期格式化为指定格式的字符串
     *
     * @param date    日期
     * @param pattern 格式，如：DateUtils.DATE_PATTERN
     * @return 返回指定格式字符串时间
     */
    public static String format(Date date, String pattern) {
        ObjectUtil.notNull(date, "传入的日期不可以为 [null]");
        ObjectUtil.hasText(pattern, "PATTERN: [" + pattern + "] 参数非法");
        return formatLocalDateTime(toLocalDateTime(date), pattern);
    }

    /**
     * LocalDate 类型的日期格式化为指定格式的字符串
     *
     * @param localDate LocalDate 类型的日期
     * @param pattern   格式，如：DateUtils.DATE_PATTERN
     * @return 返回指定格式字符串时间
     */
    public static String formartLocalDate(LocalDate localDate, String pattern) {
        ObjectUtil.notNull(localDate, "传入的日期不可以为 [null]");
        ObjectUtil.hasText(pattern, "PATTERN: [" + pattern + "] 参数非法");
        return formatLocalDateTime(toLocalDateTime(localDate), pattern);
    }

    /**
     * LocalDateTime 类型的时间格式化为指定格式的字符串
     *
     * @param localDateTime LocalDateTime 类型的时间
     * @param pattern       格式，如 DateUtils.DATE_PATTERN
     * @return 指定格式字符串时间
     */
    public static String formatLocalDateTime(LocalDateTime localDateTime, String pattern) {
        ObjectUtil.notNull(localDateTime, "传入的日期不可以为 [null]");
        ObjectUtil.hasText(pattern, "PATTERN: [" + pattern + "] 参数非法");
        return localDateTime.format(getFormatter(pattern));
    }

    /**
     * 字符串转成 Date 类型
     *
     * @param str     日期字符串
     * @param pattern 日期的格式：如：DateUtils.DATE_PATTERN
     * @return Date 类型的时间
     */
    public static Date toDate(String str, String pattern) {
        ObjectUtil.hasText(str, "STR: [" + str + "] 参数非法");
        ObjectUtil.hasText(pattern, "PATTERN: [" + pattern + "] 参数非法");
        return toDate(LocalDateTime.parse(str, getFormatter(pattern)));
    }

    /**
     * LocalDate 转成 Date
     *
     * @param localDate LocalDate 类型日期
     * @return Date 类型的日期
     */
    public static Date toDate(LocalDate localDate) {
        ObjectUtil.notNull(localDate, "传入的日期不可以为 [null]");
        return Date.from(localDate.atStartOfDay(DEFAULT_ZONE_ID).toInstant());
    }

    /**
     * LocalDateTime 转成 Date
     *
     * @param localDateTime LocalDateTime 类型时间
     * @return Date 类型的时间
     */
    public static Date toDate(LocalDateTime localDateTime) {
        ObjectUtil.notNull(localDateTime, "传入的日期不可以为 [null]");
        return Date.from(localDateTime.atZone(DEFAULT_ZONE_ID).toInstant());
    }

    /**
     * 字符串转成 LocalDate 类型的日期 默认 str 形如 "yyyy-MM-dd"
     *
     * @param str 字符串日期
     * @return LocalDate 类型的日期
     */
    public static LocalDate toLocalDate(String str) {
        ObjectUtil.hasText(str, "STR: [" + str + "] 参数非法");
        return toLocalDate(str, DATE_PATTERN1);
    }

    /**
     * 字符串转成 LocalDate 类型的日期
     *
     * @param str     字符串日期
     * @param pattern 字符串格式，如 DateUtils.DATE_PATTERN
     * @return LocalDate 类型的日期
     */
    public static LocalDate toLocalDate(String str, String pattern) {
        ObjectUtil.hasText(str, "STR: [" + str + "] 参数非法");
        ObjectUtil.hasText(pattern, "PATTERN: [" + pattern + "] 参数非法");
        return LocalDate.parse(str, getFormatter(pattern));
    }

    /**
     * Date 类型日期转成 LocalDate 类型的日期
     *
     * @param date Date 类型的日期
     * @return LocalDate 类型的日期
     */
    public static LocalDate toLocalDate(Date date) {
        ObjectUtil.notNull(date, "传入的日期不可以为 [null]");
        return toZonedDateTime(date).toLocalDate();
    }

    /**
     * 字符串类型的时间转成 LocalDateTime 类型的时间，默认形如 "yyyy-MM-dd HH:mm:ss"
     *
     * @param str 字符串时间，默认形如 "yyyy-MM-dd HH:mm:ss"
     * @return LocalDateTime 类型的时间
     */
    public static LocalDateTime toLocalDateTime(String str) {
        ObjectUtil.hasText(str, "STR: [" + str + "] 参数非法");
        return toLocalDateTime(str, DATE_PATTERN2);
    }

    /**
     * 字符串类型的时间转成 LocalDateTime 类型的时间
     *
     * @param str     字符串时间
     * @param pattern 字符串时间格式
     * @return LocalDateTime 类型的时间
     */
    public static LocalDateTime toLocalDateTime(String str, String pattern) {
        ObjectUtil.hasText(str, "STR: [" + str + "] 参数非法");
        ObjectUtil.hasText(pattern, "PATTERN: [" + pattern + "] 参数非法");
        return LocalDateTime.parse(str, getFormatter(pattern));
    }


    /**
     * Date 类型的时间转成 LocalDateTime 类型的时间
     *
     * @param date Date 类型的时间
     * @return LocalDateTime 类型的时间
     */
    public static LocalDateTime toLocalDateTime(Date date) {
        ObjectUtil.notNull(date, "传入的日期不可以为 [null]");
        return toZonedDateTime(date).toLocalDateTime();
    }

    /**
     * LocalDate 时间转成 LocalDateTime 类型时间为当天开始时间
     *
     * @param localDate LocalDate 类型的时间
     * @return LocalDateTime 类型时间为当天开始时间
     */
    public static LocalDateTime toLocalDateTime(LocalDate localDate) {
        ObjectUtil.notNull(localDate, "传入的日期不可以为 [null]");
        return localDate.atStartOfDay();
    }

    /**
     * 对日期的【秒】进行加/减
     *
     * @param date    日期
     * @param seconds 秒数，负数为减
     * @return 加/减几秒后的日期
     */
    public static Date addDateSeconds(Date date, int seconds) {
        ObjectUtil.notNull(date, "传入的日期不可以为 [null]");
        return toDate(toLocalDateTime(date).plusSeconds(seconds));
    }

    /**
     * 对日期的【分钟】进行加/减
     *
     * @param date    日期
     * @param minutes 分钟数，负数为减
     * @return 加/减几分钟后的日期
     */
    public static Date addDateMinutes(Date date, int minutes) {
        ObjectUtil.notNull(date, "传入的日期不可以为 [null]");
        return toDate(toLocalDateTime(date).plusMinutes(minutes));
    }

    /**
     * 对日期的【小时】进行加/减
     *
     * @param date  日期
     * @param hours 小时数，负数为减
     * @return 加/减几小时后的日期
     */
    public static Date addDateHours(Date date, int hours) {
        ObjectUtil.notNull(date, "传入的日期不可以为 [null]");
        return toDate(toLocalDateTime(date).plusHours(hours));
    }

    /**
     * 对日期的【天】进行加/减
     *
     * @param date 日期
     * @param days 天数，负数为减
     * @return 加/减几天后的日期
     */
    public static Date addDateDays(Date date, int days) {
        ObjectUtil.notNull(date, "传入的日期不可以为 [null]");
        return toDate(toLocalDateTime(date).plusDays(days));
    }

    /**
     * 对日期的【周】进行加/减
     *
     * @param date  日期
     * @param weeks 周数，负数为减
     * @return 加/减几周后的日期
     */
    public static Date addDateWeeks(Date date, int weeks) {
        ObjectUtil.notNull(date, "传入的日期不可以为 [null]");
        return toDate(toLocalDateTime(date).plusWeeks(weeks));
    }

    /**
     * 对日期的【月】进行加/减
     *
     * @param date   日期
     * @param months 月数，负数为减
     * @return 加/减几月后的日期
     */
    public static Date addDateMonths(Date date, int months) {
        ObjectUtil.notNull(date, "传入的日期不可以为 [null]");
        return toDate(toLocalDateTime(date).plusMonths(months));
    }

    /**
     * 对日期的【年】进行加/减
     *
     * @param date  日期
     * @param years 年数，负数为减
     * @return 加/减几年后的日期
     */
    public static Date addDateYears(Date date, int years) {
        ObjectUtil.notNull(date, "传入的日期不可以为 [null]");
        return toDate(toLocalDateTime(date).plusYears(years));
    }

    /**
     * 从 Date 获取特定时区的时间
     *
     * @param date Date 类型的时间
     * @return DateUtils.DEFAULT_ZONE_ID 标定的时区时间
     */
    public static ZonedDateTime toZonedDateTime(Date date) {
        ObjectUtil.notNull(date, "传入的日期不可以为 [null]");
        return date.toInstant().atZone(DEFAULT_ZONE_ID);
    }

    /**
     * 从秒数拿到 LocalDateTime
     *
     * @param seconds 秒数
     * @return localDateTime
     */
    public static LocalDateTime fromSeconds(long seconds) {
        return LocalDateTime.ofEpochSecond(seconds, 0, DEFAULT_ZONE_OFFSET);
    }

    /**
     * 从毫秒数拿到 LocalDateTime
     *
     * @param millSeconds 毫秒数
     * @return localDateTime
     */
    public static LocalDateTime fromMillSeconds(long millSeconds) {
        Instant instant = Instant.ofEpochMilli(millSeconds);
        return LocalDateTime.ofInstant(instant, DEFAULT_ZONE_ID);
    }

    /**
     * 从 LocalDateTime 拿到秒数
     *
     * @param localDateTime localDateTime
     * @return 秒数
     */
    public static long getSeconds(LocalDateTime localDateTime) {
        ObjectUtil.notNull(localDateTime, "传入的日期不可以为 [null]");
        return getMillSeconds(localDateTime) / 1000;
    }

    /**
     * 从 LocalDateTime 拿到毫秒数
     *
     * @param localDateTime localDateTime
     * @return 毫秒数
     */
    public static long getMillSeconds(LocalDateTime localDateTime) {
        ObjectUtil.notNull(localDateTime, "传入的日期不可以为 [null]");
        return localDateTime.toInstant(DEFAULT_ZONE_OFFSET).toEpochMilli();
    }

    /**
     * 获取所有时间格式
     */
    public static String[] getPatterns() {
        try {
            if (ObjectUtil.isNotEmpty(DATE_TIME_FORMATTER_MAP)) {
                return DATE_TIME_FORMATTER_MAP.keySet().stream().toArray(String[]::new);
            }
            return null;
        } catch (Exception e) {
            log.error("error={}", e, e);
            return null;
        }
    }


    /**
     * 万能转换 "2020-11-05"...
     *
     * @param str
     * @return
     */
    public static Date parseDate(Object str) {
        try {
            if (str instanceof Long) {
                return new Date((Long) str);
            } else if (str instanceof Integer) {
                return new Date(Long.parseLong(str.toString()));
            } else if (str instanceof String) {
                ObjectUtil.hasText((String) str, "PATTERN: [" + str + "] 参数非法");
                String[] patterns = getPatterns();
                if (ObjectUtil.isEmpty(patterns)) return null;
                return DateUtils.parseDate((String) str, patterns);
            }
            return null;
        } catch (ParseException e) {
            log.error("error={}", e, e);
            return null;
        }
    }


    // -------------------------------------------------------2-------------------------------------------------------

//    /**
//     * 昨天零点
//     */
//    public static Date yesterday() {
//
//        Calendar calendar = Calendar.getInstance();
//        calendar.add(Calendar.DATE, -1);
//
//        return DateUtils.truncate(calendar.getTime(), Calendar.DATE);
//    }

//    /**
//     * 今天零点
//     */
//    public static Date today() {
//
//        Calendar calendar = Calendar.getInstance();
//        return DateUtils.truncate(calendar.getTime(), Calendar.DATE);
//    }
//
//    /**
//     * 明天整点
//     */
//    public static Date tomorrow() {
//        Calendar calendar = Calendar.getInstance();
//        calendar.add(Calendar.DATE, 1);
//
//        return DateUtils.truncate(calendar.getTime(), Calendar.DATE);
//    }

    /**
     * 上周周一零点
     */
    public static Date lastMonday() {

        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DAY_OF_MONTH, -7);
        calendar.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY);

        return DateUtils.truncate(calendar.getTime(), Calendar.DATE);
    }

    /**
     * 本周周一零点
     */
    public static Date monday() {

        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY);

        return DateUtils.truncate(calendar.getTime(), Calendar.DATE);
    }


    /**
     * 周一00:00:00:000
     * plusWeeks 0是这周 1是下周 以此类推
     *
     * @return
     */
    public static Date weekStart(Integer plusWeeks) {
        LocalDate monday = LocalDate.now().with(DayOfWeek.MONDAY);
        monday = monday.plusWeeks(plusWeeks);
        long endtime = monday.atStartOfDay().toInstant(ZoneOffset.ofHours(8)).toEpochMilli();
        Date date = new Date();
        date.setTime(endtime);
        return date;
    }

    /**
     * 周日23:59:59.999
     * plusWeeks 0是这周 1是下周 以此类推
     *
     * @return
     */
    public static Date weekEnd(Integer plusWeeks) {
        LocalDate sunday = LocalDate.now().with(DayOfWeek.SUNDAY);
        sunday = sunday.plusWeeks(plusWeeks);
        long endtime = LocalDateTime.of(sunday.getYear(), sunday.getMonth(), sunday.getDayOfMonth(), 23, 59, 59, 999999999).toInstant(ZoneOffset.ofHours(8)).toEpochMilli();
        Date date = new Date();
        date.setTime(endtime);
        return date;
    }

    /**
     * 去年1月1日零点
     */
    public static Date lastYear() {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.YEAR, -1);
        calendar.set(Calendar.DAY_OF_YEAR, Calendar.YEAR);

        return DateUtils.truncate(calendar.getTime(), Calendar.DATE);
    }

    /**
     * 今年1月1日零点
     */
    public static Date year() {
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.DAY_OF_YEAR, Calendar.YEAR);

        return DateUtils.truncate(calendar.getTime(), Calendar.DATE);
    }

    /**
     * 计算指定天数的日期
     */
    public static Date designatedDate(int day) {

        Calendar calendar = Calendar.getInstance();

        calendar.add(Calendar.DATE, day);
        return DateUtils.truncate(calendar.getTime(), Calendar.DATE);
    }

    /**
     * 计算与当前时间差值
     */
    public static int dateDelta(Date date) {

        Calendar calendar = Calendar.getInstance();
        long delta = calendar.getTime().getTime() - date.getTime();

        return (int) delta;
    }

    /**
     * 补齐时间区间
     */
    public static List<String> completionDate(LocalDateTime startDate, LocalDateTime endDate) {

        DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");
        List<String> dateList = new ArrayList<>();
        for (int i = 0; !Duration.between(startDate.plusDays(i), endDate).isNegative(); i++) {

            dateList.add(startDate.plusDays(i).format(dateTimeFormatter));
        }
        return dateList;
    }


    /**
     * 判断某一时间时间是不是今天
     *
     * @param time
     * @return
     */
    public static boolean isToday(long time) {
        //Calendar使用单例，多次调用不重复创建对象
        Calendar calendar = Calendar.getInstance();
        //使用System获取当前时间
        calendar.setTimeInMillis(System.currentTimeMillis());
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        long today = calendar.getTimeInMillis();
        if (time - today < 3600000 * 24 && time - today > 0)
            return true;
        return false;
    }

}
