package cn.godmao.utils;

import io.protostuff.LinkedBuffer;
import io.protostuff.ProtostuffIOUtil;
import io.protostuff.Schema;
import io.protostuff.runtime.DefaultIdStrategy;
import io.protostuff.runtime.IdStrategy;
import io.protostuff.runtime.RuntimeSchema;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;

public class ProtostuffUtil {
    private static final Map<Class<?>, Schema<?>> CACHE_SCHEMA;
    private static final DefaultIdStrategy STRATEGY;
    private static final Set<Class<?>> WRAPPER_SET;
    private static final Schema<SerializeDeserializeWrapper> WRAPPER_SCHEMA;

    static {
        //
        WRAPPER_SCHEMA = RuntimeSchema.createFrom(SerializeDeserializeWrapper.class);
        //
        STRATEGY = new DefaultIdStrategy(
                IdStrategy.DEFAULT_FLAGS |
                        IdStrategy.PRESERVE_NULL_ELEMENTS |
                        IdStrategy.MORPH_COLLECTION_INTERFACES |
                        IdStrategy.MORPH_MAP_INTERFACES |
                        IdStrategy.MORPH_NON_FINAL_POJOS
        );
        //
        CACHE_SCHEMA = new ConcurrentHashMap<>();
        CACHE_SCHEMA.put(boolean.class, RuntimeSchema.getSchema(Boolean.class));
        CACHE_SCHEMA.put(byte.class, RuntimeSchema.getSchema(Byte.class));
        CACHE_SCHEMA.put(char.class, RuntimeSchema.getSchema(Character.class));
        CACHE_SCHEMA.put(short.class, RuntimeSchema.getSchema(Short.class));
        CACHE_SCHEMA.put(int.class, RuntimeSchema.getSchema(Integer.class));
        CACHE_SCHEMA.put(long.class, RuntimeSchema.getSchema(Long.class));
        CACHE_SCHEMA.put(float.class, RuntimeSchema.getSchema(Float.class));
        CACHE_SCHEMA.put(double.class, RuntimeSchema.getSchema(Double.class));
//        CACHE_SCHEMA.put(void.class, RuntimeSchema.getSchema(Void.class));

        //
        WRAPPER_SET = new HashSet<>();
        WRAPPER_SET.add(List.class);
        WRAPPER_SET.add(ArrayList.class);
        WRAPPER_SET.add(CopyOnWriteArrayList.class);
        WRAPPER_SET.add(LinkedList.class);
        WRAPPER_SET.add(Stack.class);
        WRAPPER_SET.add(Vector.class);
        WRAPPER_SET.add(HashMap.class);
        WRAPPER_SET.add(TreeMap.class);
        WRAPPER_SET.add(Hashtable.class);
        WRAPPER_SET.add(SortedMap.class);
        WRAPPER_SET.add(Map.class);
        WRAPPER_SET.add(Object.class);
        WRAPPER_SET.add(LinkedHashMap.class);
        WRAPPER_SET.add(byte[].class);
    }

    @SuppressWarnings("unchecked")
    private static <T> Schema<T> getSchema(Class<T> cls) {
        Schema<T> schema = (Schema<T>) CACHE_SCHEMA.get(cls);
        if (null == schema) {
            //            schema = RuntimeSchema.getSchema(clazz, STRATEGY);
            schema = RuntimeSchema.createFrom(cls);
            CACHE_SCHEMA.put(cls, schema);
        }
        return schema;
    }

    /**
     * 序列化
     */
    @SuppressWarnings({"unchecked", "rawtypes"})
    public static <T> byte[] serialize(T obj) {
        LinkedBuffer BUFFER = LinkedBuffer.allocate();
        try {
            Class<T> clazz = (Class<T>) obj.getClass();
            Object serializeObject = obj;
            Schema schema = WRAPPER_SCHEMA;
            if (!WRAPPER_SET.contains(clazz)) {
                schema = getSchema(clazz);
            } else {
                serializeObject = SerializeDeserializeWrapper.builder(obj);
            }
            return ProtostuffIOUtil.toByteArray(serializeObject, schema, BUFFER);
        } catch (Exception e) {
            throw new IllegalStateException(e.getMessage(), e);
        } finally {
            BUFFER.clear();
        }
    }

    /**
     * 反序列化
     */
    @SuppressWarnings("unchecked")
    public static <T> T deserialize(byte[] data, Class<T> clazz) {
        T message = null;
        if (!WRAPPER_SET.contains(clazz)) {
            Schema<T> schema = getSchema(clazz);
            message = schema.newMessage();
            ProtostuffIOUtil.mergeFrom(data, message, schema);
            return message;
        } else {
            SerializeDeserializeWrapper<T> wrapper = WRAPPER_SCHEMA.newMessage();
            ProtostuffIOUtil.mergeFrom(data, wrapper, WRAPPER_SCHEMA);
            message = wrapper.getData();
        }
        return message;
    }

    @SuppressWarnings("unchecked")
    public static <T> T copy(T t) {
        return (T) deserialize(serialize(t), t.getClass());
    }

    static class SerializeDeserializeWrapper<T> {
        public T getData() {
            return data;
        }

        public void setData(T data) {
            this.data = data;
        }

        private T data;

        public static <T> SerializeDeserializeWrapper<T> builder(T data) {
            SerializeDeserializeWrapper<T> wrapper = new SerializeDeserializeWrapper<>();
            wrapper.setData(data);
            return wrapper;
        }
    }
}