package cn.godmao.utils;

import java.util.*;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;

public class CollectUtil{

    /**
     * 集合 转 Map
     * K 是 hashCode
     */
    public static <T> Map<Integer, T> toMap(Collection<T> collection) {
        if (collection == null) {
            return null;
        }
        HashMap<Integer, T> map = new HashMap<>();
        for (T next : collection) {
            map.put(next.hashCode(), next);
        }
        return map;
    }


    /**
     * 集合 转 Map
     * K 是 Function
     */
    public static <K, V> Map<K, V> toMap(Collection<V> collection, Function<V, K> function) {
        if (collection == null) {
            return null;
        }
        HashMap<K, V> map = new HashMap<>();
        for (V next : collection) {
            K apply = function.apply(next);
            map.put(apply, next);
        }
        return map;
    }


    /**
     * 集合 转 Map
     * K 是 Function
     */
    public static <K1, K2, V> Map<K1, K2> toMap(Collection<V> collection, Function<V, K1> f1, Function<V, K2> f2) {
        if (collection == null) {
            return null;
        }
        HashMap<K1, K2> map = new HashMap<>();
        for (V next : collection) {
            K1 k1 = f1.apply(next);
            K2 k2 = f2.apply(next);
            map.put(k1, k2);
        }
        return map;
    }

    /**
     * 集合 转 Map
     * K 是 Function
     */
    public static <K, V> Map<K, V> toMap(V[] collection, Function<V, K> function) {
        if (collection == null) {
            return null;
        }
        HashMap<K, V> map = new HashMap<>();
        for (V next : collection) {
            K apply = function.apply(next);
            map.put(apply, next);
        }
        return map;
    }


    /**
     * 集合分组
     */
    public static <K, V> Map<K, List<V>> toGroup(Collection<V> collection, Function<V, K> function) {
        if (collection == null) {
            return null;
        }
        HashMap<K, List<V>> map = new HashMap<>();
        for (V next : collection) {
            K apply = function.apply(next);
            List<V> list = map.getOrDefault(apply, new ArrayList<>());
            list.add(next);
            map.put(apply, list);
        }
        return map;
    }


    /**
     * 获取数组中指定字段满足条件的第一个值
     *
     * @param collection 数组
     * @param function   字段
     * @param value      值 条件
     */
    public static <T, V> T getOne(Collection<? extends T> collection, Function<T, V> function, V value) {
        for (T t : collection) {
            V apply = function.apply(t);
            if (apply.equals(value)) {
                return t;
            }
        }
        return null;
    }

    /**
     * 获取数组中指定字段满足条件的所有值
     *
     * @param collection 数组
     * @param function   字段
     * @param values     所有值 所有条件
     */
//    @SafeVarargs
    public static <T, V> List<T> getArr(List<T> collection, Function<T, V> function, V... values) {
        if (values.length == 0) return collection;

        List<T> result = new ArrayList<T>(collection.size());
        for (T t : collection) {
            V apply = function.apply(t);
            for (V value : values) {
                if (apply instanceof Collection) {
                    if (((Collection<?>) apply).contains(value)) {
                        result.add(t);
                        break;
                    }
                } else {
                    if (apply.equals(value)) {
                        result.add(t);
                        break;
                    }
                }
            }
        }
        return result;
    }


    /**
     * 获取数组中指定字段的集合
     *
     * @param collection 数组
     * @param function   指定字段
     * @return IFunction
     */
    public static <T, V> List<V> getArr(Collection<? extends T> collection, Function<T, V> function) {
        List<V> result = new ArrayList<V>(collection.size());
        for (T t : collection) {
            V apply = function.apply(t);
            result.add(apply);
        }
        return result;
    }


    /**
     * 获取数组中指定字段的集合
     *
     * @param collection 数组
     * @param function   指定字段
     * @return IFunction
     */
    public static <T, V> Set<V> getSet(Collection<? extends T> collection, Function<T, V> function) {
        Set<V> result = new HashSet<>(collection.size());
        for (T t : collection) {
            V apply = function.apply(t);
            result.add(apply);
        }
        return result;
    }


    /**
     * 获取数组中指定字段满足条件的所有值
     *
     * @param collection 数组
     * @param function   字段
     * @param values     所有值 所有条件
     */
//    @SafeVarargs
    public static <T, V> Set<? extends T> getSet(Collection<? extends T> collection, Function<T, V> function, V... values) {
        if (values.length == 0) return new HashSet<>(collection);

        Set<T> result = new HashSet<>();
        for (T t : collection) {
            V apply = function.apply(t);
            for (V value : values) {
                if (apply instanceof Collection) {
                    if (((Collection<?>) apply).contains(value)) {
                        result.add(t);
                        break;
                    }
                } else {
                    if (apply.equals(value)) {
                        result.add(t);
                        break;
                    }
                }
            }
        }
        return result;
    }


    public static <K, V> Collection<V> getValues(Map<K, V> map, Collection<? extends K> keys) {
        List<V> vs = new ArrayList<>(keys.size());
        for (K key : keys) {
            V v = map.get(key);
            if (v == null) {
                continue;
            }
            vs.add(v);
        }
        return vs;
    }

    public static <S> Set<S> setOf(S... es) {
        HashSet<S> result = new HashSet<>(es.length);
        for (S e : es) {
            result.add(e);
        }
        return result;
    }


    public static <S> List<S> listOf(S... es) {
        List<S> result = new ArrayList<>(es.length);
        for (S e : es) {
            result.add(e);
        }
        return result;
    }

    @SuppressWarnings({"unchecked", "rawtypes"})
    public static <T, V extends Comparable<? super V>> T max(Collection<? extends T> collection, Function<T, V> function) {
        if (collection == null) {
            return (T) Collections.max((Collection) collection);
        }

        Iterator<? extends T> i = collection.iterator();
        T r = i.next();
        V candidate = function.apply(r);

        while (i.hasNext()) {
            T next = i.next();
            V comp = function.apply(next);
            if (comp.compareTo(candidate) > 0) {
                candidate = comp;
                r = next;
            }

        }
        return r;
    }


    @SuppressWarnings({"unchecked", "rawtypes"})
    public static <T, V extends Comparable<? super V>> T min(Collection<? extends T> collection, Function<T, V> function) {
        if (collection == null) {
            return (T) Collections.min((Collection) collection);
        }

        Iterator<? extends T> i = collection.iterator();
        T r = i.next();
        V candidate = function.apply(r);

        while (i.hasNext()) {
            T next = i.next();
            V comp = function.apply(next);
            if (comp.compareTo(candidate) < 0) {
                candidate = comp;
                r = next;
            }
        }
        return r;
    }


    public static <C extends Collection<E>, E> C fill(C array, E obj) {
        return fill(array, obj, null);
    }

    public static <C extends Collection<E>, E> C fill(C array, E obj, Integer size) {
        final int arraySize = null == size ? array.size() : size;
        array.clear();
        for (int i = 0; i < arraySize; i++) {
            array.add(obj);
        }
        return array;
    }

    // -----------------------------------------------------------------------------
    private static final Map<Object, AtomicLong> counterGroup = new HashMap<>();

    private static AtomicLong getCounter(Object key) {
        AtomicLong atomicLong = counterGroup.get(key);
        if (null == atomicLong) {
            atomicLong = new AtomicLong(-1L);
            counterGroup.put(key, atomicLong);
        }
        return atomicLong;
    }

    public static <T> T next(Object key, List<T> array) {
        if (null == array || array.isEmpty()) {
            return null;
        }
        return array.get(Math.abs((int) (getCounter(key).incrementAndGet() % array.size())));
    }

    public static <T> T next(List<T> array, AtomicLong counter) {
        if (null == array || array.isEmpty()) {
            return null;
        }
        return array.get(Math.abs((int) (counter.incrementAndGet() % array.size())));
    }

    public static <T> T random(List<T> array) {
        if (null == array || array.isEmpty()) {
            return null;
        }
        return array.get(MathUtil.RANDOM.nextInt(array.size()));
    }

}
