package cn.godmao.shell;

import cn.hutool.core.lang.ClassScanner;
import cn.godmao.executor.DefaultExecutorService;
import cn.godmao.shell.annotation.Shell;
import cn.godmao.shell.annotation.ShellController;
import cn.godmao.utils.ClassUtil;
import cn.godmao.utils.CollectUtil;
import cn.godmao.utils.ObjectUtil;
import cn.godmao.utils.clazz.MethodAccess;
import org.fusesource.jansi.Ansi;
import org.springframework.core.annotation.AnnotationUtils;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.*;

public class JavaShell {
    private String                   format_content_heard = Ansi.ansi().eraseScreen().render("@|MAGENTA \t%s      \t%s                    \t%s \n|@").toString();
    private String                   format_content       = Ansi.ansi().eraseScreen().render("@|BLUE \t%s      \t%s                    \t%s \n|@").toString();
    private String                   format_frame         = Ansi.ansi().eraseScreen().render("@|CYAN =====================================%s============================================ \n|@").toString();
    private Class<?>                 mainClass;
    private Scanner                  scanner;
    private Formatter                formatter;
    private String                   currentPath;
    private DefaultExecutorService   executorService;
    private Map<String, ShellAccess> depository           = new HashMap<>();
    private Collection<Object>       beans;

    private static class ShellInfo {
        private Shell       shell;
        private Set<String> keys;
        private String      methodName;
        private String[]    params;
        private Class<?>[]  parameterTypes;
        private Integer     index;
        private Method      method;

        public ShellInfo(Shell shell, Method method, Integer index) {
            this.method = method;
            this.shell = shell;
            this.methodName = method.getName();
            this.params = null;
            this.parameterTypes = method.getParameterTypes();
            this.index = index;
            this.keys = CollectUtil.setOf(shell.value());
        }

        public Shell getShell() {
            return shell;
        }

        public String getMethodName() {
            return methodName;
        }

        public String[] getParams() {
            return params;
        }

        public Class<?>[] getParameterTypes() {
            return parameterTypes;
        }

        public Integer getIndex() {
            return index;
        }

        public Set<String> getKeys() {
            return keys;
        }

        public Method getMethod() {
            return method;
        }
    }

    private static class ShellAccess extends MethodAccess {
        private final Object               bean;
        private final ShellController      controller;
        private final ArrayList<ShellInfo> shells;

        public ShellAccess(Object bean) {
            super(bean);
            this.bean = bean;
            this.shells = new ArrayList<>();
            this.controller = AnnotationUtils.findAnnotation(bean.getClass(), ShellController.class);

            List<Method> methods = ClassUtil.getMethods(ClassUtil.getDeclaredMethods(bean.getClass()), Modifier.PRIVATE);
            for (Method method : methods) {
                Shell shell = AnnotationUtils.findAnnotation(method, Shell.class);
                if (null == shell) {
                    continue;
                }

                final String[] values = shell.value();
                final String methodName = method.getName();
                final Class<?>[] parameterTypes = method.getParameterTypes();
                final int index = getMethodAccess().getIndex(methodName, parameterTypes);
                ShellInfo shellInfo = new ShellInfo(shell, method, index);

                this.shells.add(shellInfo);
            }
        }

        public ShellController getController() {
            return controller;
        }

        public ArrayList<ShellInfo> getShells() {
            return shells;
        }

        public Object getBean() {
            return bean;
        }

        public Object invoke(int methodIndex, Object... args) {
            return super.invoke(methodIndex, args);
        }

    }

    public JavaShell(Class<?> mainClass) {
        this.beans = new ArrayList<>();
        this.mainClass = mainClass;
        this.currentPath = null;

        //
        ClassScanner classScanner = new ClassScanner(this.mainClass.getPackage().getName(), clazz -> Objects.nonNull(AnnotationUtils.findAnnotation(clazz, ShellController.class)));
        for (Class<?> clazz : classScanner.scan()) {
            ShellController controller = AnnotationUtils.findAnnotation(clazz, ShellController.class);
            if (null == controller) {
                continue;
            }
            beans.add(ClassUtil.getConstructorAccess(clazz).newInstance());
        }
    }


    public JavaShell(Object... beanArr) {
        this(false, beanArr);
    }


    public JavaShell(Collection<Object> beanArr) {
        this(false, beanArr);
    }

    public JavaShell(boolean start, Object... beanArr) {
        this(start, CollectUtil.listOf(beanArr));
    }


    public JavaShell(boolean start, Collection<Object> beanArr) {
        this.beans = beanArr;
        this.currentPath = null;
        if (start) {
            start();
        }
    }


    public synchronized void start() {
        String colorStr = "@|BLACK BLACK|@ @|RED RED|@ @|GREEN GREEN|@ @|YELLOW YELLOW|@ @|BLUE BLUE|@ @|MAGENTA MAGENTA|@ @|CYAN CYAN|@ @|WHITE WHITE|@ @|DEFAULT DEFAULT|@";
        System.out.println(Ansi.ansi().eraseScreen().render(colorStr));
        System.out.println();
        System.out.println();
        System.out.println();
        System.out.println();

        //
        this.formatter = new Formatter(System.out);
        this.scanner = new Scanner(System.in);
        this.executorService = new DefaultExecutorService("java-shell");
//
        //
        for (Object bean : beans) {
            ShellController controller = AnnotationUtils.findAnnotation(bean.getClass(), ShellController.class);
            if (null == controller) {
                continue;
            }
            this.depository.put(controller.value(), new ShellAccess(bean));
        }
        //
        menu(null);
    }

    private void menu(String path) {
        if (!this.depository.containsKey(path)) {
            this.currentPath = null;
            menu_main();
        } else {
            this.currentPath = path;
            menu_sub();
        }
        doRun();
    }

    /**
     * 主菜单
     */
    private void menu_main() {
        formatter.format(format_frame, null == currentPath ? "首页" : currentPath);
        formatter.format(format_content_heard, "title", "path", "method");
        for (String key : this.depository.keySet()) {
            ShellAccess shellAccess = this.depository.get(key);
            ShellController controller = shellAccess.getController();
            ArrayList<ShellInfo> shells = shellAccess.getShells();
            HashSet<String> methods = new HashSet<>();
            for (ShellInfo shellInfo : shells) {
                Shell shell = shellInfo.getShell();
                if (ObjectUtil.isEmpty(shell.title())) {
                    methods.addAll(shellInfo.getKeys());
                } else {
                    methods.add(shell.title());
                }
            }
            final String title = ObjectUtil.isEmpty(controller.title()) ? key : controller.title();
            final String path = controller.value();
            formatter.format(format_content, title, path, methods);
        }
        formatter.format(format_frame, null == currentPath ? "首页" : currentPath);
    }

    private void menu_sub() {
        formatter.format(format_frame, currentPath);
        formatter.format(format_content_heard, "title", "path", "param");
        ShellAccess shellAccess = this.depository.get(currentPath);
        for (ShellInfo shellInfo : shellAccess.getShells()) {
            final Shell shell = shellInfo.getShell();
            final String methodName = shellInfo.getMethodName();
            final String title = ObjectUtil.isNotEmpty(shell.title()) ? shell.title() : methodName;
            final Set<String> keys = shellInfo.getKeys();
            final List<String> parameterTypeNames = CollectUtil.getArr(CollectUtil.listOf(shellInfo.getParameterTypes()), Class::getSimpleName);
            formatter.format(format_content, title, keys, parameterTypeNames);
        }
        formatter.format(format_frame, currentPath);
    }

    private void doRun() {
        executorService.execute(currentPath, () -> {
            System.err.print("\n: ");
            String nextLine = scanner.nextLine();
            String[] parms = nextLine.split(" ");
            String parm = parms[0];
            if ("cd".equals(parm)) {
                menu(parms.length > 1 ? parms[1] : null);
            } else if ("home".equals(parm)) {
                menu(null);
            } else if (null == this.currentPath && this.depository.containsKey(parm)) {
                menu(parm);
            } else if (!parm.isEmpty()) {
                doShell(parm, Arrays.copyOfRange(parms, 1, parms.length));
//                doShell(parms);
            }
        });
    }

//    private boolean doShell(String... obj) {
//        String key = obj[0];
//        if (this.depository.containsKey(key)) {
//            boolean b = doShell(Arrays.copyOfRange(obj, 1, obj.length));
//            if (b) {
//                return b;
//            }
//        }
//
//        ShellAccess shellAccess = this.depository.get(currentPath);
//        if (null == shellAccess) {
//            return false;
//        }
//        boolean ss = false;
//        ArrayList<ShellInfo> shells = shellAccess.getShells();
//        for (ShellInfo shellInfo : shells) {
//            if (shellInfo.getKeys().contains(key)) {
//                invoke(shellAccess, shellInfo, Arrays.copyOfRange(obj, 1, obj.length));
//                ss = true;
//            }
//        }
//        return ss;
//    }

    private void doShell(String key, String... obj) {
        ShellAccess shellAccess = this.depository.get(currentPath);
        if (null == shellAccess) {
            return;
        }
        ArrayList<ShellInfo> shells = shellAccess.getShells();
        for (ShellInfo shellInfo : shells) {
            if (shellInfo.getKeys().contains(key)) {
                invoke(shellAccess, shellInfo, obj);
            }
        }
        doRun();
    }


    private void invoke(ShellAccess shellAccess, ShellInfo shellInfo, String... obj) {
        Object[] objects = new Object[obj.length];
        Class<?>[] parameterTypes = shellInfo.getParameterTypes();
        Object result = shellAccess.invoke(shellInfo.getIndex(), (Object[]) obj);
        if (shellInfo.getMethod().getReturnType().equals(Void.TYPE)) {

        } else {
            if (null == result) {
                result = "空";
            }
            String pri = String.format("@| %s |@", result);
            System.out.println(Ansi.ansi().eraseScreen().render(pri));
        }
    }
}
