package cn.godmao.pay;

import com.github.wxpay.sdk.WXPayConstants;
import com.github.wxpay.sdk.WXPayUtil;

import javax.net.ssl.HttpsURLConnection;
import java.io.*;
import java.net.URL;
import java.text.DecimalFormat;
import java.util.*;

/**
 * 微信支付接口封装服务
 */
public class WeChatPayService {
    private static final DecimalFormat df = new DecimalFormat("#");

    /**
     * 微信支付统一预下单接口 请查看接口规则 <a href="https://pay.weixin.qq.com/wiki/doc/api/native_sl.php?chapter=9_1">...</a>
     *
     * @param weChatPay 参数值appid 商户id等等
     * @return NATIVE支付则返回二维码扫描地址
     */
    public static Map<String, String> unifiedorder(WeChatPay weChatPay) throws Exception {
        // 创建请求参数
        SortedMap<String, String> req = new TreeMap<String, String>();
        req.put("appid", weChatPay.getAppid());    //公众号
        req.put("mch_id", weChatPay.getMch_id());  // 商户号
        req.put("out_trade_no", weChatPay.getOut_trade_no());   // 商户订单号
        req.put("total_fee", df.format(Double.parseDouble(weChatPay.getTotal_fee()) * 100));    // 标价金额(分)
        req.put("notify_url", weChatPay.getNotify_url());  // 回调地址
        req.put("trade_type", weChatPay.getTrade_type());    // 交易类型

        if (null != weChatPay.getBody() && !weChatPay.getBody().isEmpty()) {
            req.put("body", weChatPay.getBody()); // 商品描述
        }

        if (null != weChatPay.getAttach() && !weChatPay.getAttach().isEmpty()) {
            req.put("attach", weChatPay.getAttach());  // 签名
        }

        if (null != weChatPay.getSpbill_create_ip() && !weChatPay.getSpbill_create_ip().isEmpty()) {
            req.put("spbill_create_ip", weChatPay.getSpbill_create_ip());   // 终端IP
        }

        if (null == weChatPay.getNonce_str() || weChatPay.getNonce_str().isEmpty()) {
            weChatPay.setNonce_str(UUID.randomUUID().toString().replaceAll("-", "").substring(0, 32));// 32位随机字符串
        }

        req.put("nonce_str", weChatPay.getNonce_str()); // 32位随机字符串

        if (null != weChatPay.getSub_mch_id() && !weChatPay.getSub_mch_id().isEmpty()) {
            // 服务商模式
            req.put("sub_mch_id", weChatPay.getSub_mch_id());//子商户号    微信支付 分配的子商户号
        }
        if (null != weChatPay.getTime_expire() && !weChatPay.getTime_expire().isEmpty()) {
            // 设置订单结束时间
            //交易结束时间 订单失效时间，格式为yyyyMMddHHmmss，如2009年12月27日9点10分10秒表示为20091227091010。
            req.put("time_expire", weChatPay.getTime_expire());
        }
        if (null != weChatPay.getOpenid() && !weChatPay.getOpenid().isEmpty()) {
            // JSAPI支付
            req.put("openid", weChatPay.getOpenid());//用户标识 trade_type=JSAPI，此参数必传，用户在主商户appid下的唯一标识。openid和sub_openid可以选传其中之一，如果选择传sub_openid,则必须传sub_appid。下单前需要调用【网页授权获取用户信息: https://developers.weixin.qq.com/doc/offiaccount/OA_Web_Apps/Wechat_webpage_authorization.html 】接口获取到用户的Openid。
        }
        req.put(WXPayConstants.FIELD_SIGN, WXPayUtil.generateSignature(req, weChatPay.getApi_key(), WXPayConstants.SignType.MD5));  // 签名

        // 生成要发送的 xml
        String xmlBody = WXPayUtil.generateSignedXml(req, weChatPay.getApi_key());
        System.err.println(String.format("微信支付预下单请求 xml 格式:\n%s", xmlBody));

        // 发送 POST 请求 统一下单 API 并携带 xmlBody 内容,然后获得返回接口结果
        String result = httpsRequest(WXPayConstants.UNIFIEDORDER_URL, "POST", xmlBody);
        System.err.println(String.format("%s", result));
        // 将返回结果从 xml 格式转换为 map 格式
        Map<String, String> WxResultMap = WXPayUtil.xmlToMap(result);

        return WxResultMap;
    }

    /**
     * 微信支付统一预下单接口 请查看接口规则 <a href="https://pay.weixin.qq.com/wiki/doc/api/native_sl.php?chapter=9_1">...</a>
     *
     * @param weChatPay 参数值appid 商户id等等
     * @return NATIVE支付则返回二维码扫描地址
     */
    public static Map<String, Object> Unifiedorder(WeChatPay weChatPay) throws Exception {
        Map<String, Object> ResultMap = new HashMap<>();
        Map<String, String> WxResultMap = unifiedorder(weChatPay);
        // 判断通信状态 此字段是通信标识，非交易标识
        String return_code = WxResultMap.get("return_code");
        if (null != return_code && !return_code.isEmpty() && WXPayConstants.SUCCESS.equals(return_code)) {
            // 业务结果
            if (WxResultMap.get("result_code").equals(WXPayConstants.SUCCESS)) {
                // 预下单成功
                ResultMap.put("code", 0);
                ResultMap.put("msg", "预下单成功");
                //微信订单号
                ResultMap.put("out_trade_no", weChatPay.getOut_trade_no());
                switch (WxResultMap.get("trade_type")) {
                    case "NATIVE":
                        //二维码地址
                        ResultMap.put("QrCode", WxResultMap.get("code_url"));
                        break;
                    case "MWEB":
                        //二维码地址
                        ResultMap.put("mweb_url", WxResultMap.get("mweb_url"));
                        break;
                    case "JSAPI":
                        //预支付交易会话标识 微信生成的预支付回话标识，用于后续接口调用中使用，该值有效期为2小时
                        ResultMap.put("prepay_id", WxResultMap.get("prepay_id"));
                        break;
                }
            } else {
                // 下单失败
                ResultMap.put("code", 2);
                ResultMap.put("msg", WxResultMap.get("err_code_des"));
            }
        } else {
            // 通信异常
            ResultMap.put("code", 2);
            ResultMap.put("msg", WxResultMap.get("return_msg"));//当return_code为FAIL时返回信息为错误原因 ，例如 签名失败 参数格式校验错误
        }
        return ResultMap;
    }

    /**
     * 查询微信支付订单状态
     * shuqingyou 2022/9/16
     *
     * @param weChatPay 参数值appid 商户id等等
     * @return 支付状态
     */
    public static Map<String, String> queryPayStatus(WeChatPay weChatPay) throws Exception {
        //  创建请求参数
        SortedMap<String, String> req = new TreeMap<String, String>();
        req.put("appid", weChatPay.getAppid()); // 公众号ID
        req.put("mch_id", weChatPay.getMch_id());   // 商户号
        req.put("out_trade_no", weChatPay.getOut_trade_no());//商户订单号
        req.put("nonce_str", weChatPay.getNonce_str());// 随机字符串
        if (null != weChatPay.getSub_mch_id() && !weChatPay.getSub_mch_id().isEmpty()) {
            req.put("sub_mch_id", weChatPay.getSub_mch_id());//子商户号 微信支付 分配的子商户号（服务商模式的时候填入）
        }
        req.put("sign", WXPayUtil.generateSignature(req, weChatPay.getApi_key(), WXPayConstants.SignType.MD5));
        //  生成要发送的 xml
        String xmlBody = WXPayUtil.generateSignedXml(req, weChatPay.getApi_key());
        System.err.println(String.format("查询订单支付状态 xml 格式:\n%s", xmlBody));
        //  调用查询订单支付状态 API
        String result = httpsRequest(WXPayConstants.ORDERQUERY_URL, "POST", xmlBody);
        //  返回解析后的 map 数据
        Map<String, String> WxResultMap = WXPayUtil.xmlToMap(result);

        return WxResultMap;
    }

    /**
     * 查询微信支付订单状态
     * shuqingyou 2022/9/16
     *
     * @param weChatPay 参数值appid 商户id等等
     * @return 支付状态
     */
    public static Map<String, Object> QueryPayStatus(WeChatPay weChatPay) throws Exception {
        Map<String, Object> ResultMap = new HashMap<String, Object>();
        Map<String, String> WxResultMap = queryPayStatus(weChatPay);
        //  判断通信状态 此字段是通信标识，非交易标识
        String return_code = WxResultMap.get("return_code");
        if (null != return_code && !return_code.isEmpty() && WXPayConstants.SUCCESS.equals(return_code)) {

            //  业务结果
            if (WxResultMap.get("result_code").equals("SUCCESS")) {
                //  状态查询成功
                ResultMap.put("code", 0);
                ResultMap.put("daata", WxResultMap);
                switch (WxResultMap.get("trade_state")) {
                    case "SUCCESS":
                        ResultMap.put("OrderCode", WxResultMap.get("trade_state"));//订单交易状态code
                        ResultMap.put("msg", "支付成功");
                        ResultMap.put("out_trade_no", WxResultMap.get("out_trade_no"));//商户订单号
                        ResultMap.put("time_end", WxResultMap.get("time_end"));//支付完成时间
                        ResultMap.put("attach", WxResultMap.get("attach"));//下单时候传过去的附加数据
                        break;
                    case "REFUND":
                        ResultMap.put("msg", "转入退款");
                        break;
                    case "NOTPAY":
                        ResultMap.put("msg", "未支付");
                        break;
                    case "CLOSED":
                        ResultMap.put("msg", "已关闭");
                        break;
                    case "REVOKED":
                        ResultMap.put("msg", "已撤销(刷卡支付)");
                        break;
                    case "USERPAYING":
                        ResultMap.put("msg", "用户支付中");
                        break;
                    case "PAYERROR":
                        ResultMap.put("msg", "支付失败(其他原因，如银行返回失败)");
                        break;
                    case "ACCEPT":
                        ResultMap.put("msg", "已接收，等待扣款");
                        break;
                }
            } else {
                //  下单失败
                ResultMap.put("code", 2);
                ResultMap.put("msg", WxResultMap.get("err_code_des"));
            }
        } else {
            //  通信异常
            ResultMap.put("code", 2);
            ResultMap.put("msg", WxResultMap.get("return_msg"));//当return_code为FAIL时返回信息为错误原因 ，例如 签名失败 参数格式校验错误
        }
        return ResultMap;
    }

    /**
     * 发送 http 请求
     *
     * @param requestUrl    请求路径
     * @param requestMethod 请求方式(GET/POST/PUT/DELETE/...)
     * @param outputStr     请求参数体
     * @return 结果信息
     */
    private static String httpsRequest(String requestUrl, String requestMethod, String outputStr) {
        try {
            URL url = new URL(requestUrl);
            HttpsURLConnection conn = (HttpsURLConnection) url.openConnection();
            conn.setDoOutput(true);
            conn.setDoInput(true);
            conn.setUseCaches(false);
            // 设置请求方式（GET/POST）
            conn.setRequestMethod(requestMethod);
            conn.setRequestProperty("content-type", "application/x-www-form-urlencoded");
            // 当outputStr不为null时向输出流写数据
            if (null != outputStr) {
                OutputStream outputStream = conn.getOutputStream();
                // 注意编码格式
                outputStream.write(outputStr.getBytes("UTF-8"));
                outputStream.close();
            }
            // 从输入流读取返回内容
            InputStream inputStream = conn.getInputStream();
            InputStreamReader inputStreamReader = new InputStreamReader(inputStream, "utf-8");
            BufferedReader bufferedReader = new BufferedReader(inputStreamReader);
            String str = null;
            StringBuffer buffer = new StringBuffer();
            while ((str = bufferedReader.readLine()) != null) {
                buffer.append(str);
            }
            // 释放资源
            bufferedReader.close();
            inputStreamReader.close();
            inputStream.close();
            inputStream = null;
            conn.disconnect();
            return buffer.toString();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }
}