package cn.godmao.netty.client;

import cn.godmao.netty.ChannelService;
import cn.godmao.netty.NTP;
import cn.godmao.netty.client.base.ClientBaseInitializer;
import cn.godmao.netty.handler.IConnect;
import cn.godmao.netty.NettyUtil;
import cn.godmao.netty.handler.IConnectHandler;
import io.netty.bootstrap.Bootstrap;
import io.netty.buffer.PooledByteBufAllocator;
import io.netty.channel.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.net.URI;
import java.util.concurrent.TimeUnit;

public class DefaultClient implements IConnect, IConnectHandler {
    public final  Logger          log = LoggerFactory.getLogger(this.getClass());
    private final IConnectHandler connectHandler;

    //
    private Bootstrap                   boot;
    private EventLoopGroup              group;
    private ChannelService              channelService;
    private ChannelInitializer<Channel> channelInitializer;

    public DefaultClient() {
        this.connectHandler = DefaultClient.this;
    }

    public DefaultClient(IConnectHandler connectHandler) {
        this.connectHandler = connectHandler;
    }

    @Override
    public void start() {
        // 设置一个默认的加载器
        if (null == this.channelInitializer) {
            this.channelInitializer = new ClientBaseInitializer(this);
        }

        if (null == this.channelService) {
            this.channelService = new ChannelService("client-channel-group");
        }

        if (null == this.group) {
            this.group = NettyUtil.newEventLoopGroup("client-work-loop-group");
        }

        if (null == this.boot) {
            this.boot = new Bootstrap();
        }

        init();
    }

    private void init() {
        /*
          ChannelOption.SINGLE_EVENTEXECUTOR_PER_GROUP
          Netty参数，单线程执行ChannelPipeline中的事件，默认值为True。
          该值控制执行ChannelPipeline中执行ChannelHandler的线程。
          如果为True，整个pipeline由一个线程执行，这样不需要进行线程切换以及线程同步，是Netty4的推荐做法；
          如果为False，ChannelHandler中的处理过程会由Group中的不同线程执行

          //        this.boot.option(ChannelOption.SINGLE_EVENTEXECUTOR_PER_GROUP, false);



          ChannelOption.WRITE_BUFFER_WATER_MARK
          设置写的高低水位线

         */

        //
        this.boot.group(this.group);
        //
        this.boot.option(ChannelOption.SO_KEEPALIVE, true);
        this.boot.option(ChannelOption.TCP_NODELAY, true);
        this.boot.option(ChannelOption.ALLOCATOR, PooledByteBufAllocator.DEFAULT);
//        this.boot.option(ChannelOption.SO_SNDBUF, 65535);
//        this.boot.option(ChannelOption.SO_RCVBUF, 65535);
        //
//        this.boot.option(ChannelOption.WRITE_BUFFER_WATER_MARK, new WriteBufferWaterMark(64 * 1024, 128 * 1024));
        //
        this.boot.channel(NettyUtil.getSocketChannelClass(NTP.TCP));
        //
        this.boot.handler(this.channelInitializer);
    }

    public ChannelFuture connect(String uri) {
        return connect(URI.create(uri));
    }

    public ChannelFuture connect(URI uri) {
        return connect(uri.getHost(), uri.getPort());
    }

    public ChannelFuture connect(String host, int port) {
        return connect(new InetSocketAddress(host, -1 == port ? 443 : port));
    }

    public ChannelFuture connect(SocketAddress address, boolean reconnect) {
        //
        ChannelFuture channelFuture_ = this.boot.connect(address);

        //
        channelFuture_.channel().attr(ChannelService.CHANNELADDRESS).set(address.toString());

        // 断线重连
        channelFuture_.channel().closeFuture().addListener((ChannelFutureListener) channelFutureClose -> {
            if (channelFutureClose.isSuccess() && reconnect) {
                channelFutureClose.channel().eventLoop().schedule(() -> {
                    log.warn("connect: {} reconnect...", address);
                    connect(address, reconnect);
                }, 5, TimeUnit.SECONDS);
            } else {
                log.info("connect: {} success!", address);
            }
        });

        return channelFuture_;
    }

    public ChannelFuture connect(SocketAddress address) {
        return connect(address, true);
    }

    @Override
    public IConnectHandler getConnectHandler() {
        return connectHandler;
    }

    @Override
    public ChannelService getChannelService() {
        return channelService;
    }

    public void setChannelService(ChannelService channelService) {
        this.channelService = channelService;
    }

    public void setBoot(Bootstrap boot) {
        this.boot = boot;
    }

    public void setGroup(EventLoopGroup group) {
        this.group = group;
    }

    public void setChannelInitializer(ChannelInitializer<Channel> channelInitializer) {
        this.channelInitializer = channelInitializer;
    }


    @Override
    public void onOpen(ChannelHandlerContext ctx) {
        // non
    }

    @Override
    public void onClose(ChannelHandlerContext ctx) {
        // non
    }

    @Override
    public void onError(ChannelHandlerContext ctx, Throwable throwable) {
        // non
    }

    @Override
    public void onMessage(ChannelHandlerContext ctx, Object message) {
        // non
    }
}
