package cn.godmao.netty.channel;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufHolder;
import io.netty.channel.*;
import io.netty.channel.group.*;
import io.netty.util.AttributeKey;
import io.netty.util.AttributeMap;
import io.netty.util.ReferenceCountUtil;
import io.netty.util.concurrent.*;
import io.netty.util.internal.ObjectUtil;
import io.netty.util.internal.PlatformDependent;
import io.netty.util.internal.StringUtil;

import java.util.*;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicInteger;

public class DefaultChannelGroup extends AbstractSet<Channel> implements ChannelGroup {
    private static final AtomicInteger                     nextId     = new AtomicInteger();
    private final        String                            name;
    private final        EventExecutor                     executor;
    private final        ConcurrentMap<ChannelId, Channel> channels   = PlatformDependent.newConcurrentHashMap();
    private final        VoidChannelGroupFuture            voidFuture = new VoidChannelGroupFuture(this);
    private final        ChannelFutureListener             remover    = future -> remove(future.channel());

    public DefaultChannelGroup() {
        this(GlobalEventExecutor.INSTANCE);
    }

    public DefaultChannelGroup(String name) {
        this(name, GlobalEventExecutor.INSTANCE);
    }

    public DefaultChannelGroup(EventExecutor executor) {
        this("group-0x" + Integer.toHexString(nextId.incrementAndGet()), executor);
    }

    public DefaultChannelGroup(String name, EventExecutor executor) {
        ObjectUtil.checkNotNull(name, "name");
        this.name = name;
        this.executor = executor;

    }

    @Override
    public String name() {
        return name;
    }

    public Collection<Channel> getChannels() {
        return channels.values();
    }

    public ConcurrentMap<ChannelId, Channel> getChannelGroup() {
        return channels;
    }

    public Set<ChannelId> getChannelIds() {
        return channels.keySet();
    }

    @Override
    public Channel find(ChannelId id) {
        return channels.get(id);
    }

    @Override
    public boolean isEmpty() {
        return channels.isEmpty();
    }

    @Override
    public int size() {
        return channels.size();
    }

    @Override
    public boolean contains(Object o) {
        return channels.containsValue(o);
    }

    @Override
    public boolean add(Channel channel) {
        boolean added = channels.putIfAbsent(channel.id(), channel) == null;
        if (added) {
            channel.closeFuture().addListener(remover);
        }

        return added;
    }

    @Override
    public boolean remove(Object o) {
        Channel c = null;
        if (o instanceof ChannelId) {
            c = channels.remove(o);
        } else if (o instanceof Channel) {
            c = (Channel) o;
            c = channels.remove(c.id());
        }

        if (c == null) {
            return false;
        }

        c.closeFuture().removeListener(remover);
        return true;
    }

    @Override
    public void clear() {
        channels.clear();
    }

    @Override
    public Iterator<Channel> iterator() {
        return channels.values().iterator();
    }

    @Override
    public Object[] toArray() {
        return new ArrayList<>(channels.values()).toArray();
    }

    @Override
    public <T> T[] toArray(T[] a) {
        return new ArrayList<>(channels.values()).toArray(a);
    }

    @Override
    public ChannelGroupFuture close() {
        return close(ChannelMatchers.all());
    }

    @Override
    public ChannelGroupFuture disconnect() {
        return disconnect(ChannelMatchers.all());
    }

    @Override
    public ChannelGroupFuture deregister() {
        return deregister(ChannelMatchers.all());
    }

    @Override
    public ChannelGroupFuture write(Object message) {
        return write(message, ChannelMatchers.all());
    }

    // Create a safe duplicate of the message to write it to a channel but not affect other writes.
    // See https://github.com/netty/netty/issues/1461
    private static Object safeDuplicate(Object message) {
        if (message instanceof ByteBuf) {
            return ((ByteBuf) message).retainedDuplicate();
        } else if (message instanceof ByteBufHolder) {
            return ((ByteBufHolder) message).retainedDuplicate();
        } else {
            return ReferenceCountUtil.retain(message);
        }
    }


    @Override
    public ChannelGroupFuture write(Object message, ChannelMatcher matcher) {
        return write(message, matcher, false);
    }

    @Override
    public ChannelGroupFuture write(Object message, ChannelMatcher matcher, boolean voidPromise) {
        ObjectUtil.checkNotNull(message, "message");
        ObjectUtil.checkNotNull(matcher, "matcher");

        final ChannelGroupFuture future;
        if (voidPromise) {
            for (Channel c : channels.values()) {
                if (matcher.matches(c)) {
                    c.write(safeDuplicate(message), c.voidPromise());
                }
            }
            future = voidFuture;
        } else {
            Map<Channel, ChannelFuture> futures = new LinkedHashMap<Channel, ChannelFuture>(channels.size());
            for (Channel c : channels.values()) {
                if (matcher.matches(c)) {
                    futures.put(c, c.write(safeDuplicate(message)));
                }
            }
            future = new DefaultChannelGroupFuture(this, futures, executor);
        }
        ReferenceCountUtil.release(message);
        return future;
    }

    @Override
    public ChannelGroup flush() {
        return flush(ChannelMatchers.all());
    }

    @Override
    public ChannelGroupFuture flushAndWrite(Object message) {
        return writeAndFlush(message);
    }

    @Override
    public ChannelGroupFuture writeAndFlush(Object message) {
        return writeAndFlush(message, ChannelMatchers.all());
    }

    @Override
    public ChannelGroupFuture disconnect(ChannelMatcher matcher) {
        ObjectUtil.checkNotNull(matcher, "matcher");

        Map<Channel, ChannelFuture> futures =
                new LinkedHashMap<Channel, ChannelFuture>(size());

        for (Channel c : channels.values()) {
            if (matcher.matches(c)) {
                futures.put(c, c.disconnect());
            }
        }
        return new DefaultChannelGroupFuture(this, futures, executor);
    }

    @Override
    public ChannelGroupFuture close(ChannelMatcher matcher) {
        ObjectUtil.checkNotNull(matcher, "matcher");

        Map<Channel, ChannelFuture> futures =
                new LinkedHashMap<Channel, ChannelFuture>(size());

        for (Channel c : channels.values()) {
            if (matcher.matches(c)) {
                futures.put(c, c.close());
            }
        }

        return new DefaultChannelGroupFuture(this, futures, executor);
    }

    @Override
    public ChannelGroupFuture deregister(ChannelMatcher matcher) {
        ObjectUtil.checkNotNull(matcher, "matcher");

        Map<Channel, ChannelFuture> futures =
                new LinkedHashMap<Channel, ChannelFuture>(size());

        for (Channel c : channels.values()) {
            if (matcher.matches(c)) {
                futures.put(c, c.deregister());
            }
        }

        return new DefaultChannelGroupFuture(this, futures, executor);
    }

    @Override
    public ChannelGroup flush(ChannelMatcher matcher) {
        for (Channel c : channels.values()) {
            if (matcher.matches(c)) {
                c.flush();
            }
        }
        return this;
    }

    @Override
    public ChannelGroupFuture flushAndWrite(Object message, ChannelMatcher matcher) {
        return writeAndFlush(message, matcher);
    }


    @Override
    public ChannelGroupFuture writeAndFlush(Object message, ChannelMatcher matcher) {
        return writeAndFlush(message, matcher, false);
    }

    @Override
    public ChannelGroupFuture writeAndFlush(Object message, ChannelMatcher matcher, boolean voidPromise) {
        ObjectUtil.checkNotNull(message, "message");

        final ChannelGroupFuture future;
        if (voidPromise) {
            for (Channel c : channels.values()) {
                if (matcher.matches(c)) {
                    c.writeAndFlush(safeDuplicate(message), c.voidPromise());
                }
            }
            future = voidFuture;
        } else {
            Map<Channel, ChannelFuture> futures = new LinkedHashMap<Channel, ChannelFuture>(channels.size());
            for (Channel c : channels.values()) {
                if (matcher.matches(c)) {
                    futures.put(c, c.writeAndFlush(safeDuplicate(message)));
                }
            }
            future = new DefaultChannelGroupFuture(this, futures, executor);
        }
        ReferenceCountUtil.release(message);
        return future;
    }

    @Override
    public ChannelGroupFuture newCloseFuture() {
        return newCloseFuture(ChannelMatchers.all());
    }

    @Override
    public ChannelGroupFuture newCloseFuture(ChannelMatcher matcher) {
        Map<Channel, ChannelFuture> futures =
                new LinkedHashMap<Channel, ChannelFuture>(size());

        for (Channel c : channels.values()) {
            if (matcher.matches(c)) {
                futures.put(c, c.closeFuture());
            }
        }

        return new DefaultChannelGroupFuture(this, futures, executor);
    }

    @Override
    public int hashCode() {
        return System.identityHashCode(this);
    }

    @Override
    public boolean equals(Object o) {
        return this == o;
    }

    @Override
    public int compareTo(ChannelGroup o) {
        int v = name().compareTo(o.name());
        if (v != 0) {
            return v;
        }

        return System.identityHashCode(this) - System.identityHashCode(o);
    }

    @Override
    public String toString() {
        return StringUtil.simpleClassName(this) + "(name: " + name() + ", size: " + size() + ')';
    }

    // ======================================================================================================
    public static final AttributeKey<Object> KEYID = AttributeKey.valueOf("channel-key-id");


    public boolean add(Channel channel, Object id) {
        ObjectUtil.checkNotNull(id, KEYID.name());
        boolean add = add(channel);
        channel.attr(KEYID).set(id);
        return add;
    }

    public Object getId(AttributeMap attributeMap) {
        return attributeMap.attr(KEYID).get();
    }

    public ChannelFuture writeAndFlush(Object message, ChannelOutboundInvoker channelOutboundInvoker) {
        return channelOutboundInvoker.writeAndFlush(message);
    }

    public <T> ChannelGroupFuture writeAndFlush(Object message, Set<Channel> channels) {
        return writeAndFlush(message, channels::contains);
    }
}
