package cn.godmao.netty;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufHolder;
import io.netty.channel.*;
import io.netty.channel.epoll.*;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.nio.NioDatagramChannel;
import io.netty.channel.socket.nio.NioServerSocketChannel;
import io.netty.channel.socket.nio.NioSocketChannel;
import io.netty.util.NettyRuntime;
import io.netty.util.ReferenceCountUtil;
import io.netty.util.concurrent.DefaultThreadFactory;

import java.util.concurrent.ThreadFactory;

public class NettyUtil {
    private static final String  OS_NAME           = System.getProperty("os.name");
    private static       boolean isLinuxPlatform   = false;
    private static       boolean isWindowsPlatform = false;

    static {
        if (OS_NAME != null && OS_NAME.toLowerCase().contains("linux")) {
            isLinuxPlatform = true;
        }
        if (OS_NAME != null && OS_NAME.toLowerCase().contains("windows")) {
            isWindowsPlatform = true;
        }
    }


    public static Class<? extends Channel> getSocketChannelClass(NTP ntp) {
        boolean epoll = isEpoll();

        if (epoll && NTP.TCP.equals(ntp)) {
            return EpollSocketChannel.class;
        }

        if (epoll && NTP.UDP.equals(ntp)) {
            return EpollDatagramChannel.class;
        }

        if (!epoll && NTP.TCP.equals(ntp)) {
            return NioSocketChannel.class;
        }

        if (!epoll && NTP.UDP.equals(ntp)) {
            return NioDatagramChannel.class;
        }

        return null;
    }

    public static Class<? extends ServerChannel> getServerSocketChannelClass() {
        return isEpoll() ? EpollServerSocketChannel.class : NioServerSocketChannel.class;
    }

    public static EventLoopGroup newEventLoopGroup(String threadName) {
        return newEventLoopGroup(NettyRuntime.availableProcessors(), threadName);
    }

    public static EventLoopGroup newEventLoopGroup(int threadNum, String threadName) {
        return newEventLoopGroup(threadNum, new DefaultThreadFactory(threadName));
    }

    public static EventLoopGroup newEventLoopGroup(int threadNum) {
        return newEventLoopGroup(threadNum, (ThreadFactory) null);
    }

    public static EventLoopGroup newEventLoopGroup(int threadNum, ThreadFactory threadFactory) {
        return isEpoll() ?
                new EpollEventLoopGroup(threadNum, threadFactory) :
                new NioEventLoopGroup(threadNum, threadFactory);
    }


    public static boolean isEpoll() {
        return isLinuxPlatform && Epoll.isAvailable();
    }

    public static Object safeDuplicate(Object message) {
        if (message instanceof ByteBuf) {
            return ((ByteBuf) message).retainedDuplicate();
        } else if (message instanceof ByteBufHolder) {
            return ((ByteBufHolder) message).retainedDuplicate();
        } else {
            return ReferenceCountUtil.retain(message);
        }
    }

    public static ChannelFuture close(Channel channel) {
        if (channel.isActive()) {
            return channel.close();
        }
        return channel.closeFuture();
    }

    public static ChannelFuture send(ChannelOutboundInvoker channel, Object message) {
        return channel.writeAndFlush(message);
    }
}
