package cn.godmao.netty.client;

import cn.godmao.netty.ChannelService;
import cn.godmao.netty.NTP;
import cn.godmao.netty.client.base.ClientBaseInitializer;
import cn.godmao.netty.handler.IConnect;
import cn.godmao.netty.NettyUtil;
import io.netty.bootstrap.Bootstrap;
import io.netty.channel.*;
import io.netty.util.NettyRuntime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.net.URI;
import java.util.concurrent.TimeUnit;

public abstract class AbstractClient implements IConnect {
    public final  Logger         log = LoggerFactory.getLogger(this.getClass());
    private final Bootstrap      boot;
    private final EventLoopGroup group;
    private final ChannelService channelService;

    //
    private ChannelInitializer<Channel> channelInitializer;

    public AbstractClient() {
        this.boot = new Bootstrap();
        this.group = NettyUtil.newEventLoopGroup(NettyRuntime.availableProcessors(), "client-work-loop-group");
        this.channelService = new ChannelService("client");
    }

    @Override
    public void start() {
        // 设置一个默认的加载器
        if (null == this.channelInitializer) {
            this.channelInitializer = new ClientBaseInitializer(this);
        }

        init();
    }

    private void init() {
        /*
          ChannelOption.SINGLE_EVENTEXECUTOR_PER_GROUP
          Netty参数，单线程执行ChannelPipeline中的事件，默认值为True。
          该值控制执行ChannelPipeline中执行ChannelHandler的线程。
          如果为True，整个pipeline由一个线程执行，这样不需要进行线程切换以及线程同步，是Netty4的推荐做法；
          如果为False，ChannelHandler中的处理过程会由Group中的不同线程执行

          //        this.boot.option(ChannelOption.SINGLE_EVENTEXECUTOR_PER_GROUP, false);



          ChannelOption.WRITE_BUFFER_WATER_MARK
          设置写的高低水位线

         */

        //
        this.boot.group(this.group);
        //
        this.boot.option(ChannelOption.SO_KEEPALIVE, true);
        this.boot.option(ChannelOption.TCP_NODELAY, true);
        //
//        this.boot.option(ChannelOption.WRITE_BUFFER_WATER_MARK, new WriteBufferWaterMark(64 * 1024, 128 * 1024));
        //
        this.boot.channel(NettyUtil.getSocketChannelClass(NTP.TCP));
        //
        this.boot.handler(this.channelInitializer);
    }

    public ChannelFuture connect(String uri) {
        return connect(URI.create(uri));
    }

    public ChannelFuture connect(URI uri) {
        return connect(uri.getHost(), uri.getPort());
    }

    public ChannelFuture connect(String host, int port) {
        return connect(new InetSocketAddress(host, -1 == port ? 443 : port));
    }

    public ChannelFuture connect(SocketAddress address) {
        //
        ChannelFuture channelFuture_ = this.boot.connect(address);

        // 断线重连
        channelFuture_.channel().closeFuture().addListener((ChannelFutureListener) channelFutureClose -> {
            log.info("connect: {} status: {}", address.toString(), !channelFutureClose.isSuccess());
            if (channelFutureClose.isSuccess()) {
                channelFutureClose.channel().eventLoop().schedule(() -> {
                    connect(address);
                }, 3, TimeUnit.SECONDS);
            }
        });

        return channelFuture_;
    }

    @Override
    public ChannelService getChannelService() {
        return channelService;
    }


    public void setChannelInitializer(ChannelInitializer<Channel> channelInitializer) {
        this.channelInitializer = channelInitializer;
    }

}
