package cn.godmao.netty.server;

import cn.godmao.netty.ChannelService;
import cn.godmao.netty.handler.IConnect;
import cn.godmao.netty.NettyUtil;
import cn.godmao.netty.server.base.ServerBaseInitializer;
import io.netty.bootstrap.ServerBootstrap;
import io.netty.buffer.PooledByteBufAllocator;
import io.netty.channel.*;
import io.netty.util.NettyRuntime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractServer implements IConnect {
    public final Logger log = LoggerFactory.getLogger(this.getClass());

    //
    private final int                         port;
    private final ChannelService              channelService;
    private final ServerBootstrap             boot;
    private final EventLoopGroup              bossGroup;
    private final EventLoopGroup              workGroup;
    //
    private       ChannelInitializer<Channel> channelInitializer;
    private       ChannelFuture               channelFuture;


    public AbstractServer(int port) {
        this.port = port;
        this.boot = new ServerBootstrap();
        this.channelService = new ChannelService("server" + port);
        this.bossGroup = NettyUtil.newEventLoopGroup(1, "server" + port + "-" + "boss-loop-group");
        this.workGroup = NettyUtil.newEventLoopGroup(NettyRuntime.availableProcessors(), "server" + port + "-" + "work-loop-group");
    }


    @Override
    public void start() {
        // 设置一个默认的加载器
        if (null == this.channelInitializer) {
            this.channelInitializer = new ServerBaseInitializer(this);
        }

        init();
        bind();
    }

    private void init() {
        /*
          ChannelOption.SO_REUSEADDR
          某个服务占用了TCP的8080端口，其他服务再对这个端口进行监听就会报错，
          SO_REUSEADDR这个参数就是用来解决这个问题的，该参数允许服务公用一个端口，
          这个在服务器程序中比较常用，例如某个进程非正常退出，对一个端口的占用可能不会立即释放，
          这时候如果不设置这个参数，其他进程就不能立即使用这个端口

          TCP_NODELAY
          对应于socket选项中的TCP_NODELAY
          该参数的使用和Nagle算法有关，Nagle算法是将小的数据包组装为更大的帧进行发送，
          而不会来一个数据包发送一次，目的是为了提高每次发送的效率，因此在数据包没有组成足够大的帧时，
          就会延迟该数据包的发送，虽然提高了网络负载却造成了延时，TCP_NODELAY参数设置为true，就可以禁用Nagle算法，即使用小数据包即时传输
          TCP_NODELAY就是用于启用或关闭Nagle算法。如果要求高实时性，
          有数据发送时就马上发送，就将该选项设置为true关闭Nagle算法；
          如果要减少发送次数减少网络交互，就设置为false等累积一定大小后再发送。默认为false

          ChannelOption.WRITE_BUFFER_WATER_MARK
          设置写的高低水位线

          SO_SNDBUF & SO_RCVBUF 这两个值建议使用系统默认
          SO_SNDBUF：调整客户端 滑动窗口大小，在建立连接后系统自动设置，netty默认的是1024，属于 SocketChannal 参数
          SO_RCVBUF：服务器端接收缓冲区大小，单位字节。既可用于 SocketChannal 参数，也可以用于 ServerSocketChannal 参数（建议设置到 ServerSocketChannal 上）
         */

        //
        this.boot.group(bossGroup, workGroup);
        //
        this.boot.option(ChannelOption.SO_BACKLOG, 1024);
        this.boot.option(ChannelOption.SO_REUSEADDR, true);
        this.boot.option(ChannelOption.ALLOCATOR, PooledByteBufAllocator.DEFAULT);
        //
//        this.boot.childOption(ChannelOption.SO_SNDBUF, 65535);
//        this.boot.childOption(ChannelOption.SO_RCVBUF, 65535);
        this.boot.childOption(ChannelOption.SO_KEEPALIVE, true);
        this.boot.childOption(ChannelOption.TCP_NODELAY, true);
        this.boot.childOption(ChannelOption.ALLOCATOR, PooledByteBufAllocator.DEFAULT);
//        this.boot.childOption(ChannelOption.WRITE_BUFFER_WATER_MARK, new WriteBufferWaterMark(64 * 1024, 128 * 1024));
        //
        this.boot.channel(NettyUtil.getServerSocketChannelClass());
        //
        this.boot.childHandler(this.channelInitializer);
    }


    private void bind() {
        try {
            this.channelFuture = this.boot.bind(port).sync();
            this.channelFuture.channel().closeFuture().sync();
        } catch (Exception e) {
            log.error("server-" + this.port + " start error!", e);
            System.exit(0);
        } finally {
            stop();
        }
    }


    public void stop() {
        this.bossGroup.shutdownGracefully();
        this.workGroup.shutdownGracefully();
        this.getChannelService().getExecutorGroup().shutdownGracefully();
        this.channelFuture.channel().close();
    }

    public int getPort() {
        return port;
    }

    @Override
    public ChannelService getChannelService() {
        return channelService;
    }


    public void setChannelInitializer(ChannelInitializer<Channel> channelInitializer) {
        this.channelInitializer = channelInitializer;
    }

}