package cn.godmao.netty;

import cn.godmao.utils.RemotingUtil;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufHolder;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.EventLoopGroup;
import io.netty.channel.epoll.Epoll;
import io.netty.channel.epoll.EpollEventLoopGroup;
import io.netty.channel.epoll.EpollServerSocketChannel;
import io.netty.channel.epoll.EpollSocketChannel;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.ServerSocketChannel;
import io.netty.channel.socket.SocketChannel;
import io.netty.channel.socket.nio.NioServerSocketChannel;
import io.netty.channel.socket.nio.NioSocketChannel;
import io.netty.util.NettyRuntime;
import io.netty.util.ReferenceCountUtil;
import io.netty.util.concurrent.DefaultThreadFactory;

import java.util.concurrent.ThreadFactory;

public class NettyUtil {

    public static Class<? extends SocketChannel> getSocketChannelClass() {
        return isEpoll() ? EpollSocketChannel.class : NioSocketChannel.class;
    }

    public static Class<? extends ServerSocketChannel> getServerSocketChannelClass() {
        return isEpoll() ? EpollServerSocketChannel.class : NioServerSocketChannel.class;
    }

    public static EventLoopGroup newEventLoopGroup(String threadName) {
        return newEventLoopGroup(NettyRuntime.availableProcessors(), threadName);
    }

    public static EventLoopGroup newEventLoopGroup(int threadNum, String threadName) {
        return newEventLoopGroup(threadNum, new DefaultThreadFactory(threadName));
    }

    public static EventLoopGroup newEventLoopGroup(int threadNum) {
        return newEventLoopGroup(threadNum, (ThreadFactory) null);
    }

    public static EventLoopGroup newEventLoopGroup(int threadNum, ThreadFactory threadFactory) {
        return isEpoll() ?
                new EpollEventLoopGroup(threadNum, threadFactory) :
                new NioEventLoopGroup(threadNum, threadFactory);
    }


    public static boolean isEpoll() {
        return RemotingUtil.isLinuxPlatform() && Epoll.isAvailable();
    }

    public static Object safeDuplicate(Object message) {
        if (message instanceof ByteBuf) {
            return ((ByteBuf) message).retainedDuplicate();
        } else if (message instanceof ByteBufHolder) {
            return ((ByteBufHolder) message).retainedDuplicate();
        } else {
            return ReferenceCountUtil.retain(message);
        }
    }

    public static ChannelFuture close(Channel channel) {
        if (channel.isActive()) {
            return channel.close();
        }
        return channel.closeFuture();
    }

    public static ChannelFuture send(Channel channel, Object message) {
        return channel.writeAndFlush(message);
    }
}
