package cn.godmao.getty.server_websocket.client;

import cn.godmao.getty.common.GettyCommon;
import cn.godmao.getty.common.IServer;
import cn.godmao.common.Init;
import cn.godmao.getty.common.codec.IDecoder;
import cn.godmao.getty.common.codec.IEncoder;
import cn.godmao.utils.ClassUtil;
import io.netty.bootstrap.Bootstrap;
import io.netty.channel.*;
import io.netty.channel.epoll.EpollEventLoopGroup;
import io.netty.channel.epoll.EpollSocketChannel;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.nio.NioSocketChannel;
import io.netty.util.concurrent.DefaultThreadFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import java.net.InetSocketAddress;
import java.util.concurrent.TimeUnit;

public abstract class AbstractWebsocketClient implements IServer, Init.Init0 {
    public final Logger log = LoggerFactory.getLogger(this.getClass());
    private final WebsocketClientConfig config;
    private final IEncoder<?, ?> encoder;
    private final IDecoder<?, ?> decoder;
    private final ChannelInitializer<Channel> channelInitializer;

    //
    private Bootstrap bootstrap;
    private EventLoopGroup group;
    private ChannelFuture channelFuture;
    private Channel channel;

    public AbstractWebsocketClient(String uri) {
        this.config = new WebsocketClientConfig(uri);
        this.channelInitializer = ClassUtil.instance(config.getChannelInitializerClass());
        this.encoder = ClassUtil.instance(config.getEncoderClass());
        this.decoder = ClassUtil.instance(config.getDecoderClass());
    }

    @Override
    public void init() throws Exception {
        //
        GettyCommon.init(channelInitializer, this, config);
        GettyCommon.init(encoder, this, config);
        GettyCommon.init(decoder, this, config);
    }

    public String getHost() {
        return config.getHost();
    }

    public Integer getPort() {
        return config.getPort();
    }

    public String getUri() {
        return config.getUri();
    }

    @Override
    public WebsocketClientConfig getConfig() {
        return config;
    }

    @SuppressWarnings("unchecked")
    public <EP, ER> IEncoder<EP, ER> getEncoder() {
        return (IEncoder<EP, ER>) encoder;
    }

    @SuppressWarnings("unchecked")
    public <DP, DR> IDecoder<DP, DR> getDecoder() {
        return (IDecoder<DP, DR>) decoder;
    }

    @Override
    public void start() throws Exception {
        init();
        //
        final boolean epoll = GettyCommon.isEpoll();
        log.info("uri: {}", getUri());
        log.info("host: {}", getHost());
        log.info("port: {}", getPort());
        //
        Integer workTheards = config.getWorkTheard();
        String workTheardName = config.getWorkTheardName();
        DefaultThreadFactory threadFactory = new DefaultThreadFactory(workTheardName);

        //
        group = epoll ? new EpollEventLoopGroup(workTheards, threadFactory) : new NioEventLoopGroup(workTheards, threadFactory);
        bootstrap = new Bootstrap();
        bootstrap
                .option(ChannelOption.SO_KEEPALIVE, true)
                .option(ChannelOption.TCP_NODELAY, true)
                .group(group)
                .channel(epoll ? EpollSocketChannel.class : NioSocketChannel.class)
                .handler(channelInitializer)
                .remoteAddress(new InetSocketAddress(config.getHost(), config.getPort()))
        ;
        //
        connect();
    }

    public void connect() throws InterruptedException {
        //
        channelFuture = bootstrap.connect();
        channel = channelFuture.channel();
        channelFuture.addListener((ChannelFutureListener) channelFuture -> {
            log.debug("connect: {} status: {}", config.getUri(), channelFuture.isSuccess());
            if (channelFuture.isSuccess()) {
                return;
            }
            channelFuture.channel().eventLoop().schedule(() -> {
                try {
                    connect();
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }, 3, TimeUnit.SECONDS);
        });
        //
        channelFuture.channel().closeFuture().sync();
    }

    @Override
    public void stop() {
        if (null != channelFuture) {
            GettyCommon.close(channelFuture.channel());
        }
        final int quietPeriod = 5;
        final int timeout = 30;
        TimeUnit timeUnit = TimeUnit.SECONDS;
        if (null != group) {
            group.shutdownGracefully(quietPeriod, timeout, timeUnit);
        }
    }


    public Channel getChannel() {
        return channel;
    }

    public boolean isOnline() {
        return null != channel && channel.isActive();
    }


    public String getChannelId() {
        return null == channel ? null : channel.id().asLongText();
    }

    public void send(Object message) {
        GettyCommon.send(channel, message);
    }

    public abstract void onOpen();

    public abstract void onClose();

    public abstract void onError(Throwable throwable);

    public abstract void onMessage(Object message);
}
