package cn.godmao.getty.server.server;

import cn.godmao.common.Init;
import cn.godmao.getty.common.GettyCommon;
import cn.godmao.getty.common.IServer;
import cn.godmao.getty.common.channel.IChannel;
import cn.godmao.getty.common.channel.IChannelService;
import cn.godmao.getty.common.codec.IDecoder;
import cn.godmao.getty.common.codec.IEncoder;
import cn.godmao.utils.ClassUtil;
import io.netty.bootstrap.ServerBootstrap;
import io.netty.buffer.PooledByteBufAllocator;
import io.netty.channel.*;
import io.netty.channel.epoll.EpollEventLoopGroup;
import io.netty.channel.epoll.EpollServerSocketChannel;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.nio.NioServerSocketChannel;
import io.netty.util.concurrent.DefaultThreadFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.InvocationTargetException;
import java.util.concurrent.TimeUnit;


public abstract class AbstractServer<C extends IChannel<?>> implements IServer, Init.Init0 {
    public final Logger log = LoggerFactory.getLogger(this.getClass());
    private final ServerConfig config;
    private final IEncoder<?, ?> encoder;
    private final IDecoder<?, ?> decoder;
    private final IChannelService<?, C> channelService;
    private final ChannelInitializer<Channel> channelInitializer;

    //
    private ServerBootstrap bootstrap;
    private EventLoopGroup bossGroup;
    private EventLoopGroup childGroup;
    private ChannelFuture channelFuture;

    public AbstractServer(Integer port) {
        this.config = new ServerConfig(port);
        this.channelInitializer = ClassUtil.instance(config.getChannelInitializerClass());
        this.channelService = (IChannelService<?, C>) ClassUtil.instance(config.getChannelServiceClass());
        this.encoder = ClassUtil.instance(config.getEncoderClass());
        this.decoder = ClassUtil.instance(config.getDecoderClass());
    }

    @Override
    public void init() throws InvocationTargetException, NoSuchMethodException, InstantiationException, IllegalAccessException {
        GettyCommon.init(channelInitializer, this, config);
        GettyCommon.init(channelService, this, config);
        GettyCommon.init(encoder, this, config);
        GettyCommon.init(decoder, this, config);
    }

    @Override
    public void start() throws Exception{
        init();
        //
        final boolean epoll = GettyCommon.isEpoll();
        log.info("port: {}", config.getPort());
        //
        Integer workTheards = config.getWorkTheard();
        String workTheardName = config.getWorkTheardName();
        DefaultThreadFactory threadFactory = new DefaultThreadFactory(workTheardName);

        //
        bossGroup = epoll ? new EpollEventLoopGroup(1) : new NioEventLoopGroup(1);
        childGroup = epoll ? new EpollEventLoopGroup(workTheards, threadFactory) : new NioEventLoopGroup(workTheards, threadFactory);
        bootstrap = new ServerBootstrap();
        //
        bootstrap.group(bossGroup, childGroup)
                //
                .channel(epoll ? EpollServerSocketChannel.class : NioServerSocketChannel.class)
                //
                .option(ChannelOption.SO_BACKLOG, 1024)
                .option(ChannelOption.SO_REUSEADDR, true)
//                    .option(ChannelOption.SO_KEEPALIVE, false)
                //
//                    .childOption(ChannelOption.SO_KEEPALIVE, false)
                .childOption(ChannelOption.SO_SNDBUF, 65535)
                .childOption(ChannelOption.SO_RCVBUF, 65535)
                .childOption(ChannelOption.TCP_NODELAY, true)
                .childOption(ChannelOption.ALLOCATOR, PooledByteBufAllocator.DEFAULT)
                //
                .childHandler(channelInitializer);

        //
        bind();
    }

    private void bind() {
        try {
            //
            channelFuture = bootstrap.bind(config.getPort()).sync();
            channelFuture.channel().closeFuture().sync();
        } catch (Exception e) {
            log.error("Server start error!", e);
            System.exit(0);
        } finally {
            childGroup.shutdownGracefully();
            bossGroup.shutdownGracefully();
        }
    }

    @Override
    public void stop() {
        if (channelFuture != null) {
            GettyCommon.close(channelFuture.channel());
        }
        final int quietPeriod = 5;
        final int timeout = 30;
        final TimeUnit timeUnit = TimeUnit.SECONDS;
        if (null != childGroup) {
            childGroup.shutdownGracefully(quietPeriod, timeout, timeUnit);
        }
        if (null != bossGroup) {
            bossGroup.shutdownGracefully(quietPeriod, timeout, timeUnit);
        }
    }

    @Override
    public ServerConfig getConfig() {
        return config;
    }


    public IChannelService<?, C> getChannelService() {
        return channelService;
    }

    @SuppressWarnings("unchecked")
    public <EP, ER> IEncoder<EP, ER> getEncoder() {
        return (IEncoder<EP, ER>) encoder;
    }

    @SuppressWarnings("unchecked")
    public <DP, DR> IDecoder<DP, DR> getDecoder() {
        return (IDecoder<DP, DR>) decoder;
    }

    public abstract void onOpen(C channel);

    public abstract void onClose(C channel);

    public abstract void onError(C channel, Throwable throwable);

    public abstract void onMessage(C channel, Object message);

}
