package cn.godmao.getty.common;

import cn.godmao.common.Init;
import cn.godmao.getty.common.channel.IChannelService;
import cn.godmao.getty.server.server.AbstractServer;
import cn.godmao.utils.ClassUtil;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandlerContext;
import io.netty.util.ReferenceCountUtil;
import io.netty.util.concurrent.DefaultThreadFactory;
import io.netty.util.concurrent.EventExecutor;
import io.netty.util.internal.PlatformDependent;

import java.lang.reflect.InvocationTargetException;
import java.util.Collection;
import java.util.concurrent.ConcurrentMap;

public abstract class AbstractChannelService<C extends AbstractChannel<String>> implements IChannelService<String, C>, Init.Init1<AbstractServer<C>> {
    private final ConcurrentMap<String, C> channelGroup = PlatformDependent.newConcurrentHashMap();
    private final ChannelFutureListener remover = future -> remove(future.channel());
    private final Class<C> channelClass;
    private ExecutorService executorService;
    private AbstractServer<?> abstractServer;

    @SuppressWarnings("unchecked")
    public AbstractChannelService() {
        this.channelClass = (Class<C>) ClassUtil.find0(this, AbstractChannelService.class, "C");
    }

    public ConcurrentMap<String, C> getChannelGroup() {
        return channelGroup;
    }

    public Class<C> getChannelClass() {
        return channelClass;
    }

    @Override
    public synchronized void init(AbstractServer<C> abstractServer) {
        this.abstractServer = abstractServer;
        this.executorService = new ExecutorService(abstractServer.getConfig().getUserTheard(), new DefaultThreadFactory(abstractServer.getConfig().getUserTheardName()));
    }

    @Override
    public Collection<C> get() {
        return channelGroup.values();
    }

    @Override
    public C get(Object o) throws InvocationTargetException, NoSuchMethodException, InstantiationException, IllegalAccessException {
        Channel channel = null;
        String key = null;
        if (o instanceof Channel) {
            channel = (Channel) o;
        } else if (o instanceof ChannelHandlerContext) {
            channel = ((ChannelHandlerContext) o).channel();
        } else if (o instanceof String) {
            key = (String) o;
        } else {
            throw new IllegalArgumentException("未知类型: " + o);
        }

        if (null != channel) {
            key = getKey(channel);
        }

        C serverChannel = getChannelGroup().get(key);
        if (null == serverChannel) {
            serverChannel = ClassUtil.instance(getChannelClass(), new Class<?>[]{String.class, Channel.class, EventExecutor.class}, key, channel, getExecutorService().next());
            add(serverChannel);
        }

        return serverChannel;
    }

    @Override
    public ExecutorService getExecutorService() {
        return executorService;
    }


    public void send(Collection<? extends String> ids, Object message) {
        Object data = abstractServer.getEncoder().encode(message);
        boolean emptyIds = null == ids;
        for (C channelUser : get()) {
            if (emptyIds || ids.contains(channelUser.getId())) {
                channelUser.send(GettyCommon.safeDuplicate(data));
            }
        }
        ReferenceCountUtil.release(data);
    }


    public void send(Object message) {
        send(null, message);
    }

    protected String getKey(Channel channel) {
        return channel.id().asLongText();
    }

    protected void add(C channelUser) {
        boolean added = channelGroup.putIfAbsent(getKey(channelUser.getChannel()), channelUser) == null;
        if (added) {
            channelUser.getChannel().closeFuture().addListener(remover);
        }
    }

    private void remove(Channel channel) {
        C channelUser = channelGroup.remove(getKey(channel));
        if (channelUser != null) {
            channelUser.getChannel().closeFuture().removeListener(remover);
        }
    }
}
