package cn.godmao.getty.client;

import cn.godmao.getty.GettyCommon;
import cn.godmao.getty.IServer;
import cn.godmao.getty.codec.IDecoder;
import cn.godmao.getty.codec.IEncoder;
import cn.godmao.common.Init;
import cn.godmao.utils.ClassUtil;
import io.netty.bootstrap.Bootstrap;
import io.netty.channel.*;
import io.netty.channel.epoll.EpollEventLoopGroup;
import io.netty.channel.epoll.EpollSocketChannel;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.nio.NioSocketChannel;
import io.netty.util.concurrent.DefaultThreadFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import java.lang.reflect.InvocationTargetException;
import java.net.InetSocketAddress;
import java.net.URI;
import java.util.concurrent.TimeUnit;

public abstract class AbstractClient implements IServer, Init.Init0 {
    public final Logger log = LoggerFactory.getLogger(this.getClass());
    private final ClientConfig config;
    private final URI uri;

    //
    private  IEncoder<?, ?> encoder;
    private  IDecoder<?, ?> decoder;
    private  ChannelInitializer<Channel> channelInitializer;

    //
    private Bootstrap bootstrap;
    private EventLoopGroup group;
    private ChannelFuture channelFuture;
    private Channel channel;


    public AbstractClient(String uri) {
        this.uri = URI.create(uri);
        this.config = new ClientConfig(uri);
    }

    @Override
    public void init() throws InvocationTargetException, NoSuchMethodException, InstantiationException, IllegalAccessException {

        //
        this.channelInitializer = ClassUtil.instance(config.getChannelInitializerClass());
        this.encoder = ClassUtil.instance(config.getEncoderClass());
        this.decoder = ClassUtil.instance(config.getDecoderClass());

        //
        GettyCommon.init(channelInitializer, this, config);
        GettyCommon.init(encoder, this, config);
        GettyCommon.init(decoder, this, config);
    }

    @Override
    public ClientConfig getConfig() {
        return config;
    }

    @SuppressWarnings("unchecked")
    public <EP, ER> IEncoder<EP, ER> getEncoder() {
        return (IEncoder<EP, ER>) encoder;
    }

    @SuppressWarnings("unchecked")
    public <DP, DR> IDecoder<DP, DR> getDecoder() {
        return (IDecoder<DP, DR>) decoder;
    }

    public URI getUri() {
        return uri;
    }

    @Override
    public void start() throws Exception {
        init();
        //
        final boolean epoll = GettyCommon.isEpoll();
        log.info("uri: {}", config.getUri());

        //
        Integer workTheards = config.getWorkTheard();
        String workTheardName = config.getWorkTheardName();
        DefaultThreadFactory threadFactory = new DefaultThreadFactory(workTheardName);

        //
        group = epoll ? new EpollEventLoopGroup(workTheards, threadFactory) : new NioEventLoopGroup(workTheards, threadFactory);
        bootstrap = new Bootstrap();
        bootstrap.group(group)
                .channel(epoll ? EpollSocketChannel.class : NioSocketChannel.class)
                .remoteAddress(new InetSocketAddress(uri.getHost(), (uri.getPort() == -1) ? 443 : uri.getPort()))
                .option(ChannelOption.SO_KEEPALIVE, true)
                .handler(channelInitializer);
        //
        connect();
    }


    public void connect() {
        //
        channelFuture = bootstrap.connect();
        channel = channelFuture.channel();
        channelFuture.addListener((ChannelFutureListener) channelFuture -> {
            log.debug("connect: {} status: {}", config.getUri(), channelFuture.isSuccess());
            if (!channelFuture.isSuccess()) {
                channelFuture.channel().eventLoop().schedule(this::connect, 3, TimeUnit.SECONDS);
            }
        });
    }

    @Override
    public void stop() {
        if (null != channelFuture) {
            GettyCommon.close(channelFuture.channel());
        }
        final int quietPeriod = 5;
        final int timeout = 30;
        TimeUnit timeUnit = TimeUnit.SECONDS;
        if (null != group) {
            group.shutdownGracefully(quietPeriod, timeout, timeUnit);
        }
    }

    public boolean isOnline() {
        return null != channel && channel.isActive();
    }


    public String getChannelId() {
        return null == channel ? null : channel.id().asLongText();
    }

    public void send(Object message) {
        GettyCommon.send(channel, message);
    }

    public abstract void onOpen();

    public abstract void onClose();

    public abstract void onError(Throwable throwable);

    public abstract void onMessage(Object message);
}
