package cn.godmao.getty;

import cn.godmao.common.Init;
import cn.godmao.getty.channel.IChannelService;
import cn.godmao.getty.server.AbstractServer;
import cn.godmao.utils.ClassUtil;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandlerContext;
import io.netty.util.ReferenceCountUtil;
import io.netty.util.concurrent.DefaultThreadFactory;
import io.netty.util.internal.PlatformDependent;
import java.util.Collection;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;

public abstract class AbstractChannelService<C extends AbstractChannel> implements IChannelService<Object, C>, Init.Init1<AbstractServer<C>> {
    private final ConcurrentMap<String, C> channelGroup = PlatformDependent.newConcurrentHashMap();
    private final ChannelFutureListener remover = future -> remove(future.channel());
    private final Class<C> channelClass;
    private ExecutorService executorService;
    private AbstractServer<?> server;

    @SuppressWarnings("unchecked")
    public AbstractChannelService() {
        this.channelClass = (Class<C>) ClassUtil.find0(this, AbstractChannelService.class, "C");
    }

    public ConcurrentMap<String, C> getChannelGroup() {
        return channelGroup;
    }

    public Class<C> getChannelClass() {
        return channelClass;
    }

    @Override
    public synchronized void init(AbstractServer<C> abstractServer) {
        this.server = abstractServer;
        this.executorService = new ExecutorService(abstractServer.getConfig().getUserTheard(), new DefaultThreadFactory(abstractServer.getConfig().getUserTheardName()));
    }

    public AbstractServer<?> getServer() {
        return server;
    }

    @Override
    public Collection<C> get() {
        return channelGroup.values();
    }

    @Override
    public C get(Object o) {
        Channel channel = null;
        String key = null;
        if (o instanceof Channel) {
            channel = (Channel) o;
        } else if (o instanceof ChannelHandlerContext) {
            channel = ((ChannelHandlerContext) o).channel();
        } else if (o instanceof String) {
            key = (String) o;
        } else {
            throw new IllegalArgumentException("未知类型: " + o);
        }

        if (null != channel) {
            key = getKey(channel);
        }

        C serverChannel = getChannelGroup().get(key);
        if (null == serverChannel) {
            serverChannel = ClassUtil.instance(getChannelClass());
            serverChannel.init(key, channel, getExecutorService().next());
            add(serverChannel);
        }

        return serverChannel;
    }

    @Override
    public ExecutorService getExecutorService() {
        return executorService;
    }


    public void send(Set<?> ids, Object message) {
        Object data = server.getEncoder().encode(message);
        boolean emptyIds = null == ids;
        for (C channelUser : get()) {
            if (emptyIds || ids.contains(channelUser.getId())) {
                channelUser.send(GettyCommon.safeDuplicate(data));
            }
        }
        ReferenceCountUtil.release(data);
    }


    public void send(Object message) {
        send(null, message);
    }

    protected String getKey(Channel channel) {
        return channel.id().asLongText();
    }

    protected void add(C channelUser) {
        boolean added = channelGroup.putIfAbsent(getKey(channelUser.getChannel()), channelUser) == null;
        if (added) {
            channelUser.getChannel().closeFuture().addListener(remover);
        }
    }

    private void remove(Channel channel) {
        C channelUser = channelGroup.remove(getKey(channel));
        if (channelUser != null) {
            channelUser.getChannel().closeFuture().removeListener(remover);
        }
    }
}
