package cn.godmao.executor;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicLong;

public abstract class AbstractExecutorService<E extends ExecutorService> implements IExecutorService<E> {
    protected final AtomicLong counter;
    protected       E[]        executors;
    protected       String     thread_name;

    public AbstractExecutorService(E[] executors) {
        this(null, executors);
    }

    public AbstractExecutorService(String thread_name, E[] executors) {
        this.counter = new AtomicLong();
        this.executors = executors;
        this.thread_name = thread_name;
    }

    public AtomicLong getCounter() {
        return counter;
    }

    public String getThreadName() {
        return thread_name;
    }

    @Override
    public boolean isTerminated() {
        boolean isTerminated = true;
        for (E executor : getExecutors()) {
            if (!executor.isTerminated()) {
                isTerminated = false;
                break;
            }
        }
        return isTerminated;
    }

    @Override
    public boolean isShutdown() {
        boolean isShutdown = true;
        for (E executor : getExecutors()) {
            if (!executor.isShutdown()) {
                isShutdown = false;
                break;
            }
        }
        return isShutdown;
    }

    @Override
    public void shutdown() {
        for (E executor : getExecutors()) {
            executor.shutdown();
        }
    }

    @Override
    public E[] getExecutors() {
        return executors;
    }

    @Override
    public E next() {
        return getExecutors()[Math.abs((int) (getCounter().incrementAndGet() % getExecutors().length))];
    }

    @Override
    public void execute(Runnable runnable) {
        execute(next(), runnable);
    }

    @Override
    public Future<?> submit(Runnable task) {
        return submit(next(), task);
    }

    @Override
    public <T> Future<T> submit(Runnable task, T result) {
        return submit(next(), task, result);
    }

    @Override
    public <T> Future<T> submit(Callable<T> task) {
        return submit(next(), task);
    }

}
