package cn.godmao.airserver.action;

import cn.godmao.exception.Code;
import cn.godmao.airserver.action.annotation.Action;
import cn.godmao.airserver.action.annotation.ActionController;
import cn.godmao.json.JsonUtil;
import cn.godmao.utils.ClassUtil;
import cn.godmao.utils.CollectUtil;
import cn.godmao.utils.ObjectUtil;
import cn.godmao.utils.clazz.MethodAccess;
import cn.hutool.core.lang.ClassScanner;
import org.springframework.core.DefaultParameterNameDiscoverer;
import org.springframework.core.annotation.AnnotationUtils;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.*;

public class ActionHandler {
    public static final DefaultParameterNameDiscoverer defaultParameterNameDiscoverer = new DefaultParameterNameDiscoverer();
    private final       Collection<Object>             beans;
    private final       Map<String, Set<MethodNode>>   actionGroup;

    public void init(Class<?> mainClass) {
        ClassScanner classScanner = new ClassScanner(mainClass.getPackage().getName(), clazz -> Objects.nonNull(AnnotationUtils.findAnnotation(clazz, ActionController.class)));
        Collection<Object> beans = new ArrayList<>();
        for (Class<?> clazz : classScanner.scan()) {
            ActionController controller = AnnotationUtils.findAnnotation(clazz, ActionController.class);
            if (null == controller) {
                continue;
            }
            beans.add(ClassUtil.getConstructorAccess(clazz).newInstance());
        }
        this.init(beans);
    }

    public synchronized void init(Collection<Object> beans) {
        ObjectUtil.checkNotNull(beans, "beans is null");
        this.beans.addAll(beans);
        for (Object bean : this.beans) {
            Class<?> beanClass = bean.getClass();
            List<Method> methods = ClassUtil.getMethods(ClassUtil.getDeclaredMethods(beanClass), Modifier.PRIVATE);

            ActionController actionController = AnnotationUtils.findAnnotation(beanClass, ActionController.class);
            if (null == actionController) {
                continue;
            }
            final String[] values_actionController = actionController.value();
            for (int i = 0; i < methods.size(); i++) {
                Method method = methods.get(i);
                Action action = AnnotationUtils.findAnnotation(method, Action.class);
                if (null == action) {
                    continue;
                }
                final String[] values_action = action.value();
                List<String> valuesAction = CollectUtil.listOf(values_action);
                for (String value_actionController : values_actionController) {
                    valuesAction.replaceAll(value_action -> value_actionController + value_action);
                }

                String methodName = method.getName();
                Class<?>[] parameterTypes = method.getParameterTypes();
                String[] parameterNames = defaultParameterNameDiscoverer.getParameterNames(method);

                Set<Param> params = new LinkedHashSet<>();
                for (int index_ = 0; index_ < parameterTypes.length; index_++) {
                    Class<?> parameterType = parameterTypes[index_];
                    String parameterName = parameterNames[index_];
                    params.add(new Param(index_, parameterType, parameterName, ObjectUtil.defaultValue(parameterType)));
                }
                MethodNode methodNode = new MethodNode(params, methodName, !method.getReturnType().isAssignableFrom(void.class), action, bean, i);
                for (String key : valuesAction) {
                    Set<MethodNode> methodNodes = actionGroup.getOrDefault(key, new HashSet<>());
                    boolean add = methodNodes.add(methodNode);
                    if (!add) {
                        throw Code.FAIL.exception(beanClass.getSimpleName() + "." + key + "." + methodName + ": 接口重复,请检查!");
                    }
                    actionGroup.put(key, methodNodes);
                }
            }
        }
    }

    public Collection<Object> getBeans() {
        return beans;
    }


    public MethodNode match(String key) {
        return match(key, new HashSet<>());
    }

    public Set<ActionHandler.Param> getParm(Object object) {
        final Set<ActionHandler.Param> params = new HashSet<>();
        if (object instanceof Collection<?>) {
            for (Object o : (Collection<?>) object) {
                if (o instanceof ActionHandler.Param) {
                    params.add((ActionHandler.Param) o);
                }
            }
        } else if (object instanceof Map<?, ?>) {
            Map<?, ?> map = (Map<?, ?>) object;
            for (Object key : map.keySet()) {
                params.add(new ActionHandler.Param(String.valueOf(key), map.get(key)));
            }
        } else if (object instanceof ActionHandler.Param) {
            params.add((ActionHandler.Param) object);
        } else {
            params.add(new Param("args0", object));
        }
        return params;
    }

    public MethodNode match(String key, Set<Param> params) {
        Set<MethodNode> methodNodes = actionGroup.get(key);
        if (null == methodNodes || methodNodes.isEmpty()) {
            throw Code.FAIL.exception(404, "not found", key);
        }
        double weight_ = 0d;
        MethodNode methodNode_ = null;
        for (MethodNode methodNode : methodNodes) {
            double weight = methodNode.match(params);
            if (null == methodNode_ || weight > weight_) {
                weight_ = weight;
                methodNode_ = methodNode;
            }
        }
        return methodNode_;
    }


    public static class Param {
        private final int      index;
        private final Class<?> pClass;
        private final Object   value;
        private final String   name;

        public Param(int index, Class<?> pClass, String name, Object value) {
            this.index = index;
            this.pClass = pClass;
            this.value = value;
            this.name = name;
        }

        public Param(int index, String name, Object value) {
            this(index, null != value ? value.getClass() : null, name, value);
        }

        public Param(String name, Object value) {
            this(-1, name, value);
        }

        public Class<?> getpClass() {
            return pClass;
        }

        public Object getValue() {
            return value;
        }

        public String getName() {
            return name;
        }

        public int getIndex() {
            return index;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            Param param = (Param) o;
            return Objects.equals(pClass, param.pClass) && Objects.equals(name, param.name);
        }

        @Override
        public int hashCode() {
            return Objects.hash(pClass, name);
        }

        @Override
        public String toString() {
            return "{" +
                    "name:" + name +
                    "," +
                    "value:" + value +
                    "}";
        }
    }


    @FunctionalInterface
    private interface Matcher<T, R> {
        R match(T t);
    }


    public static class MethodNode extends MethodAccess implements Matcher<Set<Param>, Double> {
        private final Set<Param> params;
        private final String     methodName;
        private final Action     action;
        private final Object     bean;
        private final Integer    index;
        private final boolean    hasReturn;

        public MethodNode(Set<Param> params, String methodName, boolean hasReturn, Action action, Object bean, int index) {
            super(bean);
            this.params = params;
            this.methodName = methodName;
            this.hasReturn = hasReturn;
            this.action = action;
            this.bean = bean;
            this.index = index;
        }

        public String getMethodName() {
            return methodName;
        }

        public Action getAction() {
            return action;
        }

        public Object getBean() {
            return bean;
        }

        public Integer getIndex() {
            return index;
        }

        public Set<Param> getParams() {
            return params;
        }

        public boolean hasReturn() {
            return hasReturn;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            MethodNode that = (MethodNode) o;
            return Objects.equals(params, that.params);
        }

        @Override
        public int hashCode() {
            return Objects.hash(params);
        }

        @Override
        public Double match(Set<Param> paramsThat) {
            Set<Param> paramsThis = getParams();

            double weight = 0.0d;
            //  根据参数长度加权重
            double addweight = Math.abs(paramsThat.size() - getParams().size());
            if (addweight == 0) {
                addweight = -1;
            }
            weight = weight - addweight;

            // 根据参数名加权重 + 根据参数类型加权重
            for (Param paramThis : paramsThis) {
                final Class<?> paramThisClass = paramThis.getpClass();
                final String paramThisName = paramThis.getName();

                if (paramsThat.contains(paramThis)) {
                    // +3
                    weight += 3.0d;
                } else if (paramsThat.stream().anyMatch(param -> param.getName().equals(paramThisName))) {
                    // +2
                    weight += 2.0d;
                } else if (paramsThat.stream().anyMatch(param -> paramThisClass.isAssignableFrom(param.getpClass()))) {
                    // +1
                    weight += 1.0d;
                }
            }
            return weight;
        }

        public Object invoke() {
            return invoke((int) index, (Object[]) new Object[0]);
        }

        public Object invoke(Object... args) {
            return invoke(index, (Object[]) args);
        }

        public Object invoke(Param... params) {
            return invoke((Collection<Param>) Arrays.asList(params));
        }

        public Object invoke(Collection<Param> params) {

            final Object[] args = new Object[getParams().size()];
            final List<Param> params_ = new ArrayList<>(getParams());

            // 0 匹配所有参数下标相同的值
            if (!params_.isEmpty()) {
                //
                final Map<Integer, Param> paramIndexGroup = CollectUtil.toMap(params, Param::getIndex);

                for (int i = params_.size() - 1; i >= 0; i--) {
                    Param paramOrig = params_.get(i);
                    Param paramByIndex = paramIndexGroup.get(paramOrig.getIndex());
                    //
                    if (null != paramByIndex) {
                        args[paramOrig.getIndex()] = paramOrig.getpClass().isAssignableFrom(paramByIndex.getpClass()) ?
                                paramByIndex.getValue()
                                :
                                JsonUtil.parse(paramByIndex.getValue(), paramOrig.getpClass());
                        params_.remove((int) i);
                    }
                }
            }


            // 1 匹配参数名相同并且参数类型相同的值
            if (!params_.isEmpty()) {
                //
                final Map<Integer, Param> paramHashGroup = CollectUtil.toMap(params, Param::hashCode);
                for (int i = params_.size() - 1; i >= 0; i--) {
                    Param paramOrig = params_.get(i);
                    Param paramByHash = paramHashGroup.get(paramOrig.hashCode());
                    //
                    if (null != paramByHash) {
                        args[paramOrig.getIndex()] = paramByHash.getValue();
                        params_.remove((int) i);
                    }
                }
            }


            // 2 匹配所有参数名相同的值
            if (!params_.isEmpty()) {
                //
                final Map<String, Param> paramNameGroup = CollectUtil.toMap(params, Param::getName);
                for (int i = params_.size() - 1; i >= 0; i--) {
                    Param paramOrig = params_.get(i);
                    Param paramByName = paramNameGroup.get(paramOrig.getName());
                    //
                    if (null != paramByName) {
                        args[paramOrig.getIndex()] = paramOrig.getpClass().isAssignableFrom(paramByName.getpClass()) ?
                                paramByName.getValue()
                                :
                                JsonUtil.parse(paramByName.getValue(), paramOrig.getpClass());
                        params_.remove((int) i);
                    }
                }
            }

            // 3 匹配所有参数类型相同的值
            if (!params_.isEmpty()) {
                //
                final Map<? extends Class<?>, Param> paramClassGroup = CollectUtil.toMap(params, Param::getpClass);
                for (int i = params_.size() - 1; i >= 0; i--) {
                    Param paramOrig = params_.get(i);
                    Param paramByClass = paramClassGroup.get(paramOrig.getpClass());
                    //
                    if (null != paramByClass) {
                        args[paramOrig.getIndex()] = paramByClass.getValue();
                        params_.remove((int) i);
                    }
                }
            }


            // 4 匹配剩余的值 设置默认值
            if (!params_.isEmpty()) {
                //
                for (int i = params_.size() - 1; i >= 0; i--) {
                    Param paramOrig = params_.get(i);
                    args[paramOrig.getIndex()] = ObjectUtil.defaultValue(paramOrig.getpClass());
                }
            }

            return invoke((Object[]) args);
        }
    }

    public ActionHandler() {
        this.beans = new ArrayList<>();
        this.actionGroup = new HashMap<>();
    }

    public static ActionHandler me() {
        return Holder.ME;
    }


    private static class Holder {
        static final ActionHandler ME = new ActionHandler();
    }

}
