package cn.godmao.airserver.action;

import cn.godmao.exception.Code;
import cn.godmao.airserver.action.annotation.Action;
import cn.godmao.airserver.action.annotation.ActionController;
import cn.godmao.json.JsonUtil;
import cn.godmao.utils.ClassUtil;
import cn.godmao.utils.CollectUtil;
import cn.godmao.utils.ObjectUtil;
import cn.godmao.utils.clazz.MethodAccess;
import cn.hutool.core.lang.ClassScanner;
import org.springframework.core.DefaultParameterNameDiscoverer;
import org.springframework.core.annotation.AnnotationUtils;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.*;

public class ActionHandler {
    public static final DefaultParameterNameDiscoverer defaultParameterNameDiscoverer = new DefaultParameterNameDiscoverer();
    private final       Collection<Object>             beans;
    private final       Map<String, Set<MethodNode>>   actionGroup;

    public void init(Class<?> mainClass) {
        ClassScanner classScanner = new ClassScanner(mainClass.getPackage().getName(), clazz -> Objects.nonNull(AnnotationUtils.findAnnotation(clazz, ActionController.class)));
        Collection<Object> beans = new ArrayList<>();
        for (Class<?> clazz : classScanner.scan()) {
            ActionController controller = AnnotationUtils.findAnnotation(clazz, ActionController.class);
            if (null == controller) {
                continue;
            }
            beans.add(ClassUtil.getConstructorAccess(clazz).newInstance());
        }
        this.init(beans);
    }

    public synchronized void init(Collection<Object> beans) {
        ObjectUtil.checkNotNull(beans, "beans is null");
        this.beans.addAll(beans);
        for (Object bean : this.beans) {
            Class<?> beanClass = bean.getClass();
            List<Method> methods = ClassUtil.getMethods(ClassUtil.getDeclaredMethods(beanClass), Modifier.PRIVATE);

            ActionController actionController = AnnotationUtils.findAnnotation(beanClass, ActionController.class);
            if (null == actionController) {
                continue;
            }
            final String[] values_actionController = actionController.value();
            for (int i = 0; i < methods.size(); i++) {
                Method method = methods.get(i);
                Action action = AnnotationUtils.findAnnotation(method, Action.class);
                if (null == action) {
                    continue;
                }
                final String[] values_action = action.value();
                List<String> valuesAction = CollectUtil.listOf(values_action);
                for (String value_actionController : values_actionController) {
                    valuesAction.replaceAll(value_action -> value_actionController + value_action);
                }

                String methodName = method.getName();
                Class<?>[] parameterTypes = method.getParameterTypes();
                String[] parameterNames = defaultParameterNameDiscoverer.getParameterNames(method);

                Set<Param> params = new LinkedHashSet<>();
                for (int i1 = 0; i1 < parameterTypes.length; i1++) {
                    Class<?> parameterType = parameterTypes[i1];
                    String parameterName = parameterNames[i1];
                    params.add(new Param(parameterType, parameterName, ObjectUtil.defaultValue(parameterType)));
                }
                MethodNode methodNode = new MethodNode(params, methodName, !method.getReturnType().isAssignableFrom(void.class), action, bean, i);
                for (String key : valuesAction) {
                    Set<MethodNode> methodNodes = actionGroup.getOrDefault(key, new HashSet<>());
                    boolean add = methodNodes.add(methodNode);
                    if (!add) {
                        throw Code.FAIL.exception(beanClass.getSimpleName() + "." + key + "." + methodName + ": 接口重复,请检查!");
                    }
                    actionGroup.put(key, methodNodes);
                }
            }
        }
    }

    public Collection<Object> getBeans() {
        return beans;
    }


    public MethodNode match(String key) {
        return match(key, new HashSet<>());
    }


    public MethodNode match(String key, Set<Param> params) {
        Set<MethodNode> methodNodes = actionGroup.get(key);
        if (null == methodNodes || methodNodes.isEmpty()) {
            throw Code.FAIL.exception(404, "not found", key);
        }
        double weight_ = 0d;
        MethodNode methodNode_ = null;
        for (MethodNode methodNode : methodNodes) {
            double weight = methodNode.match(params);
            if (null == methodNode_ || weight > weight_) {
                weight_ = weight;
                methodNode_ = methodNode;
            }
        }
        return methodNode_;
    }


    public static class Param {
        private final Class<?> pClass;
        private final Object   value;
        private final String   name;

        public Param(Class<?> pClass, String name, Object value) {
            this.pClass = pClass;
            this.value = value;
            this.name = name;
        }

        public Param(String name, Object value) {
            this(null != value ? value.getClass() : null, name, value);
        }

        public Class<?> getpClass() {
            return pClass;
        }

        public Object getValue() {
            return value;
        }

        public String getName() {
            return name;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            Param param = (Param) o;
            return Objects.equals(pClass, param.pClass) && Objects.equals(name, param.name);
        }

        @Override
        public int hashCode() {
            return Objects.hash(pClass, name);
        }

        @Override
        public String toString() {
            return "{" +
                    "name:" + name +
                    "," +
                    "value:" + value +
                    "}";
        }
    }


    @FunctionalInterface
    private interface Matcher<T, R> {
        R match(T t);
    }


    public static class MethodNode extends MethodAccess implements Matcher<Set<Param>, Double> {
        private final Set<Param> params;
        private final String     methodName;
        private final Action     action;
        private final Object     bean;
        private final Integer    index;
        private final boolean    hasReturn;

        public MethodNode(Set<Param> params, String methodName, boolean hasReturn, Action action, Object bean, int index) {
            super(bean);
            this.params = params;
            this.methodName = methodName;
            this.hasReturn = hasReturn;
            this.action = action;
            this.bean = bean;
            this.index = index;
        }

        public String getMethodName() {
            return methodName;
        }

        public Action getAction() {
            return action;
        }

        public Object getBean() {
            return bean;
        }

        public Integer getIndex() {
            return index;
        }

        public Set<Param> getParams() {
            return params;
        }

        public boolean hasReturn() {
            return hasReturn;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            MethodNode that = (MethodNode) o;
            return Objects.equals(params, that.params);
        }

        @Override
        public int hashCode() {
            return Objects.hash(params);
        }

        @Override
        public Double match(Set<Param> paramsThat) {
            Set<Param> paramsThis = getParams();

            double weight = 0.0d;
            //  根据参数长度加权重
            double addweight = Math.abs(paramsThat.size() - getParams().size());
            if (addweight == 0) {
                addweight = -1;
            }
            weight = weight - addweight;

            // 根据参数名加权重 + 根据参数类型加权重
            for (Param paramThis : paramsThis) {
                final Class<?> paramThisClass = paramThis.getpClass();
                final String paramThisName = paramThis.getName();

                if (paramsThat.contains(paramThis)) {
                    // +3
                    weight += 3.0d;
                } else if (paramsThat.stream().anyMatch(param -> param.getName().equals(paramThisName))) {
                    // +2
                    weight += 2.0d;
                } else if (paramsThat.stream().anyMatch(param -> paramThisClass.isAssignableFrom(param.getpClass()))) {
                    // +1
                    weight += 1.0d;
                }
            }
            return weight;
        }

        public Object invoke(Object... args) {
            return invoke(index, args);
        }

        public Object invoke(Param... params) {
            return invoke(Arrays.asList(params));
        }

        public Object invoke(Collection<Param> params) {

            Object[] args = new Object[getParams().size()];

            int index = 0;
            Map<Integer, Param> paramIndexGroup = new HashMap<>(getParams().size());
            for (Param param : getParams()) {
                paramIndexGroup.put(index++, param);
            }

            // 1 匹配参数名相同并且参数类型相同的值
            if (!paramIndexGroup.isEmpty()) {
                //
                Map<Integer, Param> paramHashGroup = CollectUtil.toMap(params, Param::hashCode);

                for (Integer paramIndex : paramIndexGroup.keySet()) {
                    Param paramOrig = paramIndexGroup.get(paramIndex);
                    Param paramByHash = paramHashGroup.get(paramOrig.hashCode());
                    //
                    if (null != paramByHash) {
                        args[paramIndex] = paramByHash.getValue();
                        paramIndexGroup.remove(paramIndex);
                    }
                }
            }

            // 2 匹配所有参数名相同的值
            if (!paramIndexGroup.isEmpty()) {
                //
                Map<String, Param> paramNameGroup = CollectUtil.toMap(params, Param::getName);

                for (Integer paramIndex : paramIndexGroup.keySet()) {
                    Param paramOrig = paramIndexGroup.get(paramIndex);
                    Param paramByName = paramNameGroup.get(paramOrig.getName());
                    //
                    if (null != paramByName) {
                        args[paramIndex] = JsonUtil.parse(paramByName.getValue(), paramOrig.getpClass());
                        paramIndexGroup.remove(paramIndex);
                    }
                }
            }

            // 3 匹配所有参数类型相同的值
            if (!paramIndexGroup.isEmpty()) {
                //
                Map<? extends Class<?>, Param> paramClassGroup = CollectUtil.toMap(params, Param::getpClass);

                for (Integer paramIndex : paramIndexGroup.keySet()) {
                    Param paramOrig = paramIndexGroup.get(paramIndex);
                    Param paramByClass = paramClassGroup.get(paramOrig.getpClass());
                    //
                    if (null != paramByClass) {
                        args[paramIndex] = paramByClass.getValue();
                        paramIndexGroup.remove(paramIndex);
                    }
                }
            }


            // 4 匹配剩余的值 设置默认值
            if (!paramIndexGroup.isEmpty()) {
                //
                for (Integer paramIndex : paramIndexGroup.keySet()) {
                    Param paramOrig = paramIndexGroup.get(paramIndex);
                    args[paramIndex] = ObjectUtil.defaultValue(paramOrig.getpClass());
                }
            }

            return invoke(args);
        }
    }

    public ActionHandler() {
        this.beans = new ArrayList<>();
        this.actionGroup = new HashMap<>();
    }

    public static ActionHandler me() {
        return Holder.ME;
    }


    private static class Holder {
        static final ActionHandler ME = new ActionHandler();
    }

}
