package cn.gjing.tools.excel.read.listener;

import cn.gjing.tools.excel.metadata.RowType;
import cn.gjing.tools.excel.metadata.listener.ExcelReadListener;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;

/**
 * Excel row read listener,
 * which listens for cell read events, row read events, and read start and end events
 *
 * @author Gjing
 **/
@FunctionalInterface
public interface ExcelRowReadListener<R> extends ExcelReadListener {
    /**
     * Read each row successfully
     *
     * @param r        The object generated by the current row. This object only exists for imports where that row is a data row and is of binding type
     * @param row      The current read completed row
     * @param rowIndex The current read completed row index
     * @param rowType  Current row type
     */
    void readRow(R r, Row row, int rowIndex, RowType rowType);

    /**
     * Read each cell successfully.
     * Occurs after data converter processing, if present
     *
     * @param cellValue Current cell value
     * @param cell      Current cell
     * @param rowIndex  Current row index
     * @param colIndex  Current col index
     * @param rowType   Current row type
     * @return Your processed cell value
     */
    default Object readCell(Object cellValue, Cell cell, int rowIndex, int colIndex, RowType rowType) {
        return cellValue;
    }

    /**
     * File data read finished
     */
    default void readFinish() {
    }

    /**
     * Before you start reading the data
     */
    default void readBefore() {
    }

    /**
     * Continue read next row , once set to false will immediately stop reading down,
     * Triggered after each row is read {@link #readRow}
     *
     * @return Default true
     */
    default boolean continueRead() {
        return true;
    }
}
