package cn.geminis.data.jpa.utils;

import cn.geminis.core.data.query.QueryParameters;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;

import javax.persistence.EntityManager;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.*;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author puddi
 */
public class QueryUtils {

    public static <T> Predicate setQueryParameters(QueryParameters queryParameters, Root<T> root,
                                                   CriteriaBuilder criteriaBuilder, CriteriaQuery<?> query) {
        List<From<?, ?>> fetches = new ArrayList<>();
        if (!query.getResultType().equals(Long.class)) {
            fetches = setIncludes(queryParameters, root);
        }
        return PredicateUtils.createPredicate(queryParameters.getFilterGroup(), root, criteriaBuilder, fetches);
    }

    private static <T> List<From<?, ?>> setIncludes(QueryParameters queryParameters, Root<T> root) {
        var result = new ArrayList<From<?, ?>>();

        var includes = queryParameters.getIncludes();
        for (var include : includes) {
            var fieldNames = include.split("\\.");
            var fetch = root.fetch(fieldNames[0], JoinType.LEFT);
            result.add((From<?, ?>) fetch);
            for (var i = 1; i < fieldNames.length; i++) {
                fetch = fetch.fetch(fieldNames[i], JoinType.LEFT);
                result.add((From<?, ?>) fetch);
            }
        }
        return result;
    }

    public static Sort createSort(QueryParameters queryParameters) {
        return Sort.by(queryParameters.getSorts()
                .stream()
                .map(sort -> new Sort.Order(
                        Sort.Direction.valueOf(sort.getOrder().toUpperCase()),
                        sort.getField()))
                .collect(Collectors.toList()));
    }

    public static <T extends Object> TypedQuery<T> createTypedQuery(EntityManager entityManager,
                                                                    Class<T> clazz,
                                                                    QueryParameters queryParameters) {
        var criteriaBuilder = entityManager.getCriteriaBuilder();
        var query = criteriaBuilder.createQuery(clazz);
        var root = query.from(clazz);

        var predicate = QueryUtils.setQueryParameters(queryParameters, root, criteriaBuilder, query);
        query.where(predicate);

        var sort = QueryUtils.createSort(queryParameters);
        if (sort.isSorted()) {
            query.orderBy(org.springframework.data.jpa.repository.query.QueryUtils.toOrders(sort, root, criteriaBuilder));
        }

        return entityManager.createQuery(query);
    }

    public static Pageable createPageable(QueryParameters queryParameters) {
        if (queryParameters.getPageIndex() < 0) {
            return Pageable.unpaged();
        } else {
            return PageRequest.of(queryParameters.getPageIndex(),
                    queryParameters.getPageSize(),
                    QueryUtils.createSort(queryParameters));
        }
    }
}
