package cn.geminis.data.jpa;

import cn.geminis.core.data.query.FilterGroupCreator;
import cn.geminis.core.data.query.QueryParameters;
import cn.geminis.data.jpa.utils.QueryUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.jpa.repository.support.JpaEntityInformationSupport;
import org.springframework.data.util.ProxyUtils;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.persistence.EntityManager;
import javax.persistence.NoResultException;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * @author puddi
 */
@Repository
public class GeminiRepository {

    @Autowired
    private EntityManager entityManager;

    @Transactional(readOnly = true, rollbackFor = Exception.class)
    public <T extends Object> Page<T> findPage(Class<T> clazz) {
        try {
            return findPage(clazz, QueryParameters.instance());
        } finally {
            QueryParameters.clean();
        }
    }

    @Transactional(readOnly = true, rollbackFor = Exception.class)
    public <T extends Object> Page<T> findPage(Class<T> clazz, QueryParameters queryParameters) {
        List<T> content = new ArrayList<T>();
        var count = count(clazz, queryParameters);
        var page = QueryUtils.createPageable(queryParameters);
        if (count > 0) {
            if (queryParameters.getPageIndex() < 0) {
                content = QueryUtils.createTypedQuery(this.entityManager, clazz, queryParameters)
                        .getResultList();
            } else {
                content = QueryUtils.createTypedQuery(this.entityManager, clazz, queryParameters)
                        .setFirstResult(queryParameters.getPageSize() * queryParameters.getPageIndex())
                        .setMaxResults(queryParameters.getPageSize())
                        .getResultList();
            }
        }
        return new PageImpl<T>(content, page, count);
    }

    @Transactional(readOnly = true, rollbackFor = Exception.class)
    public <T extends Object> List<T> findList(Class<T> clazz) {
        try {
            return findList(clazz, QueryParameters.instance());
        } finally {
            QueryParameters.clean();
        }
    }

    @Transactional(readOnly = true, rollbackFor = Exception.class)
    public <T extends Object> List<T> findList(Class<T> clazz, QueryParameters queryParameters) {
        return QueryUtils.createTypedQuery(this.entityManager, clazz, queryParameters).getResultList();
    }

    @Transactional(readOnly = true, rollbackFor = Exception.class)
    public <T extends Object> Optional<T> findOne(Class<T> clazz) {
        try {
            return findOne(clazz, QueryParameters.instance());
        } finally {
            QueryParameters.clean();
        }
    }

    @Transactional(readOnly = true, rollbackFor = Exception.class)
    public <T extends Object> Optional<T> findOne(Class<T> clazz, QueryParameters queryParameters) {
        try {
            return Optional.of(QueryUtils.createTypedQuery(this.entityManager, clazz, queryParameters).getSingleResult());
        } catch (NoResultException var3) {
            return Optional.empty();
        }
    }

    @Transactional(readOnly = true, rollbackFor = Exception.class)
    public <T extends Object> Optional<T> findById(Class<T> clazz, Object id) {
        Assert.notNull(id, "The given id must not be null!");
        return Optional.ofNullable(this.entityManager.find(clazz, id));
    }

    @Transactional(readOnly = true, rollbackFor = Exception.class)
    public <T extends Object> Optional<T> getOne(Class<T> clazz, Object id) {
        return Optional.of(this.entityManager.getReference(clazz, id));
    }

    public long count(Class<?> clazz) {
        try {
            return count(clazz, QueryParameters.instance());
        } finally {
            QueryParameters.clean();
        }
    }

    public long count(Class<?> clazz, QueryParameters queryParameters) {
        var criteriaBuilder = this.entityManager.getCriteriaBuilder();
        var query = criteriaBuilder.createQuery(Long.class);
        var root = query.from(clazz);

        var predicate = QueryUtils.setQueryParameters(queryParameters, root, criteriaBuilder, query);
        query.where(predicate);

        if (query.isDistinct()) {
            query.select(criteriaBuilder.countDistinct(root));
        } else {
            query.select(criteriaBuilder.count(root));
        }
        return this.entityManager.createQuery(query).getSingleResult();
    }

    @Transactional(rollbackFor = Exception.class)
    public <T extends Object> T save(T entity) {
        Assert.notNull(entity, "Entity must not be null!");
        var type = (Class<T>) ProxyUtils.getUserClass(entity);
        var entityInformation = JpaEntityInformationSupport.getEntityInformation(type, this.entityManager);
        if (entityInformation.isNew(entity)) {
            this.entityManager.persist(entity);
            return entity;
        } else {
            return this.entityManager.merge(entity);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public <T extends Object> void delete(T entity) {
        Assert.notNull(entity, "Entity must not be null!");
        var type = (Class<T>) ProxyUtils.getUserClass(entity);
        var entityInformation = JpaEntityInformationSupport.getEntityInformation(type, this.entityManager);
        if (!entityInformation.isNew(entity)) {
            var existing = this.entityManager.find(type, entityInformation.getId(entity));
            if (existing != null) {
                this.entityManager.remove(this.entityManager.contains(entity) ? entity : this.entityManager.merge(entity));
            }
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public <T extends Object> void deleteById(Class<T> clazz, Object id) {
        Assert.notNull(id, "id must not be null!");
        this.delete(this.findById(clazz, id)
                .orElseThrow(() -> new EmptyResultDataAccessException(String.format("No %s entity with id %s exists!", clazz.getName(), id), 1)));
    }

    public GeminiRepository where(String field, String compareType, Object value) {
        QueryParameters.instance().where(field, compareType, value);
        return this;
    }

    public GeminiRepository where(String field, Object value) {
        QueryParameters.instance().where(field, value);
        return this;
    }

    public GeminiRepository whereGroup(String type, FilterGroupCreator creator) {
        QueryParameters.instance().whereGroup(type, creator);
        return this;
    }

    public GeminiRepository include(String field) {
        QueryParameters.instance().include(field);
        return this;
    }

    public GeminiRepository orderBy(String field, String order) {
        QueryParameters.instance().orderBy(field, order);
        return this;
    }

    public GeminiRepository orderBy(String field) {
        QueryParameters.instance().orderBy(field);
        return this;
    }

    public GeminiRepository pageIndex(int pageIndex) {
        QueryParameters.instance().setPageIndex(pageIndex);
        return this;
    }

    public GeminiRepository pageSize(int pageSize) {
        QueryParameters.instance().setPageSize(pageSize);
        return this;
    }
}
