package cn.geminis.data.jpa;

import cn.geminis.core.data.query.QueryParameters;
import cn.geminis.core.data.query.QueryService;
import cn.geminis.data.jpa.utils.QueryUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.repository.NoRepositoryBean;
import org.springframework.transaction.annotation.Transactional;

import java.io.Serializable;
import java.util.ArrayList;

/**
 * @author Allen
 */
@NoRepositoryBean
public interface DynamicRepository<T, TIdType extends Serializable> extends
        JpaRepository<T, TIdType>,
        JpaSpecificationExecutor<T>,
        QueryService<T, DynamicRepository<T, TIdType>> {

    /**
     * 根据参数查询数据
     *
     * @param queryParameters 查询参数
     * @return 查询结果
     */
    @Override
    @Transactional(readOnly = true, rollbackFor = Exception.class)
    default Page<T> findPage(QueryParameters queryParameters) {
        var sort = QueryUtils.createSort(queryParameters);
        var pageable = QueryUtils.createPageable(queryParameters);

        if (queryParameters.getPageIndex() < 0) {
            var content = this.findAll((Specification<T>) (root, query, criteriaBuilder) -> QueryUtils
                    .setQueryParameters(queryParameters, root, criteriaBuilder, query), sort);
            return new PageImpl<T>(content, pageable, content.size());
        } else if (queryParameters.getPageSize() == 1) {
            var resultOne = this.findOne((Specification<T>) (root, query, criteriaBuilder) -> QueryUtils
                    .setQueryParameters(queryParameters, root, criteriaBuilder, query));
            var content = new ArrayList<T>();
            resultOne.ifPresent(content::add);
            return new PageImpl<T>(content, pageable, 1);
        } else {
            return this.findAll((Specification<T>) (root, query, criteriaBuilder) -> QueryUtils
                    .setQueryParameters(queryParameters, root, criteriaBuilder, query), pageable);
        }

    }

    /**
     * 查询数量
     *
     * @param queryParameters 查询参数
     * @return 数量
     */
    @Override
    default long countQuery(QueryParameters queryParameters) {
        return this.count((Specification<T>) (root, query, criteriaBuilder) -> QueryUtils
                .setQueryParameters(queryParameters, root, criteriaBuilder, query));
    }

}
